//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Sep 18, 2007
//---------------------

package org.cleversafe.layer.grid.smartcontroller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ExecutorService;

import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.CorruptedSliceObserver;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.GridTransaction;
import org.cleversafe.layer.grid.OperationScorecard;
import org.cleversafe.layer.grid.PlacedSliceName;
import org.cleversafe.layer.grid.ReadController;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.OperationScorecard.Entry;
import org.cleversafe.layer.grid.exceptions.ControllerDataTransformationException;
import org.cleversafe.layer.grid.exceptions.ControllerException;
import org.cleversafe.layer.grid.exceptions.ControllerGridStateUnknownException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerInformationDispersalException;
import org.cleversafe.layer.grid.exceptions.ControllerStoresNotFoundException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.util.ThreadlessExecutor;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.exceptions.DataTransformationException;
import org.cleversafe.vault.exceptions.InformationDispersalException;

import com.evelopers.common.exception.CommonException;
import com.evelopers.unimod.runtime.ModelEngine;

public class SmartReadController implements ReadController
{
   private static Logger _logger = Logger.getLogger(SmartReadController.class);

   // Name of the WorldStateMachine ControlledObject in this state machine
   public static final String STATE_MACHINE_WORLD_NAME = "world";

   // Vault associated with this controller's writes
   private Vault vault;

   // Executor to control currency of write operations
   private ExecutorService executor;

   // Has this controller been shutdown?
   private boolean isShutdown;

   // Corrupted slice observers
   private List<CorruptedSliceObserver> corruptedSliceObservers;

   /**
    * Create a single-threaded controller
    * 
    * @param vault
    */
   public SmartReadController(Vault vault)
   {
      this(vault, new ThreadlessExecutor());
   }

   /**
    * Create a controller, specifying the threading model
    * 
    * @param vault
    * @param executor
    */
   public SmartReadController(Vault vault, ExecutorService executor)
   {
      this.vault = vault;
      this.executor = executor;
      this.corruptedSliceObservers = new LinkedList<CorruptedSliceObserver>();

      this.isShutdown = false;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.ReadController#read(org.cleversafe.layer.grid.SourceName)
    */
   public DataSource read(SourceName name) throws ControllerDataTransformationException,
         ControllerInformationDispersalException, ControllerIOException,
         ControllerGridStateUnknownException, ControllerIllegalSourceNameException,
         ControllerStoresNotFoundException
   {
      List<SourceName> sourceNames = new ArrayList<SourceName>(1);
      sourceNames.add(name);

      List<DataSource> sources = this.readImpl(sourceNames);
      assert sources.size() == 1 : "Invalid read size";
      return sources.get(0);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.ReadController#read(java.util.List)
    */
   public List<DataSource> read(List<SourceName> names)
         throws ControllerDataTransformationException, ControllerInformationDispersalException,
         ControllerIOException, ControllerGridStateUnknownException,
         ControllerIllegalSourceNameException, ControllerStoresNotFoundException
   {
      return this.readImpl(names);
   }

   private List<DataSource> readImpl(List<SourceName> names)
         throws ControllerDataTransformationException, ControllerInformationDispersalException,
         ControllerIOException, ControllerGridStateUnknownException,
         ControllerIllegalSourceNameException, ControllerStoresNotFoundException
   {
      assert !this.isShutdown : "shutdown() has been called";

      // Handle an empty list of names
      if (names.isEmpty())
      {
         return Collections.emptyList();
      }

      long checkpoint0 = System.currentTimeMillis();
      if (_logger.isTraceEnabled())
         _logger.trace("SmartReadController.read(): started");

      // FIXME: Order stores for read
      List<SliceStore> stores = this.vault.getSliceStores();

      // Populate a matrix with EXPECTED slices
      OperationScorecard matrix = new OperationScorecard(names, stores);
      for (SourceName name : names)
      {
         List<PlacedSliceName> sliceNames = this.vault.mapSourceNameToSliceNames(name);
         matrix.addPlacedSliceNames(sliceNames, OperationScorecard.Status.EXPECTED);
      }

      // Load and init state machine
      ModelEngine modelEngine;
      ReadStateMachine readStateMachine;
      try
      {
         modelEngine = ReadControllerEventProcessor.createModelEngine(false);
         readStateMachine =
               (ReadStateMachine) modelEngine.getControlledObjectsManager().getControlledObject(
                     STATE_MACHINE_WORLD_NAME);
         readStateMachine.init(modelEngine);
         readStateMachine.setExecutor(this.executor);
      }
      catch (final CommonException ex)
      {
         throw new RuntimeException("Unable to load state machine", ex);
      }

      // Set up state machine
      readStateMachine.setVault(this.vault);

      // Execute machine and block until complete
      readStateMachine.run(matrix);

      long checkpoint1 = System.currentTimeMillis();

      // Assemble datasources
      try
      {
         // Slice codecs have already been applied
         List<DataSource> assembledData = this.vault.rebuildSources(matrix, false);
         assert assembledData.size() == names.size() : "Wrong number of sources read";

         long checkpoint2 = System.currentTimeMillis();
         if (_logger.isTraceEnabled())
            _logger.trace(String.format(
                  "SmartReadController.read(): successful - %dms (1:%dms, 2:%dfms)", checkpoint2
                        - checkpoint0, checkpoint1 - checkpoint0, checkpoint2 - checkpoint1));

         // Find any corrupted slices and notify any observers
         // TODO: It would be nice to optimize this (currently: O(sources*stores)). Consider a way
         // to generalize the notion of corruption in the matrix
         if (!this.corruptedSliceObservers.isEmpty())
         {
            for (DataSource source : assembledData)
            {
               SourceName name = source.getName();
               List<SliceName> corruptedSlices = new LinkedList<SliceName>();

               for (SliceStore store : matrix.getStores())
               {
                  Entry entry = matrix.getEntry(name, store);
                  if (entry != null)
                  {
                     OperationScorecard.Status status = entry.getStatus();
                     if (status == OperationScorecard.Status.ERROR
                           && entry.getException() instanceof DataTransformationException)
                     {
                        // Corrupted slice
                        corruptedSlices.add(new SliceName(name, entry.getSliceIndex()));
                     }
                     else if (status == OperationScorecard.Status.EXISTING
                           && entry.getTransactionId() == GridTransaction.NOT_FOUND_TRANSACTION_ID)
                     {
                        // Slice not found
                        corruptedSlices.add(new SliceName(name, entry.getSliceIndex()));
                     }
                  }
               }

               // Notify observers
               for (CorruptedSliceObserver observer : this.corruptedSliceObservers)
               {
                  observer.notify(source.getTransactionId(), corruptedSlices);
               }
            }
         }

         return assembledData;
      }
      catch (final InformationDispersalException ex)
      {
         throw new ControllerInformationDispersalException("Cannot assemble source", ex);
      }
      catch (final DataTransformationException ex)
      {
         throw new ControllerDataTransformationException("Source failed pre-dispersal decoding", ex);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.OperationController#setVault(org.cleversafe.vault.Vault)
    */
   public void setVault(Vault vault)
   {
      this.vault = vault;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.OperationController#getVault()
    */
   public Vault getVault()
   {
      return this.vault;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.OperationController#addCorruptedSliceObserver(org.cleversafe.layer.grid.CorruptedSliceObserver)
    */
   public void addCorruptedSliceObserver(CorruptedSliceObserver corruptedSliceObserver)
   {
      this.corruptedSliceObservers.add(corruptedSliceObserver);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.OperationController#isOperational()
    */
   public boolean isOperational()
   {
      return !isShutdown;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.OperationController#shutdown()
    */
   public void shutdown() throws ControllerException
   {
      this.isShutdown = true;
   }

}
