//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Greg Dhuse
//
// Date: May 1, 2007
//---------------------

package org.cleversafe.layer.grid.smartcontroller;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;

import org.apache.log4j.Logger;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.layer.grid.CorruptedSliceObserver;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.GridTransaction;
import org.cleversafe.layer.grid.OperationScorecard;
import org.cleversafe.layer.grid.PlacedSourceSliceSet;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.SourceSliceSet;
import org.cleversafe.layer.grid.WriteController;
import org.cleversafe.layer.grid.exceptions.ControllerDataTransformationException;
import org.cleversafe.layer.grid.exceptions.ControllerGridStateUnknownException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerInformationDispersalException;
import org.cleversafe.layer.grid.exceptions.ControllerStoresNotFoundException;
import org.cleversafe.layer.grid.exceptions.ControllerTransactionException;
import org.cleversafe.layer.grid.exceptions.GridLayerException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.util.ThreadlessExecutor;
import org.cleversafe.util.TransactionIdGenerator;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.exceptions.DataTransformationException;
import org.cleversafe.vault.exceptions.InformationDispersalException;

import com.evelopers.common.exception.CommonException;
import com.evelopers.unimod.runtime.ModelEngine;

/**
 * A WriteController implementation that uses a finite state machine and has a robust error
 * tolerance
 */
public class SmartWriteController implements WriteController
{
   private static Logger _logger = Logger.getLogger(SmartWriteController.class);

   // Name of the WorldStateMachine ControlledObject in this state machine
   public static final String STATE_MACHINE_WORLD_NAME = "world";

   // Has this controller been shutdown?
   private boolean isShutdown;

   // Transaction ID
   private long gridTransactionId;

   // Transaction for writes using this controller
   private GridTransaction gridTransaction;

   // List of scorecards for each write done using this controller 
   private Collection<OperationScorecard> scorecards;

   // Vault associated with this controller's writes
   private Vault vault;

   // Exception thrown from the state machine, if any
   private GridLayerException gridException;

   // Executor to control currency of write operations
   private ExecutorService executor;

   // Corrupted slice observers
   private List<CorruptedSliceObserver> corruptedSliceObservers;

   // Verifies the success or failure of any grid transaction operations (commit/rollback)
   private GridTransaction.OperationVerifier transactionSuccessVerifier;

   /**
    * Construct a single-threaded WriteController
    * 
    * @param vault
    */
   public SmartWriteController(Vault vault)
   {
      // By default, do not use threading
      this(vault, new ThreadlessExecutor());
   }

   /**
    * Construct a WriteController using the provided executor for concurrency
    * 
    * @param vault
    * @param executor
    */
   public SmartWriteController(Vault vault, ExecutorService executor)
   {
      this.vault = vault;
      this.executor = executor;

      this.gridTransactionId = TransactionIdGenerator.generate();
      this.gridTransaction = new GridTransaction(this.gridTransactionId, executor);
      this.corruptedSliceObservers = new LinkedList<CorruptedSliceObserver>();
      this.scorecards = new ConcurrentLinkedQueue<OperationScorecard>();

      this.gridException = null;
      this.isShutdown = false;

      this.transactionSuccessVerifier = new ThresholdTransactionOperationVerifier();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.WriteController#write(org.cleversafe.layer.grid.DataSource)
    */
   public void write(DataSource data) throws ControllerDataTransformationException,
         ControllerInformationDispersalException, ControllerIOException,
         ControllerIllegalSourceNameException, ControllerTransactionException,
         ControllerGridStateUnknownException, ControllerStoresNotFoundException
   {
      List<DataSource> sources = new ArrayList<DataSource>(1);
      sources.add(data);
      this.writeImpl(sources);
   }

   public void write(List<DataSource> sources) throws ControllerDataTransformationException,
         ControllerInformationDispersalException, ControllerIOException,
         ControllerIllegalSourceNameException, ControllerTransactionException,
         ControllerGridStateUnknownException, ControllerStoresNotFoundException
   {
      writeImpl(sources);
   }

   private void writeImpl(List<DataSource> sources) throws ControllerDataTransformationException,
         ControllerInformationDispersalException, ControllerIOException,
         ControllerIllegalSourceNameException, ControllerTransactionException,
         ControllerGridStateUnknownException, ControllerStoresNotFoundException
   {
      assert !this.isShutdown : "shutdown() has been called";
      assert this.gridTransaction != null : "commit() or rollback() has been called";

      long checkpoint0 = System.currentTimeMillis();
      if (_logger.isTraceEnabled())
         _logger.trace(String.format("SmartWriteController.write(tx=%d): started",
               this.gridTransactionId));

      // Handle an empty source list
      if (sources.isEmpty())
      {
         return;
      }

      // Perform encoding and dispersal
      List<SourceName> sourceNames = new ArrayList<SourceName>(sources.size());
      for (DataSource source : sources)
      {
         sourceNames.add(source.getName());
      }

      OperationScorecard scorecard =
            new OperationScorecard(sourceNames, this.vault.getSliceStores());
      this.scorecards.add(scorecard);
      try
      {
         try
         {
            for (DataSource source : sources)
            {
               // Apply coding stack
               SourceSliceSet slices = this.vault.applyDispersalStack(source);

               // Find grid locations
               PlacedSourceSliceSet placedSlices = this.vault.mapSliceSetToStore(slices);

               // Add to matrix
               scorecard.addSourceSliceSet(placedSlices, OperationScorecard.Status.EXISTING);
            }
         }
         catch (final InformationDispersalException ex)
         {
            throw new ControllerInformationDispersalException("Unable to disperse source", ex);
         }
         catch (final DataTransformationException ex)
         {
            throw new ControllerDataTransformationException("Unable to encode source", ex);
         }

         long checkpoint1 = System.currentTimeMillis();

         // Load and init state machine
         ModelEngine modelEngine;
         WriteStateMachine writeStateMachine;
         try
         {
            modelEngine = WriteControllerEventProcessor.createModelEngine(false);
            writeStateMachine =
                  (WriteStateMachine) modelEngine.getControlledObjectsManager().getControlledObject(
                        STATE_MACHINE_WORLD_NAME);
            writeStateMachine.init(modelEngine);
            writeStateMachine.setExecutor(this.executor);
         }
         catch (final CommonException ex)
         {
            throw new RuntimeException("Unable to load state machine", ex);
         }

         // Set up state machine
         writeStateMachine.setGridTransaction(this.gridTransaction);
         writeStateMachine.setVault(this.vault);

         // Execute machine and block until complete
         try
         {
            writeStateMachine.run(scorecard);

            long checkpoint2 = System.currentTimeMillis();

            // Notify observers of incomplete writes
            if (!this.corruptedSliceObservers.isEmpty())
            {
               // Find slices that could not be written
               List<SliceName> corruptedSlices = new LinkedList<SliceName>();
               for (SliceStore store : scorecard.getStores())
               {
                  // A slice was not written if its status is ERROR, regardless of the exception
                  corruptedSlices.addAll(scorecard.getStoreSliceNames(store,
                        OperationScorecard.Status.ERROR));
               }

               for (CorruptedSliceObserver observer : this.corruptedSliceObservers)
               {
                  observer.notify(this.getTransactionId(), corruptedSlices);
               }
            }

            long checkpoint3 = System.currentTimeMillis();

            if (_logger.isTraceEnabled())
               _logger.trace(String.format(
                     "SmartWriteController.write(tx=%d): successful - %dms (1:%dms, 2:%dms, 3:%dms)",
                     this.gridTransactionId, checkpoint3 - checkpoint0, checkpoint1 - checkpoint0,
                     checkpoint2 - checkpoint1, checkpoint3 - checkpoint2));
         }
         catch (final ControllerIOException ex)
         {
            this.gridException = ex;
            throw ex;
         }
         catch (final ControllerIllegalSourceNameException ex)
         {
            this.gridException = ex;
            throw ex;
         }
         catch (final ControllerTransactionException ex)
         {
            this.gridException = ex;
            throw ex;
         }
         catch (final ControllerGridStateUnknownException ex)
         {
            this.gridException = ex;
            throw ex;
         }
         catch (final ControllerStoresNotFoundException ex)
         {
            this.gridException = ex;
            throw ex;
         }
      }
      finally
      {
         // Ensure that we do not hold data payload references longer than necessary
         scorecard.clearData();
      }
   }

   /**
    * Commit transaction. No more writes may be called after commit
    * 
    * @throws ControllerTransactionException
    * @throws ControllerIOException
    */
   public void commit() throws ControllerTransactionException, ControllerIOException,
         ControllerStoresNotFoundException
   {
      assert !this.isShutdown : "shutdown() has been called";

      try
      {
         this.gridTransaction.commit(this.transactionSuccessVerifier);
      }
      finally
      {
         this.gridTransaction = null;
      }
   }

   /**
    * Rollback transaction. No more writes may be called after rollback
    * 
    * @throws ControllerTransactionException
    * @throws ControllerIOException
    */
   public void rollback() throws ControllerTransactionException, ControllerIOException,
         ControllerStoresNotFoundException
   {
      assert !this.isShutdown : "shutdown() has been called";

      try
      {
         this.gridTransaction.rollback(this.transactionSuccessVerifier);
      }
      finally
      {
         this.gridTransaction = null;
      }
   }

   /**
    * Complete this write operation. No other methods may be called on this object after finish()
    * 
    * @throws ControllerTransactionException
    * @throws ControllerIOException
    */
   public void shutdown() throws ControllerTransactionException, ControllerIOException,
         ControllerStoresNotFoundException
   {
      assert !this.isShutdown : "shutdown() has been called";

      try
      {
         // Commit or rollback based on exception status
         if (this.gridTransaction != null)
         {
            if (this.gridException != null)
            {
               this.rollback();
            }
            else
            {
               this.commit();
            }
         }
      }
      finally
      {
         this.isShutdown = true;
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.WriteController#getTransactionId()
    */
   public long getTransactionId()
   {
      return this.gridTransactionId;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.OperationController#getVault()
    */
   public Vault getVault()
   {
      return this.vault;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.OperationController#isOperational()
    */
   public boolean isOperational()
   {
      return !this.isShutdown;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.OperationController#setVault(org.cleversafe.vault.Vault)
    */
   public void setVault(Vault vault)
   {
      this.vault = vault;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.grid.OperationController#addCorruptedSliceObserver(org.cleversafe.layer.grid.CorruptedSliceObserver)
    */
   public void addCorruptedSliceObserver(CorruptedSliceObserver corruptedSliceObserver)
   {
      this.corruptedSliceObservers.add(corruptedSliceObserver);
   }

   /**
    * A transaction operation is considered a success if at least the write threshold number of 
    * slices were successfully written for each source in each individual (multi-)write operation
    * governed by this controller.  Errors that occur during begin, write, and this operation
    * are all aggregated to make this decision.  
    */
   private class ThresholdTransactionOperationVerifier implements GridTransaction.OperationVerifier
   {
      public boolean verifyOperation(Collection<Tuple2<SliceStore, Exception>> errors)
      {
         SmartWriteController wc = SmartWriteController.this;
         InformationDispersalCodec ida = wc.vault.getInformationDispersalCodec();
         int maxErrors = ida.getNumSlices() - wc.vault.getWriteThreshold();

         // Add transaction to each scorecard and see if that write would still be valid
         for (OperationScorecard scorecard : SmartWriteController.this.scorecards)
         {
            // We avoid side effects by cloning the scorecard
            scorecard = (OperationScorecard) scorecard.clone();

            // Apply errors from this operation to the scorecard
            for (Tuple2<SliceStore, Exception> error : errors)
            {
               try
               {
                  scorecard.setStoreStatus(error.getFirst(), OperationScorecard.Status.ERROR,
                        error.getSecond());
               }
               catch (IllegalArgumentException ex)
               {
                  // Store is not a part of this write and error does not apply 
               }
            }

            // Check each source against the write threshold
            for (SourceName source : scorecard.getSources())
            {
               if (scorecard.getSourceStatusCount(source, OperationScorecard.Status.ERROR) > maxErrors)
               {
                  // This source was not written successfully
                  return false;
               }
            }
         }

         return true;
      }
   }
}
