//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: June 18, 2007
//---------------------

package org.cleversafe.layer.protocol;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.Map.Entry;

import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.serialization.GridSerializable;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultACLFactory;
import org.cleversafe.vault.exceptions.VaultConfigurationException;
import org.cleversafe.vault.exceptions.VaultIOException;

/**
 * Class for "create_store" request protocol operation (sent from client to server).
 * 
 * The request contains various parameters required for creating a {@link SliceStore} on a remote
 * server.
 * 
 * @see CreateStoreResponse
 * @author Jason Resch
 */
public class CreateStoreRequest extends GridProtocolRequest
{
   @GridSerializable.ElementOrder(order = 1)
   private String vaultType = null;

   @GridSerializable.ElementOrder(order = 2)
   private long maxSliceSize = -1;

   @GridSerializable.ElementOrder(order = 3)
   private long sliceStoreSize = -1;

   @GridSerializable.ElementOrder(order = 4)
   private String sliceStoreType = null;

   @GridSerializable.ElementOrder(order = 5)
   private String vaultIdentifier = null;

   @GridSerializable.ElementOrder(order = 6)
   private byte[] vaultACL = null;

   @GridSerializable.ElementOrder(order = 7)
   private byte[] vaultDescriptorBytes = null;

   // Options map is stored as two lists
   @GridSerializable.ElementOrder(order = 8)
   private List<String> keys = null;

   @GridSerializable.ElementOrder(order = 9)
   private List<String> values = null;

   // TODO: Create a constructor which properly sets the new member variables

   /**
    * Default constructor.
    * 
    */
   public CreateStoreRequest()
   {
      super(GridProtocolOperation.CREATE_STORE_REQUEST);
      this.keys = new ArrayList<String>();
      this.values = new ArrayList<String>();
   }

   /**
    * Construct a CreateStoreRequest using vault type, slice store type, UUID, options map and Vault
    * ACL
    * 
    */
   public CreateStoreRequest(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         String sliceStoreType,
         UUID vaultIdentifier,
         VaultACL vaultACL,
         byte[] vaultDescriptor,
         Map<String, String> options)
   {
      super(GridProtocolOperation.CREATE_STORE_REQUEST);
      this.vaultType = vaultType;
      this.maxSliceSize = maxSliceSize;
      this.sliceStoreSize = sliceStoreSize;
      this.sliceStoreType = sliceStoreType;
      this.vaultIdentifier = vaultIdentifier.toString();
      this.vaultACL = vaultACL.getEncoded();
      this.vaultDescriptorBytes = vaultDescriptor;
      this.keys = new ArrayList<String>();
      this.values = new ArrayList<String>();
      createListsFromMap(options);
   }

   /**
    * Sets the two list member variables which are serializable
    * 
    * @param map
    */
   private synchronized void createListsFromMap(Map<String, String> map)
   {
      keys.clear();
      values.clear();
      for (Entry<String, String> entry : map.entrySet())
      {
         keys.add(entry.getKey());
         values.add(entry.getValue());
      }
   }

   /**
    * Reforms the options map from the key and values lists
    * 
    * @return
    */
   private Map<String, String> createMapFromLists()
   {
      Map<String, String> map = new HashMap<String, String>();
      for (int itr = 0; itr < keys.size(); itr++)
      {
         map.put(keys.get(itr), values.get(itr));
      }
      return map;
   }

   /**
    * Returns a String identifier for the type of vault (file, filesystem, block) that the client
    * will use for storing data. The server makes sure that the type of vault matches with an
    * appropriate type of slice store.
    * 
    * @return a string identifier specifying the type of vault.
    */
   public String getVaultType()
   {
      return this.vaultType;
   }

   /**
    * Returns the maximum size of a single slice in bytes.
    * 
    * @return the maximum size of a single slice in bytes.
    */
   public long getMaxSliceSize()
   {
      return this.maxSliceSize;
   }

   /**
    * Returns the size of the slice store to be created.
    * 
    * @return the size of the slice store in bytes.
    */
   public long getSliceStoreSize()
   {
      return this.sliceStoreSize;
   }

   /**
    * Returns a String identifier for the type the slice store to create.
    * 
    * @return a string identifier for the type the slice store to create.
    */
   public String getSliceStoreType()
   {
      return this.sliceStoreType;
   }

   /**
    * Returns the Vault ID of the slice store to create.
    * 
    * @return the Vault ID of the slice store to create.
    */
   public UUID getVaultID()
   {
      return UUID.fromString(this.vaultIdentifier);
   }

   /**
    * Returns the Vault ID of the slice store to create.
    * 
    * @return the Vault ID of the slice store to create.
    */
   public Map<String, String> getOptions()
   {
      return createMapFromLists();
   }

   /**
    * Returns the Vault ACL of the slice store being created.
    * 
    * @return The Vault ACL of the slice store being created.
    */
   public VaultACL getVaultACL() throws VaultIOException, VaultConfigurationException
   {
      VaultACLFactory factory = new VaultACLFactory();
      return factory.getInstance(this.vaultACL);
   }

   /**
    * Returns the encoded Vault ACL byte array.
    * 
    * @return The encoded Vault ACL byte array.
    */
   public byte[] getVaultACLBytes()
   {
      return this.vaultACL;
   }

   /**
    * Returns the vault descriptor byte array of the slice store being created.
    * 
    * @return The vault descriptor byte array of the slice store being created.
    */
   public byte[] getVaultDescriptorBytes()
   {
      return this.vaultDescriptorBytes;
   }

   /**
    * Sets the Vault Type of the slice store to create.
    * 
    * @param vaultType
    *           the vault type of the slice store to create.
    */
   public void setVaultType(String vaultType)
   {
      this.vaultType = vaultType;
   }

   /**
    * Sets the maximum slice size.
    * 
    * @param maxSliceSize
    */
   public void setMaxSliceSize(long maxSliceSize)
   {
      this.maxSliceSize = maxSliceSize;
   }

   /**
    * Sets the size in bytes of the slice store to be created.
    * 
    * @param sliceStoreSize
    */
   public void setSliceStoreSize(long sliceStoreSize)
   {
      this.sliceStoreSize = sliceStoreSize;
   }

   /**
    * Sets the Slice Store Type of the slice store to create.
    * 
    * @param sliceStoreType
    *           the slice store type of the slice store to create.
    */
   public void setSliceStoreType(String sliceStoreType)
   {
      this.sliceStoreType = sliceStoreType;
   }

   /**
    * Sets the Vault ID of the slice store to create.
    * 
    * @param vaultID
    *           the vault ID of the slice store to create.
    */
   public void setVaultID(UUID vaultID)
   {
      this.vaultIdentifier = vaultID.toString();
   }

   /**
    * Sets the Slice Store Type of the slice store to create.
    * 
    * @param sliceStoreType
    *           the slice store type of the slice store to create.
    */
   public void setOptions(Map<String, String> options)
   {
      createListsFromMap(options);
   }

   /**
    * Sets the Vault ACL of the slice store to create.
    * 
    * @param vaultACL
    *           the vault ACL of the slice store to create.
    */
   public void setVaultACL(VaultACL vaultACL)
   {
      this.vaultACL = vaultACL.getEncoded();
   }

   /**
    * Sets the encoded Vault ACL byte array.
    * 
    * @param vaultACLBytes
    *           the encoded Vault ACL byte array
    */
   public void setVaultACLBytes(byte[] vaultACLBytes)
   {
      this.vaultACL = vaultACLBytes;
   }

   /**
    * Sets the vault descriptor bytes of the slice store being created.
    * 
    * @param vaultDescriptorBytes
    *           the byte array that represents a vault descriptor.
    */
   public void setVaultDescriptorBytes(byte[] vaultDescriptorBytes)
   {
      this.vaultDescriptorBytes = vaultDescriptorBytes;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.protocol.messages.Request#toString()
    */
   @Override
   public String toString()
   {
      // FIXME: When VaultACL is complete, deserialize it and use VaultACL.toString();
      return super.toString() + " Vault ID: " + this.vaultIdentifier + " Vault ACL: "
            + this.vaultACL.toString();
   }

}
