//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Manish Motwani <mmotwani@cleversafe.com>
//         John Quigley <jquigley@cleversafe.com>
//
// Date: Apr 30, 2007
//---------------------

package org.cleversafe.layer.protocol;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import org.apache.log4j.Logger;
import org.cleversafe.layer.protocol.exceptions.HeaderFormatException;
import org.cleversafe.layer.protocol.exceptions.ProtocolSerializationException;

/**
 * This class serializes and deserializes a protocol data unit's header.
 * 
 * @author Manish Motwani
 * @author John Quigley
 */
public class GridProtocolHeaderCodec
{
   private static Logger _logger = Logger.getLogger(GridProtocolHeaderCodec.class);

   public static final int PROTOCOL_MAGIC = 0xca;
   private short protocolVersion;
   private int messageID;
   private int operationCode;
   private int payloadLength;
   private short protocolID = 1;

   public static final int LENGTH = 18;
   public static final int PROTOCOL_VERSION = 0x01;

   /**
    * Constructor. Takes the deserialized integer values and creates the Header object.
    * 
    * @param protocolVersion
    * @param messageID
    * @param operationCode
    * @param payloadLength
    */
   public GridProtocolHeaderCodec(
         short protocolVersion,
         int messageID,
         int operationCode,
         int payloadLength)
   {
      if ((protocolVersion & 0xffff) != protocolVersion)
      {
         throw new IllegalArgumentException(
               "Protocol Version integer passed in the constructor is out of range!");
      }

      this.protocolVersion = protocolVersion;
      this.messageID = messageID;
      this.operationCode = operationCode;
      this.payloadLength = payloadLength;
   }

   public GridProtocolHeaderCodec(
         short protocolVersion,
         short protocolID,
         int messageID,
         int operationCode,
         int payloadLength)
   {
      if ((protocolVersion & 0xffff) != protocolVersion)
      {
         throw new IllegalArgumentException(
               "Protocol Version integer passed in the constructor is out of range!");
      }

      this.protocolVersion = protocolVersion;
      this.protocolID = protocolID;
      this.messageID = messageID;
      this.operationCode = operationCode;
      this.payloadLength = payloadLength;
   }

   /**
    * Constructor. Takes the byte array and deserializes it into a Header object.
    * 
    * @param header
    *           The header byte array.
    * @throws HeaderFormatException
    */
   public GridProtocolHeaderCodec(byte[] header) throws HeaderFormatException
   {
      // Checking array size
      if (header.length < GridProtocolHeaderCodec.LENGTH)
      {
         throw new HeaderFormatException("Input byte array is too small!");
      }

      _logger.trace("Deserializing header byte array.");
      ByteArrayInputStream bis = new ByteArrayInputStream(header);
      DataInputStream in = new DataInputStream(bis);

      // Read ints from input
      int firstInt = 0;
      try
      {
         firstInt = in.readInt();
         this.protocolID = in.readShort();
         this.messageID = in.readInt();
         this.operationCode = in.readInt();
         this.payloadLength = in.readInt();
      }
      catch (IOException ioe)
      {
         throw new HeaderFormatException(
               "Header bytes passed in constructor are of invalid format!", ioe);
      }

      // Take the upper 2 bytes and set it to magic
      int magic = (firstInt >> 16) & 0xffff;

      if (magic != PROTOCOL_MAGIC)
      {
         throw new HeaderFormatException(String.format("Magic number mismatch, expected %x got %x",
               PROTOCOL_MAGIC, magic));
      }

      // Take the lowest 2 bytes and set it to protocolVersion
      this.protocolVersion = (short) (firstInt & 0xffff);
   }

   public void serialize(DataOutputStream out) throws ProtocolSerializationException
   {
      write(out);
   }

   /**
    * Writes the header to the speicified output stream
    * 
    * @throws ProtocolSerializationException
    */
   public void write(DataOutputStream dos) throws ProtocolSerializationException
   {
      _logger.trace("Serializing a Header object into a header byte array.");

      // Serialize the first 4 bytes:
      // - First 2 bytes are magic
      // - Last 2 bytes is protocolVersion
      int magicAndProtocolVersion =
            ((PROTOCOL_MAGIC << 16) & 0xffff0000) | (this.protocolVersion & 0xffff);

      try
      {
         dos.writeInt(magicAndProtocolVersion);
         dos.writeShort(this.protocolID);
         dos.writeInt(this.messageID);
         dos.writeInt(this.operationCode);
         dos.writeInt(this.payloadLength);
         dos.flush();
      }
      catch (IOException ioe)
      {
         throw new ProtocolSerializationException(ioe);
      }
   }

   /**
    * Returns the magic number.
    * 
    * @return The magic number.
    */
   public int getMagic()
   {
      return PROTOCOL_MAGIC;
   }

   /**
    * Returns the protocol version.
    * 
    * @return The protocol version.
    */
   public short getProtocolVersion()
   {
      return this.protocolVersion;
   }

   /**
    * Returns the message ID.
    * 
    * @return The message ID.
    */
   public int getMessageID()
   {
      return this.messageID;
   }

   /**
    * Returns the payload length.
    * 
    * @return The payload length.
    */
   public int getPayloadLength()
   {
      return this.payloadLength;
   }

   /**
    * Returns the protocol operation code.
    * 
    * @return The protocol operation code.
    */
   public int getOperationCode()
   {
      return this.operationCode;
   }

   public short getProtocolID()
   {
      return this.protocolID;
   }

   public String toString()
   {
      return "Protocol Header [ V" + this.protocolVersion + " ID=" + this.messageID + " C="
            + this.operationCode + " L=" + this.payloadLength + "]";
   }

   @Deprecated
   public void fixLength(byte[] bytes)
   {
      int v = bytes.length - GridProtocolHeaderCodec.LENGTH;

      bytes[14] = (byte) ((v >>> 24) & 0xFF);
      bytes[15] = (byte) ((v >>> 16) & 0xFF);
      bytes[16] = (byte) ((v >>> 8) & 0xFF);
      bytes[17] = (byte) ((v >>> 0) & 0xFF);
   }
}
