//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: May 7, 2007
//---------------------

package org.cleversafe.layer.protocol;

import org.cleversafe.serialization.GridSerializable;
import org.cleversafe.serialization.asn1.ASN1ExceptionWrapper;

/**
 * Base class for all grid protocol response messages received by the client from the server.
 * 
 * All grid response messages contain an exception, which may be set to <code>null</code> if no
 * exception occurred on the server while processing a request.
 * 
 * The method <code>getExceptionFlag()</code> returns a boolean indicating whether an exception
 * occurred on the server while processing a request. One can handle exceptions thrown at the server
 * during a request handling operation using this method in conjunction with the
 * <code>getException()</code> method, which should return the full <code>Exception</code> with
 * a reason for the error.
 * 
 * @see GridProtocolRequest
 * @see GridProtocolUnsolicitedMessage
 * @author Manish Motwani
 */
public abstract class GridProtocolResponse extends GridProtocolMessage implements Response
{
   @GridSerializable.ElementOrder(order = 1)
   private ASN1ExceptionWrapper exception = null;

   /**
    * Protected constructor.
    * 
    * @param operation
    *           Protocol operation enumeration.
    */
   protected GridProtocolResponse(ProtocolOperation operation)
   {
      super(operation);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.protocol.messages.IResponse#getExceptionFlag()
    */
   public boolean getExceptionFlag()
   {
      return this.exception != null && this.exception.getOriginalException() != null;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.communication.protocol.Response#getException()
    */
   public Exception getException()
   {
      if (this.exception == null)
      {
         return null;
      }
      else
      {
         return this.exception.getOriginalException();
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.communication.protocol.Response#setException(java.lang.Exception)
    */
   public void setException(Exception exception)
   {
      if (exception == null)
      {
         this.exception = null;
      }
      else if (this.exception == null)
      {
         this.exception = new ASN1ExceptionWrapper(exception);
      }
      else
      {
         this.exception.setOriginalException(exception);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.protocol.messages.ProtocolMessage#isRequest()
    */
   public boolean isRequest()
   {
      return false;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.communication.protocol.ProtocolMessage#isResponse()
    */
   public boolean isResponse()
   {
      return true;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.protocol.messages.ProtocolMessage#isUnsolicited()
    */
   public boolean isUnsolicited()
   {
      return false;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.protocol.messages.ProtocolMessage#toString()
    */
   @Override
   public String toString()
   {
      StringBuffer buf = new StringBuffer();
      buf.append(super.toString());
      buf.append(" (Response)");

      if (this.exception != null)
      {
         buf.append(" Exception: ");
         buf.append(this.exception.toString());
      }

      return buf.toString();
   }
}
