//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Aug 9, 2007
//---------------------

package org.cleversafe.layer.protocol;

import java.util.Arrays;
import java.util.List;

import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.serialization.GridSerializable;

/**
 * Class for "multiple_read" request protocol operation (sent from client to server).
 * 
 * The request contains an array of {@link SliceName} objects that represent the names of the slices
 * that are being requested to read.
 * 
 * @see MultipleReadResponse
 * @author Manish Motwani
 */
public class MultipleReadRequest extends GridProtocolRequest
{
   @GridSerializable.ElementOrder(order = 1)
   private SliceName[] sliceNames = null;

   /**
    * Default constructor.
    * 
    */
   public MultipleReadRequest()
   {
      super(GridProtocolOperation.MULTIPLE_READ_REQUEST);
   }

   /**
    * Constructor.
    * 
    * @param sliceName
    */
   public MultipleReadRequest(SliceName sliceName)
   {
      super(GridProtocolOperation.MULTIPLE_READ_REQUEST);

      this.sliceNames = new SliceName[1];
      this.sliceNames[0] = sliceName;
   }

   /**
    * Constructor.
    * 
    * @param sliceNames
    */
   public MultipleReadRequest(SliceName[] sliceNames)
   {
      super(GridProtocolOperation.MULTIPLE_READ_REQUEST);
      this.sliceNames = sliceNames;
   }

   /**
    * Convenience Constructor.
    * 
    * @param sliceNames
    */
   public MultipleReadRequest(List<SliceName> sliceNames)
   {
      super(GridProtocolOperation.MULTIPLE_READ_REQUEST);
      setSliceNames(sliceNames);
   }

   /**
    * Returns the array of source names requested to be read.
    * 
    * @return
    */
   public SliceName[] getSliceNames()
   {
      return this.sliceNames;
   }

   /**
    * Sets the array of source names to be read.
    * 
    * @param sourceNames
    */
   public void setSliceNames(SliceName[] sliceNames)
   {
      this.sliceNames = sliceNames;
   }

   /**
    * Sets the array of source names to be read.
    * 
    * @param sourceNames
    */
   public void setSliceNames(List<SliceName> sliceNames)
   {
      if (sliceNames == null)
      {
         this.sliceNames = null;
      }
      else
      {
         this.sliceNames = sliceNames.toArray(new SliceName[sliceNames.size()]);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.protocol.GridProtocolRequest#toString()
    */
   @Override
   public String toString()
   {
      return super.toString() + " Source names: " + Arrays.toString(this.sliceNames);
   }
}
