//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Dusty Hendrickson
//
// Date: May 29, 2007
//---------------------

package org.cleversafe.layer.protocol;

import java.util.UUID;

import org.cleversafe.serialization.GridSerializable;

/**
 * Represents a request for authentication using a username and password. The corresponding response
 * contains the authentication status of the request.
 * 
 * @see PasswordAuthenticationResponse
 * @author Dusty Hendrickson
 */
public class PasswordAuthenticationRequest extends GridProtocolRequest
{
   @GridSerializable.ElementOrder(order = 1)
   private String username;

   @GridSerializable.ElementOrder(order = 2)
   private String password;

   @GridSerializable.ElementOrder(order = 3)
   // TODO: could this be a UUID instead of String?
   private String gridAccountID;

   /**
    * Constructs a request. Username and password are set to empty strings. The grid account ID is
    * set to 00000000-0000-0000-0000-000000000000.
    * 
    */
   public PasswordAuthenticationRequest()
   {
      super(GridProtocolOperation.PASSWORD_AUTHENTICATION_REQUEST);

      this.username = "";
      this.password = "";
      this.gridAccountID = "00000000-0000-0000-0000-000000000000";
   }

   /**
    * Constructs a request with the given parameters.
    * 
    * @param username
    *           The username.
    * @param password
    *           The password.
    * @param gridAccountID
    *           The grid account ID.
    */
   public PasswordAuthenticationRequest(String username, String password, UUID gridAccountID)
   {
      super(GridProtocolOperation.PASSWORD_AUTHENTICATION_REQUEST);

      this.username = username;
      this.password = password;
      this.gridAccountID = gridAccountID.toString();
   }

   /**
    * Sets the username.
    * 
    * @param username
    *           The username.
    */
   public void setUsername(String username)
   {
      this.username = username;
   }

   /**
    * Returns the username.
    * 
    * @return The username.
    */
   public String getUsername()
   {
      return this.username;
   }

   /**
    * Sets the password.
    * 
    * @param password
    *           The password.
    */
   public void setPassword(String password)
   {
      this.password = password;
   }

   /**
    * Returns the password.
    * 
    * @return The password.
    */
   public String getPassword()
   {
      return this.password;
   }

   /**
    * Sets the grid account ID.
    * 
    * @param gridAccountID
    *           The grid account ID.
    */
   public void setGridAccountID(UUID gridAccountID)
   {
      this.gridAccountID = gridAccountID.toString();
   }

   /**
    * Returns the grid account ID.
    * 
    * @return The grid account ID.
    */
   public UUID getGridAccountID()
   {
      return UUID.fromString(this.gridAccountID);
   }
}
