//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: Jul 30, 2007
//---------------------

package org.cleversafe.layer.slicestore;


import java.util.Map;
import java.util.UUID;

import org.cleversafe.layer.slicestore.exceptions.RepairException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultPermission;
import org.cleversafe.vault.exceptions.VaultConfigurationException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;

/**
 * Interfaces describes SliceStoreManager, which will be implemented for each type of SliceStore. It
 * allows for SliceStores to be created, deleted and loaded in a consistent manner.
 */
public interface SliceStoreManager
{
   /**
    * Creates a slice store based on the provided options, using the given vault id and ACL. The
    * options map must have all necessary key-value pairs required by the implementation.
    * 
    * @param vaultType
    * @param maxSliceSize
    * @param sliceStoreSize
    * @param vaultID
    * @param vaultACL
    * @param vaultDescriptorBytes
    * @param options
    * @throws SliceStoreExistsException
    * @throws SliceStoreIOException
    * @throws SliceStoreStateException
    */
   void createSliceStore(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         UUID vaultID,
         VaultACL vaultACL,
         byte[] vaultDescriptorBytes,
         Map<String, String> options) throws SliceStoreExistsException, SliceStoreIOException;

   /**
    * Deletes a given slice store, using the provided options. The options map must have all
    * necessary key-value pairs required by the implementation.
    * 
    * @param vaultID
    * @param options
    * @throws SliceStoreIOException
    * @throws SliceStoreStateException
    */
   void deleteSliceStore(UUID vaultID, Map<String, String> options) throws SliceStoreIOException,
         SliceStoreNotFoundException;

   /**
    * Returns a slice store object of the same type as the corresponding manager. Note that the
    * returned slice store is not active and startSession() must be called before use.
    * 
    * @param vaultID
    * @return SliceStore
    * @throws SliceStoreIOException
    */
   SliceStore loadSliceStore(UUID vaultID) throws SliceStoreIOException, 
         SliceStoreNotFoundException;

   /**
    * Returns the VaultPermission for the given accountID on the specified Vault. This permission
    * object is obtained from the vault ACL object that was provided during creation time.
    * 
    * @param vaultID
    * @param accountID
    * @return VaultPermission
    * @throws VaultIOException
    * @throws VaultSecurityException
    */
   VaultPermission loadVaultPermissions(UUID vaultID, UUID accountID) throws SliceStoreIOException,
         SliceStoreNotFoundException;

   /**
    * Checks slice store integrity
    * @param vault
    * @param acl
    * @return
    */
   RepairAdvice checkIntegrity(Vault vault, VaultACL acl) throws RepairException;

   /**
    * Performs slice store repair. Each managers implements it differently
    * @param vault
    * @param vaultDescriptorBytes
    * @param acl
    * @throws RepairException
    * @throws SliceStoreExistsException
    */
   public void repair(RepairAdvice advice, Vault vault, byte[] vaultDescriptorBytes, VaultACL acl)
         throws RepairException, SliceStoreExistsException;

   /**
    * Wraps all slice stores created or loaded by this manager using the given wrapper.
    * 
    * @param wrapper
    */
   void setWrapper(SliceStoreWrapper wrapper);
}
