//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER

package org.cleversafe.layer.slicestore.block;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.UUID;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.apache.commons.collections.bag.SynchronizedBag;
import org.apache.log4j.Logger;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.exceptions.NotImplementedException;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.slicestore.NotificationHandler;
import org.cleversafe.layer.slicestore.SliceInfo;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.layer.slicestore.exceptions.WriteTransactionDeadlockException;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;

/**
 * File block slice store implementation that stores slices as fixed size blocks in a single file
 * Requires from vault: - slice names should be positive numeric, with max size known up front - max
 * slice size should be known up front
 * 
 * TODO: ??? Do we need transaction cleanup based on timing???
 */
public class BlockFileSliceStore extends SliceStoreBase implements SliceStore
{
   // This implementation uses transaction content in memory
   // It should be very efficient for a block device

   static Logger _logger = Logger.getLogger(BlockFileSliceStore.class);

   private static final int EXISTING_BLOCK_MAGIC = 0x13579BDF;

   private static final int REMOVED_BLOCK_MAGIC = 0xACE02468;

   public static final String BLOCK_FILE_NAME = "BlockDevice.slice";

   public static final String PROPERTIES_FILE_NAME = "BlockDevice.properties";

   public static final String BLOCK_SIZE_PROPERTY_NAME = "block-size";

   public static final String MAX_SLICE_NUMBER_PROPERTY_NAME = "max-slice-number";

   public static final String SYNC_PROPERTY_NAME = "sync";

   protected Properties blockDeviceProperties = null;

   /**
    * BlockFile implementation of SliceStoreTransaction
    * 
    * @see SliceStoreTransaction
    */
   public class BlockFileSliceStoreTransaction implements SliceStoreTransaction
   {
      private long transactionID;

      Map<Long, DataSlice> content = null; // Blocks written for a transaction

      /**
       * Constructs BlockFileSliceStoreTransaction taking the transactionId
       * 
       * @param transactionID
       */
      public BlockFileSliceStoreTransaction(final long transactionID)
      {
         this.transactionID = transactionID;
      }

      /**
       * Begins transaction, creating a map to store data slices
       */
      private void begin()
      {
         this.content = Collections.synchronizedSortedMap(new TreeMap<Long, DataSlice>());
      }

      /**
       * Cleanup a transaction content. Nothing is written to the storage
       * 
       * @throws Exception
       */
      private synchronized void rollback()
      {
         if (_logger.isTraceEnabled())
            _logger.trace("Rolling back transaction " + getID());
         this.content = null;
      }

      /**
       * Write content to file
       * 
       * @throws IOException
       */
      private synchronized void commit() throws SliceStoreIOException
      {
         if (_logger.isTraceEnabled())
            _logger.trace("Commiting transaction " + getID());
         try
         {
            final Set<Long> blockNumbers = content.keySet();
            Tuple2<Long, DataSlice>[] sources = new Tuple2[blockNumbers.size()];

            int i = 0;
            for (final long blockNumber : blockNumbers)
            {
               sources[i++] = new Tuple2<Long, DataSlice>(blockNumber, content.get(blockNumber));
            }
            doRealWrite(this.getID(), sources);
         }
         finally
         {
            this.content = null;
         }
      }

      /**
       * Returns true if transaction has been begun but not committed or rolled back
       * 
       * @return active status of transaction
       */
      public synchronized boolean isActive()
      {
         return this.content != null;
      }

      /**
       * @see SliceStoreTransaction
       */
      public long getID()
      {
         return this.transactionID;
      }

      /**
       * @see SliceStoreTransaction
       */
      public SliceStore getSliceStore()
      {
         return BlockFileSliceStore.this;
      }

      /**
       * Creates a block image in memory
       * 
       * @param blockNumber
       * @param data
       * @throws SliceStoreTransactionException 
       * @throws IOException
       */
      protected void saveBlock(final long blockNumber, DataSlice dataSlice)
            throws SliceStoreTransactionException
      {
         assert this.transactionID == dataSlice.getTransactionId() : "An attempt to executr wtite witin a different transaction";
         if (dataSlice.getData().length > blockSize)
         {
            throw new IllegalArgumentException("Block storage doesn't support block larger then "
                  + blockSize);
         }
         synchronized (this)
         {
            // rollback could be issued asynchronously and could sneak before write was completed, thus 
            // causing put to fail
            if (content == null)
            {
               throw new SliceStoreTransactionException("Transaction " + this.transactionID
                     + " was likely rollback");
            }
            content.put(blockNumber, dataSlice);
         }
      }
   }

   protected File rootDir; // Directory for a block file

   protected long maxSliceNumber = SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED;

   protected int blockSize = -1; // maximum block size

   protected boolean blockSizeInitialized = false;

   protected boolean synchronous; // Use synchronous file access?

   protected String basePath = null;

   protected int realBlockSize;

   protected byte[] padding = null;
   /*
    * Number of physical blocks written into a slice file It could only increase and it is not
    * really good :(
    */
   protected long eofBlockNumber = 0;

   protected RandomAccessFile randomAccessor;

   protected int listBlockCount = 500; // Number of blocks per list call

   protected long listBlockNumber = -1; // Current block number for listing functions

   private final ConcurrentMap<Long, BlockFileSliceStoreTransaction> transactionMap =
         new ConcurrentHashMap<Long, BlockFileSliceStoreTransaction>();

   protected UUID vaultID;

   /**
    * Default constructor used by configuration framework
    * 
    */
   public BlockFileSliceStore()
   {
      this.blockDeviceProperties = new Properties();
   }

   /**
    * Constructs a slice store object given parameters. The blockSize and synchronous parameters are
    * loaded from a properties file that was created during createStore().
    * 
    * @param vaultID
    * 
    * @param path
    */
   public BlockFileSliceStore(UUID vaultID, final String path)
   {
      this();
      this.vaultID = vaultID;
      setPath(path);
      initialize();
   }

   /**
    * This method should be only called before initialize In order to enforce, it is stored in
    * temporary variables and will be officially available after initialize() is invoked. This
    * approach guarantees that later calls will have no effect
    * 
    * @param path
    */
   public void setPath(final String path)
   {
      this.basePath = path;
   }

   /**
    * Plain getter
    * 
    */
   public String getPath()
   {
      return this.rootDir.getAbsolutePath();
   }

   /** 
    * Set the number of blocks that are returned for each list call.
    * 
    * @param blockCount
    */
   public void setListBlockCount(final int blockCount)
   {
      this.listBlockCount = blockCount;
   }

   /**
    * This method should be only called before initialize In order to enforce, it is stored in
    * temporary variables and will be officially available after initialize() is invoked. This
    * approach guarantees that later calls will have no effect
    * 
    * @param blockSize
    * 
    */
   protected void setBlockSize(final int blockSize)
   {
      this.blockSize = blockSize;
      // magic + transaction + length + data
      this.realBlockSize = ((Integer.SIZE + Integer.SIZE + Long.SIZE) / 8) + this.blockSize;

      this.padding = new byte[blockSize]; // used to pad a real block
      Arrays.fill(this.padding, (byte) 0xFF);

      this.blockSizeInitialized = true;
   }

   /**
    * block size getter
    * 
    * @return maximum block size
    */
   public long getBlockSize()
   {
      return this.blockSize;
   }

   /**
    * This method should only be called before initialize
    * 
    * @param synchronous
    */
   public void setSynchronous(boolean synchronous)
   {
      this.synchronous = synchronous;
   }

   /**
    * Are writes performed synchronously?
    * 
    * @return True if writes are performed synchronously
    */
   public boolean getSynchronous()
   {
      return this.synchronous;
   }

   /**
    * Configuration framework code
    * 
    */
   public void initialize()
   {
      if (this.basePath == null)
      {
         throw new InitializationException(
               "BlockFileStore is not properly initialized: path must be set");
      }

      this.rootDir = new File(this.basePath);

      _logger.info("Completed initializing Block file store (sync=" + this.synchronous + ") in "
            + this.rootDir.getAbsolutePath());
   }

   /**
    * @see SliceStore
    */
   public String getIdentification()
   {
      return "BlockFile:[" + this.rootDir.getAbsolutePath() + "," + this.blockSize + ","
            + this.eofBlockNumber + "]";
   }

   /**
    * @see SliceStore
    */
   public boolean isOperational()
   {
      return (randomAccessor != null);
   }

   /**
    * Creates a block store for a single vault
    * 
    * @see SliceStore
    */
   public void createStore(
         final String vaultType,
         final long maxSliceSize,
         final long sliceStoreSize,
         final VaultACL accessControlList,
         final byte[] vaultDescriptorBytes,
         final Map<String, String> options) throws SliceStoreIOException, SliceStoreExistsException
   {
      // TODO: check whether vault type is supported by this slice store
      // TODO: check long value is within int bounds
      setBlockSize((int) maxSliceSize);

      // Calculate the maximum number of slices given the Store Size
      if (sliceStoreSize != SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED)
      {
         this.maxSliceNumber = sliceStoreSize / maxSliceSize;
      }
      else
      {
         this.maxSliceNumber = SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED;
      }

      _logger.info("Creating a new block store in " + this.rootDir.getAbsolutePath());

      File sliceFile = new File(this.rootDir.getAbsolutePath() + File.separator + BLOCK_FILE_NAME);
      if (sliceFile.exists() == true)
      {
         throw new SliceStoreExistsException("This slice store " + sliceFile.getAbsolutePath()
               + " already exists");
      }

      try
      {
         this.rootDir.mkdirs();
         this.randomAccessor =
               new RandomAccessFile(this.rootDir.getAbsolutePath() + File.separator
                     + BLOCK_FILE_NAME, "rw");
         this.randomAccessor.close();
         this.randomAccessor = null;

         for (String key : options.keySet())
         {
            this.blockDeviceProperties.setProperty(key, options.get(key));
         }
         this.saveProperties();
      }
      catch (final Exception e)
      {
         throw new SliceStoreIOException("Can't create a slice store", e);
      }

      // Serialize ACL and write it
      this.writeVaultACL(this.rootDir + File.separator + ACL_FILE_NAME, accessControlList);

      // Write the vault descriptor
      this.writeVaultDescriptor(this.rootDir + File.separator
            + accessControlList.getVaultIdentifier().toString() + VAULT_DESCRIPTOR_EXTENSION,
            vaultDescriptorBytes);

      _logger.info("Created block file store (block=" + this.blockSize + ") in "
            + this.rootDir.getAbsolutePath());
   }

   /**
    * Returns the vault descriptor that was saved during this store's creation.
    * 
    * @return
    * @throws SliceStoreIOException
    * @throws SliceStoreNotFoundException 
    */
   public VaultDescriptor getVaultDescriptor(PrivateKey privateKey, Credentials credentials)
         throws SliceStoreIOException, VaultIOException, VaultSecurityException,
         SliceStoreNotFoundException
   {
      try
      {
         return this.getVaultDescriptor(this.rootDir.getPath(), privateKey, credentials);
      }
      catch (VaultACLException e)
      {
         throw new VaultSecurityException("Error accessing vault ACL", e);
      }
   }

   /**
    * This method is used to store persistent information about a BlockFileSlice during the creation
    * of the SliceStore, so that subsequent loadings don't need to worry about providing an invalid
    * parameter that could lead to data corruption.
    * 
    * Currently only the blockSize value is stored, however the maxBlockNumber when it is utilized
    * should also be stored here.
    * 
    * @throws SliceStoreIOException
    */
   private void saveProperties() throws SliceStoreIOException
   {
      this.blockDeviceProperties.setProperty(MAX_SLICE_NUMBER_PROPERTY_NAME,
            Long.toString(this.maxSliceNumber));

      this.blockDeviceProperties.setProperty(BLOCK_SIZE_PROPERTY_NAME,
            Integer.toString(this.blockSize));

      this.blockDeviceProperties.setProperty(SYNC_PROPERTY_NAME, Boolean.toString(this.synchronous));

      final File outFile =
            new File(this.rootDir.getAbsolutePath() + File.separator + PROPERTIES_FILE_NAME);
      FileOutputStream fileOutputStream = null;
      try
      {
         fileOutputStream = new FileOutputStream(outFile);
         this.blockDeviceProperties.store(fileOutputStream, null);
      }
      catch (final IOException ex)
      {
         throw new SliceStoreIOException("Unable to save the BlockDevice properties", ex);
      }
      finally
      {
         try
         {
            if (fileOutputStream != null)
            {
               fileOutputStream.close();
            }
         }
         catch (IOException ignore)
         {
            _logger.warn("Unable to close file " + fileOutputStream);
         }
      }
   }

   /**
    * This method is used to load persistent information about a BlockFileSlice during the
    * construction of the SliceStore, it uses information saved about the SliceStore during the
    * creation of it.
    * 
    * Currently only the blockSize value is loaded, however the maxBlockNumber when it is utilized
    * should also be loaded here.
    * 
    * @throws SliceStoreIOException
    */
   protected void loadProperties()
   {
      final File inFile =
            new File(this.rootDir.getAbsolutePath() + File.separator + PROPERTIES_FILE_NAME);
      FileInputStream fileInputStream = null;
      try
      {
         fileInputStream = new FileInputStream(inFile);
         this.blockDeviceProperties.load(fileInputStream);
      }
      catch (final IOException ex)
      {
         _logger.debug("BlockDevice properties file not found");
      }
      finally
      {
         try
         {
            if (fileInputStream != null)
            {
               fileInputStream.close();
            }
         }
         catch (IOException ignore)
         {
            _logger.warn("Unable to close file " + fileInputStream);
         }
      }

      if (blockDeviceProperties.containsKey(MAX_SLICE_NUMBER_PROPERTY_NAME))
      {
         String parsedMaxSliceNumber =
               blockDeviceProperties.getProperty(MAX_SLICE_NUMBER_PROPERTY_NAME);
         this.maxSliceNumber = Long.parseLong(parsedMaxSliceNumber);
      }

      if (blockDeviceProperties.containsKey(BLOCK_SIZE_PROPERTY_NAME))
      {
         String parsedBlockSize = blockDeviceProperties.getProperty(BLOCK_SIZE_PROPERTY_NAME);
         int parsedBlockSizeValue = Integer.parseInt(parsedBlockSize);
         this.setBlockSize(parsedBlockSizeValue);
      }

      if (blockDeviceProperties.containsKey(SYNC_PROPERTY_NAME))
      {
         String parsedSynchronous = blockDeviceProperties.getProperty(SYNC_PROPERTY_NAME);
         boolean synchronous = Boolean.parseBoolean(parsedSynchronous);
         this.setSynchronous(synchronous);
      }
   }

   /**
    * @see SliceStore
    */
   public void deleteStore() throws SliceStoreIOException
   {
      ensureNotOperational();

      // TODO: Make sure that a store is not opened by someone else?
      try
      {
         // Delete all store files
         for (File file : this.rootDir.listFiles())
         {
            boolean success = file.delete();
            _logger.info("Deleting file " + file + (success ? " success" : " failed"));
         }
         this.rootDir.delete();

         _logger.info("Deleted  Block file store size from " + this.rootDir.getAbsolutePath());
      }
      catch (final Exception ex)
      {
         _logger.error("Failed to delete store", ex);
         throw new SliceStoreIOException("Failed to delete store", ex);
      }
   }

   public void updateStore(
         String vaultType,
         long maxSliceStize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes) throws SliceStoreIOException
   {
      throw new RuntimeException("not yet implemented");
   }

   /**
    * @see SliceStore
    */
   public void startSession() throws SliceStoreIOException
   {
      ensureNotOperational();

      loadProperties();
      if (this.blockSizeInitialized == false)
      {
         throw new InitializationException(
               "BlockFileStore is not properly initialized: block size must be set");
      }

      try
      {
         this.randomAccessor =
               new RandomAccessFile(rootDir.getAbsoluteFile() + File.separator + BLOCK_FILE_NAME,
                     this.synchronous ? "rwd" : "rw");

         this.eofBlockNumber = this.randomAccessor.length() / this.realBlockSize;
      }
      catch (final FileNotFoundException e)
      {
         throw new SliceStoreIOException("Data file not found", e);
      }
      catch (final IOException e)
      {
         throw new SliceStoreIOException(e);
      }
   }

   /**
    * @see SliceStore
    */
   public void endSession() throws SliceStoreIOException
   {
      ensureOperational();

      // TODO: check uncommitted transactions. What to do with them?
      try
      {
         // Rollback any open transactions
         for (Entry<Long, BlockFileSliceStoreTransaction> entry : this.transactionMap.entrySet())
         {
            SliceStoreTransaction trans = entry.getValue();
            if (this.isActiveTransaction(trans))
            {
               try
               {
                  this.rollbackTransaction(trans);
               }
               catch (SliceStoreTransactionException e)
               {
                  _logger.warn("Unable to rollback transaction id=" + trans.getID(), e);
               }
            }
         }
         this.randomAccessor.close();
      }
      catch (final IOException e)
      {
         _logger.error("Failed to end session", e);
         throw new SliceStoreIOException("Failed to end session", e);
      }
      finally
      {
         this.randomAccessor = null;
         this.transactionMap.clear();
      }
      _logger.info("Session ended for slice store " + this.getPath());

   }

   /**
    * @see SliceStore
    */
   public SliceStoreTransaction createTransaction(final long transactionId)
         throws SliceStoreTransactionException
   {
      ensureOperational();

      final BlockFileSliceStoreTransaction newTx =
            new BlockFileSliceStoreTransaction(transactionId);

      // The semantics of putIfAbsent are strange. It returns the existing value if there is one, or
      // null if newTx is inserted
      BlockFileSliceStoreTransaction tx = this.transactionMap.putIfAbsent(transactionId, newTx);
      return tx != null ? tx : newTx;
   }

   /**
    * @see SliceStore
    */
   public void beginTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException
   {
      ensureOperational();
      assert transaction != null;
      ensureNotActiveTransaction(transaction);
      ensureTransactionFound(transaction.getID());

      assert transaction instanceof BlockFileSliceStoreTransaction;
      final BlockFileSliceStoreTransaction blockFileTransaction =
            (BlockFileSliceStoreTransaction) transaction;

      blockFileTransaction.begin();

      if (_logger.isTraceEnabled())
         _logger.trace("Begin transaction " + blockFileTransaction.getID());
   }

   /**
    * @see SliceStore
    */
   public void commitTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreIOException, SliceStoreTransactionException
   {
      assert transaction != null;
      ensureOperational();
      ensureActiveTransaction(transaction);

      this.transactionMap.remove(transaction.getID());

      // Note: This used to be synchronized, but it does not seem necessary (famous last words ;)
      final BlockFileSliceStoreTransaction blockFileTransaction =
            (BlockFileSliceStoreTransaction) transaction;

      cleanTrasactionSources(blockFileTransaction);
      blockFileTransaction.commit();
   }

   /**
    * @see SliceStore
    */
   public void rollbackTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException
   {
      ensureOperational();
      ensureActiveTransaction(transaction);

      this.transactionMap.remove(transaction.getID());

      // Note: This used to be synchronized, but it does not seem necessary (famous last words ;)
      final BlockFileSliceStoreTransaction blockFileTransaction =
            (BlockFileSliceStoreTransaction) transaction;
      cleanTrasactionSources(blockFileTransaction);
      blockFileTransaction.rollback();
   }

   //TODO: Create a more consistent way of guaranteeing that slice name is a part of a single transaction
   private void cleanTrasactionSources(BlockFileSliceStoreTransaction blockFileTransaction)
   {
      for (Map.Entry<Long, DataSlice> entry : blockFileTransaction.content.entrySet())
      {
         removeUncommittedSlices(entry.getValue().getSliceName());
      }
   }

   /**
    * @see SliceStore
    */
   public boolean isActiveTransaction(final SliceStoreTransaction transaction)
   {
      ensureOperational();

      assert transaction != null;

      boolean isActive = false;

      if (this.transactionMap.containsKey(transaction.getID()))
      {
         final BlockFileSliceStoreTransaction blockFileTransaction =
               (BlockFileSliceStoreTransaction) transaction;
         isActive = blockFileTransaction.isActive();
      }
      return isActive;
   }

   /**
    * @see SliceStore
    */
   public SliceStoreTransaction getTransaction(final long transactionId)
         throws SliceStoreTransactionException
   {
      ensureOperational();

      final SliceStoreTransaction transaction = this.transactionMap.get(transactionId);
      if (transaction == null)
      {
         throw new SliceStoreTransactionException("Requested transaction " + transactionId
               + " does not exist");
      }
      return transaction;
   }

   /**
    * @see SliceStore
    */
   public boolean exists(final SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException
   {
      ensureOperational();

      final long blockNumber = convertNameIntoBlockNumber(name);

      // Don't need synchronization here, even if it increases during this
      // op it is still not valid
      if (this.eofBlockNumber <= blockNumber)
      {
         return false;
      }

      try
      {
         final long position = blockNumber * this.realBlockSize;
         byte[] magicBytes = new byte[Integer.SIZE];
         synchronized (this.randomAccessor)
         {
            this.randomAccessor.seek(position);
            this.randomAccessor.read(magicBytes);
         }
         final ByteArrayInputStream block = new ByteArrayInputStream(magicBytes);
         final DataInputStream input = new DataInputStream(block);

         final int magic = input.readInt();
         return (magic == EXISTING_BLOCK_MAGIC);
      }
      catch (IOException ex)
      {
         throw new SliceStoreIOException("Error reading a block# " + blockNumber, ex);
      }
   }

   /**
    * @see SliceStore
    */
   public boolean remove(final SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException
   {
      ensureOperational();

      final long blockNumber = convertNameIntoBlockNumber(name);
      final long position = blockNumber * this.realBlockSize;

      try
      {
         if (exists(name) == false)
         {
            return false;
         }
         else
         {
            final ByteArrayOutputStream block = new ByteArrayOutputStream(realBlockSize);
            final DataOutputStream output = new DataOutputStream(block);

            output.writeInt(REMOVED_BLOCK_MAGIC);
            synchronized (this.randomAccessor)
            {
               this.randomAccessor.seek(position);
               this.randomAccessor.write(block.toByteArray());
            }

            if (_logger.isTraceEnabled())
               _logger.trace("Block # transaction " + blockNumber + " deleted");

            return true;
         }
      }
      catch (final IOException e)
      {
         throw new SliceStoreIOException("Failed to write transaction data", e);
      }
   }

   /**
    * @see SliceStore
    */
   public void registerForNorNotification(final NotificationHandler handler)
   {
      ensureOperational();
      throw new NotImplementedException(this.getClass().getName() + ".registerForNotification()");
   }

   /**
    * @see SliceStore
    */
   public void listBegin() throws SliceStoreIOException
   {
      ensureOperational();

      listBlockNumber = 0;
   }

   /**
    * @see SliceStore
    */
   public void listBegin(final SliceName name)
   {
      ensureOperational();
      throw new NotImplementedException(this.getClass().getName() + ".listBegin()");
   }

   /**
    * @see SliceStore
    */
   public List<SliceInfo> listContinue() throws SliceStoreIOException
   {
      ensureOperational();

      long endBlockNumber = listBlockNumber + this.listBlockCount;

      if (endBlockNumber >= this.eofBlockNumber)
      {
         endBlockNumber = this.eofBlockNumber;
      }

      List<SliceInfo> sliceInfoList = new ArrayList<SliceInfo>();

      synchronized (this.randomAccessor)
      {
         for (; listBlockNumber < endBlockNumber; listBlockNumber++)
         {
            SourceName sourceName = new SourceName(String.valueOf(listBlockNumber));

            final long position = listBlockNumber * this.realBlockSize;

            try
            {
               this.randomAccessor.seek(position);

               final int magic = this.randomAccessor.readInt();

               if (magic == EXISTING_BLOCK_MAGIC)
               {
                  final long transactionId = this.randomAccessor.readLong();

                  sliceInfoList.add(new SliceInfo(sourceName, transactionId));
               }
            }
            catch (IOException ex)
            {
               throw new SliceStoreIOException("Error reading from file at position " + position,
                     ex);
            }
         }
      }

      return sliceInfoList;
   }

   /**
    * @see SliceStore
    */
   public boolean listInProgress()
   {
      ensureOperational();

      return (this.listBlockNumber >= 0 && this.listBlockNumber < this.eofBlockNumber);
   }

   /**
    * @see SliceStore
    */
   public void listStop()
   {
      ensureOperational();

      this.listBlockNumber = this.eofBlockNumber;
   }

   /**
    * Interface implementation. Depending on transaction either stores in transaction or writes into
    * file
    * @throws SliceStoreNotFoundException 
    * 
    * @see SliceStore
    */
   public void writeImpl(DataSlice dataSlice, boolean allowOverwriteNewer)
         throws SliceStoreTransactionException, SliceStoreIOException, IllegalSourceNameException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      ensureOperational();

      SliceStoreTransaction transaction = this.getTransaction(dataSlice.getTransactionId());
      ensureActiveTransaction(transaction);

      final long blockNumber = convertNameIntoBlockNumber(dataSlice.getSliceName());
      if (dataSlice.getData().length > this.blockSize)
      {
         throw new IllegalArgumentException("Data must be no more than " + this.blockSize
               + " bytes");
      }

      if ((blockNumber > this.maxSliceNumber)
            && (maxSliceNumber != SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED))
      {
         throw new SliceStoreQuotaException(
               "Attempted to write a block number greater than the specified maximum");
      }

      if (!allowOverwriteNewer)
      {
         checkOverwriteNewer(dataSlice);
      }
      
      addUncommittedSlice(dataSlice);

      try
      {
         final BlockFileSliceStoreTransaction blockFileTransaction =
               (BlockFileSliceStoreTransaction) transaction;
         blockFileTransaction.saveBlock(blockNumber, dataSlice);
      }
      catch (SliceStoreTransactionException e)
      {
         removeUncommittedSlices(dataSlice.getSliceName());
         throw e;
      }
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   @Override
   public void writeImpl(List<DataSlice> dataSlices, boolean allowOverwriteNewer)
         throws SliceStoreTransactionException, SliceStoreIOException, IllegalSourceNameException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      ensureOperational();

      for (int i = 0; i < dataSlices.size(); i++)
      {
         writeImpl(dataSlices.get(i), allowOverwriteNewer);
      }
   }

   /**
    * Performs write of data slices to disk
    * 
    * @param transactionId
    * @param sources
    * @throws SliceStoreIOException
    */
   protected void doRealWrite(long transactionId, Tuple2<Long, DataSlice>[] sources)
         throws SliceStoreIOException
   {
      int startIndex = 0;
      while (startIndex < sources.length) // Do we have anything not processed
      {
         // Find sequential numbers
         int index = startIndex;
         while (index < sources.length - 1
               && (sources[index + 1].getFirst() == sources[index].getFirst() + 1))
         {
            index++;
         }
         multipleWriteImpl(transactionId, startIndex, index, sources);
         startIndex = index + 1;
      }
   }

   /**
    * Helper method for do real write which writes multiple data slices to disk at once
    * 
    * @param transactionId
    * @param startIndex
    * @param endIndex
    * @param sources
    * @throws SliceStoreIOException
    */
   private void multipleWriteImpl(
         long transactionId,
         int startIndex,
         int endIndex,
         Tuple2<Long, DataSlice>[] sources) throws SliceStoreIOException
   {
      int nblocks = (int) (endIndex - startIndex + 1);
      long startBlock = sources[startIndex].getFirst();
      long endBlock = sources[endIndex].getFirst();

      final ByteArrayOutputStream blocks = new ByteArrayOutputStream(realBlockSize * nblocks);
      final DataOutputStream output = new DataOutputStream(blocks);
      try
      {
         for (int i = startIndex; i <= endIndex; i++)
         {
            byte data[] = sources[i].getSecond().getData();
            output.writeInt(EXISTING_BLOCK_MAGIC);
            output.writeLong(transactionId);
            output.writeInt(data.length);
            output.write(data);
            // Append if a block is not full
            if (data.length < this.blockSize)
            {
               output.write(this.padding, 0, this.blockSize - data.length);
            }
         }
         output.flush();
         synchronized (this.randomAccessor)
         {
            randomAccessor.seek(startBlock * this.realBlockSize);
            randomAccessor.write(blocks.toByteArray());
            if (endBlock >= this.eofBlockNumber)
            {
               this.eofBlockNumber = endBlock + 1;
            }
         }
      }
      catch (IOException ex)
      {
         throw new SliceStoreIOException("Failed to write into slice store", ex);
      }
   }

   /**
    * Reads a block and parses content TODO: consider reading not max size. may be policy?
    * 
    * @see SliceStore
    */
   public DataSlice readImpl(SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException
   {
      return readImpl(Arrays.asList(new SliceName[]{
         name
      })).get(0);
   }

   /**
    * @see SliceStore
    */
   @Override
   public List<DataSlice> readImpl(List<SliceName> names) throws SliceStoreIOException,
         IllegalSourceNameException
   {
      ensureOperational();

      // Why not to allow reading empty list. makes little sense but may be simplify logic
      List<DataSlice> results = new ArrayList<DataSlice>(names.size());

      // Normal processing
      long blockNumbers[] = new long[names.size()];
      for (int i = 0; i < names.size(); i++)
      {
         blockNumbers[i] = convertNameIntoBlockNumber(names.get(i));
      }
      // We assume that numbers are given in the order no fancy order rearrangement is done
      // Potentially any sequence can be given, we try to optimize sequential operation
      int startIndex = 0;
      while (startIndex < names.size()) // Do we have anything not processed
      {
         // Find sequential numbers
         int index = startIndex;
         while (index < names.size() - 1 && (blockNumbers[index + 1] == blockNumbers[index] + 1))
         {
            index++;
         }
         List<DataSlice> partialResults =
               multipleReadImpl(names.subList(startIndex, index + 1), blockNumbers[startIndex],
                     blockNumbers[index]);
         // Copy results to the final destination
         results.addAll(partialResults);

         startIndex = index + 1;
      }
      return results;
   }

   /**
    * @see SliceStore
    */
   private List<DataSlice> multipleReadImpl(List<SliceName> names, long startBlock, long endBlock)
         throws SliceStoreIOException, IllegalSourceNameException
   {
      final long position = startBlock * this.realBlockSize;
      int nblocks = (int) (endBlock - startBlock + 1);

      try
      {

         List<DataSlice> results = new ArrayList<DataSlice>(nblocks);

         final byte blocksContent[] = new byte[this.realBlockSize * nblocks];

         synchronized (this.randomAccessor)
         {
            this.randomAccessor.seek(position);
            this.randomAccessor.read(blocksContent);
         }
         for (int i = 0; i < nblocks; i++)
         {
            final byte singleBlockContent[] = new byte[this.realBlockSize];
            System.arraycopy(blocksContent, i * this.realBlockSize, singleBlockContent, 0,
                  this.realBlockSize);

            final ByteArrayInputStream block = new ByteArrayInputStream(singleBlockContent);
            final DataInputStream input = new DataInputStream(block);

            final int magic = input.readInt();
            if (magic != EXISTING_BLOCK_MAGIC)
            {
               results.add(new DataSlice(names.get(i)));
               continue;
            }

            final long transactionId = input.readLong();
            final int length = input.readInt();
            // TODO: This is extremely inefficient - we copy 2
            final byte[] data = new byte[length];
            input.read(data);

            if (_logger.isTraceEnabled())
            {
               _logger.trace("Read Block #" + startBlock + i);
            }
            results.add(new DataSlice(names.get(i), transactionId, data));
         }
         return results;
      }
      catch (final IOException e)
      {
         throw new SliceStoreIOException("Error reading from file at position " + position, e);
      }
   }

   /**
    * Returns string representation of this SliceStore
    */
   public String toString()
   {
      return this.rootDir.getAbsolutePath() + "[" + this.getBlockSize() + "]" + "transactions: "
            + this.transactionMap.size();
   }

   public VaultDescriptor getVaultDescriptor() throws SliceStoreIOException, VaultIOException,
         VaultSecurityException, VaultDescriptorException, SliceStoreNotFoundException
   {
      return getVaultDescriptor(null, null);
   }

   /**
    * Attempts to convert a slice name into a long block number, if unable to throws an
    * IllegalSourceNameException
    * 
    * @param name
    * @return long representation of slice name
    * @throws IllegalSourceNameException
    */
   protected long convertNameIntoBlockNumber(final SliceName name)
         throws IllegalSourceNameException
   {
      try
      {
         return Long.parseLong(name.getSourceName().getName());
      }
      catch (final Exception ex)
      {
         throw new IllegalSourceNameException("The slice name must be an integer", ex);
      }
   }

   /**
    * Ensures that the slice store is operational, otherwise throws a SliceStoreStateException
    * 
    * @throws SliceStoreStateException
    */
   protected void ensureOperational()
   {
      if (!this.isOperational())
      {
         throw new IllegalStateException("The slice store must be operational");
      }
   }

   /**
    * Ensures that the slice store is not operational, otherwise throws a SliceStoreStateException
    * 
    * @throws SliceStoreStateException
    */
   protected void ensureNotOperational()
   {
      if (this.isOperational())
      {
         throw new IllegalStateException("The slice store must not be operational");
      }
   }

   /**
    * Ensures that a transaction is active, otherwise throws a transaction exception
    * 
    * @param transaction
    * @throws SliceStoreTransactionException
    * @throws SliceStoreStateException
    */
   protected void ensureActiveTransaction(SliceStoreTransaction transaction)
         throws SliceStoreTransactionException
   {
      if (!this.isActiveTransaction(transaction))
      {
         throw new SliceStoreTransactionException("The supplied transaction was not active");
      }
   }

   /**
    * Ensures that transaction exists but is not active, otherwise throws a transaction exception
    * 
    * @param transaction
    * @throws SliceStoreTransactionException
    * @throws SliceStoreStateException
    */
   protected void ensureNotActiveTransaction(SliceStoreTransaction transaction)
         throws SliceStoreTransactionException
   {
      if (this.isActiveTransaction(transaction))
      {
         throw new SliceStoreTransactionException("The supplied transaction should not be active");
      }
   }

   /**
    * Ensures that transaction is found, otherwise throws a transaction exception
    * 
    * @param transactionId
    * @throws SliceStoreTransactionException
    */
   protected void ensureTransactionFound(long transactionId) throws SliceStoreTransactionException
   {
      if (!this.transactionMap.containsKey(transactionId))
      {
         throw new SliceStoreTransactionException("A transaction by that ID already exists");
      }
   }
}
