//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Oct 26, 2007
//---------------------

package org.cleversafe.layer.slicestore.block;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javax.naming.ConfigurationException;

import org.apache.log4j.Logger;
import org.cleversafe.config.exceptions.IllegalConfigurationContentException;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.layer.slicestore.RepairAdvice;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.SliceStoreManagerBase;
import org.cleversafe.layer.slicestore.block.BlockMultiFileSliceStore.FileStorageDefinition;
import org.cleversafe.layer.slicestore.exceptions.RepairException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultACLFactory;
import org.cleversafe.vault.VaultPermission;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultIOException;

public class BlockMultiFileSliceStoreManager extends SliceStoreManagerBase
{
   private static Logger _logger = Logger.getLogger(BlockMultiFileSliceStoreManager.class);
   private static final String PARTITION_DEF = "Partition.";
   private static final String PROPERTIES_FILE_NAME = "Partition.usage";
   private static final String DIRECTORY_PROPERTY_NAME = "directory.";
   private static final String SIZE_PROPERTY_NAME = "size.";
   private static final long UNLIMITED_FILE_SIZE = -1L;

   /**
    * An object could be configured through configuration Example:
    * 
    */

   //private final String ACL_FILE_NAME = "ACL.der";
   // Class holds run time information about single directory storage
   public static class PartitionDescriptorState extends PartitionDefinition
   {
      private long usedSize = 0;

      public PartitionDescriptorState(PartitionDefinition partition)
      {
         super(partition);
      }

      /**
       * @return the usedSize
       */
      public long getUsedSize()
      {
         return this.usedSize;
      }

      /**
       * @param usedSize
       *           the usedSize to set
       */
      public void setUsedSize(long usedSize)
      {
         this.usedSize = usedSize;
      }

      /**
       * Add used storage on a directory
       * 
       * @param size
       */
      void useStorage(long size)
      {
         assert this.usedSize + size <= getTotalSize();
         this.usedSize += size;
      }

      /**
       * Add used storage on a directory
       * 
       * @param size
       */
      void releaseStorage(long size)
      {
         assert this.usedSize - size >= 0;
         this.usedSize -= size;
      }
   }

   private List<PartitionDescriptorState> partitionDescriptors;
   private String vaultAccessPath = null;
   private long maxFileSize = UNLIMITED_FILE_SIZE;
   private int listBlockCount = -1;

   ReentrantReadWriteLock sizeLock = new ReentrantReadWriteLock();

   //private List<VaultDescriptor> vaultDescriptors;

   public static class PartitionDefinition
   {
      private String dataPath;
      private long capacity; // Partition size in bytes

      public PartitionDefinition()
      {
         // For framework, will be initialized through setters
      }

      /**
       * Initialization all fields
       * 
       * @param dataPath
       * @param blockCapacity
       */
      public PartitionDefinition(String dataPath, long capacity)
      {
         super();
         this.dataPath = dataPath;
         this.capacity = capacity;
      }

      /**
       * Copy
       * 
       * @param other
       */
      public PartitionDefinition(PartitionDefinition other)
      {
         super();
         this.dataPath = other.dataPath;
         this.capacity = other.capacity;
      }

      /**
       * @return the dataPath
       */
      public String getDataPath()
      {
         return this.dataPath;
      }

      /**
       * @param dataPath
       *           the dataPath to set
       */
      public void setDataPath(String dataPath)
      {
         this.dataPath = dataPath;
      }

      /**
       * @return the blockCapacity
       */
      public long getTotalSize()
      {
         return this.capacity;
      }

      /**
       * @param blockCapacity
       *           the blockCapacity to set
       */
      public void setTotalSize(long totalSize)
      {
         this.capacity = totalSize;
      }

      /**
       * @param blockCapacity
       *           the blockCapacity to set as a string Numeric part is treated as long KB, MB, TB
       *           specifiers are allowed
       * @throws ConfigurationException
       */
      public void setCapacity(String capacityStr) throws IllegalConfigurationContentException
      {
         this.capacity = BlockMultiFileSliceStoreManager.getSizeInBytes(capacityStr);
      }
   }

   /**
    * A constructor required by configuration framework
    */
   public BlockMultiFileSliceStoreManager()
   {
   }

   /**
    * @param vaultAccessDirectory -
    *           a directory used to persist run time information
    * @param definitions -
    *           partition definitions
    * @param maxFileSize -
    *           maximum file size, UNLIMITED_FILE_SIZE if unlimited
    */
   public BlockMultiFileSliceStoreManager(
         String vaultAccessDirectory,
         PartitionDefinition[] definitions,
         long maxFileSize)
   {
      this.partitionDescriptors = new ArrayList<PartitionDescriptorState>(definitions.length);
      for (int i = 0; i < definitions.length; i++)
      {
         this.partitionDescriptors.add(new PartitionDescriptorState(definitions[i]));
      }
      setVaultAccessPath(vaultAccessDirectory);
      this.setMaxFileSize(maxFileSize);
      initialize();
   }

   /**
    * @param vaultAccessDirectory -
    *           a directory used to persist run time information
    * @param definitions -
    *           partition definitions
    * 
    */
   public BlockMultiFileSliceStoreManager(
         String vaultAccessDirectory,
         PartitionDefinition[] definitions)
   {
      this(vaultAccessDirectory, definitions, UNLIMITED_FILE_SIZE);
   }

   /**
    * Configuration Framework method:Setter for partition definition. Adds a new partition Should be
    * called before initialize
    * 
    * @param def
    */
   public void setPartitionDefinition(PartitionDefinition def)
   {
      if (this.partitionDescriptors == null)
      {
         this.partitionDescriptors = new ArrayList<PartitionDescriptorState>();
      }
      this.partitionDescriptors.add(new PartitionDescriptorState(def));
   }

   /**
    * Configuration Framework method. Setter for access path
    * 
    * @return the vaultAccessDirectory
    */
   private String getVaultAccessPath()
   {
      return this.vaultAccessPath;
   }

   /**
    * Setter for access directory. Used to persist run time information
    * 
    * @param vaultAccessDirectory
    *           the vaultAccessDirectory to set
    */
   public void setVaultAccessPath(String vaultAccessPath)
   {
      this.vaultAccessPath = vaultAccessPath;
   }

   /**
    * Returns maximum file size allowed to create for storage, M if no limits
    * 
    * @return the maxFileSize
    */
   public long getMaxFileSize()
   {
      return this.maxFileSize;
   }

   /**
    * Sets a maximum file size allowed to create for storage, UNLIMITED_FILE_SIZE if no limits
    * 
    * @param maxFileSize
    *           the maxFileSize to set
    */
   public void setMaxFileSize(long maxFileSize)
   {
      this.maxFileSize = maxFileSize;
   }

   /**
    * Allows to specify units such as KB, MB, GB, TB
    * 
    * @param maxFileSize
    */
   public void setMaxFileSize(String maxFileSizeWithUnits)
         throws IllegalConfigurationContentException
   {
      this.maxFileSize = getSizeInBytes(maxFileSizeWithUnits);
   }

   /**
    * Setter for the list block count (number of blocks to return per list call).
    * 
    * @param listBlockCount
    */
   public void setListBlockCount(int listBlockCount)
   {
      _logger.trace("Setting listBlockCount to " + listBlockCount);
      this.listBlockCount = listBlockCount;
   }

   /**
    * Getter for the list block count (number of blocks to return per list call).
    * 
    * @param listBlockCount
    */
   public int getListBlockCount()
   {
      return this.listBlockCount;
   }

   public synchronized void initialize()
   {
      if (this.maxFileSize < 0)
      {
         _logger.info("No maximum file size is specified");
      }
      if (this.vaultAccessPath == null)
      {
         throw new InitializationException("Vaults access directory must be set");
      }
      File vaultAccessDirectory = new File(this.vaultAccessPath);
      if (!vaultAccessDirectory.exists())
      {
         vaultAccessDirectory.mkdirs();
      }
      if (!vaultAccessDirectory.exists())
      {
         throw new InitializationException("Can't create an access directory:"
               + this.vaultAccessPath);
      }

      if (this.partitionDescriptors == null || this.partitionDescriptors.size() == 0)
      {
         throw new InitializationException("At least one partition should be defined");
      }
      try
      {
         loadServerProperties();
      }
      catch (SliceStoreIOException e)
      {
         throw new InitializationException("Failed to load properties", e);
      }
   }

   /**
    * 
    * @throws SliceStoreIOException
    */
   private void saveServerProperties() throws SliceStoreIOException
   {
      assert (this.partitionDescriptors.size() > 0);
      try
      {
         this.sizeLock.writeLock().lock();

         Properties props = new Properties();
         for (int i = 0; i < this.partitionDescriptors.size(); i++)
         {
            props.setProperty(PARTITION_DEF + (i + 1), Long.toString(this.partitionDescriptors.get(
                  i).getUsedSize()));
         }

         final File outFile = new File(constructManagerPropertyFileName());

         FileOutputStream fileOutputStream = null;
         try
         {
            fileOutputStream = new FileOutputStream(outFile);
            props.store(fileOutputStream, null);
         }
         catch (final IOException ex)
         {
            throw new SliceStoreIOException(
                  "Unable to save the BlockMultiFileSliceStoremanager properties", ex);
         }
         finally
         {
            try
            {
               if (fileOutputStream != null)
               {
                  fileOutputStream.close();
               }
            }
            catch (IOException ignore)
            {
               _logger.warn("Unable to close file " + fileOutputStream);
            }
         }
      }
      finally
      {
         this.sizeLock.writeLock().unlock();
      }
   }

   private void loadServerProperties() throws SliceStoreIOException
   {
      File propFile = new File(constructManagerPropertyFileName());
      if (propFile.exists())
      {
         try
         {
            this.sizeLock.writeLock().lock();

            FileInputStream fileInputStream = null;
            try
            {
               fileInputStream = new FileInputStream(propFile);
               Properties props = new Properties();
               props.load(fileInputStream);
               for (int i = 0; i < this.partitionDescriptors.size(); i++)
               {
                  // TODO: real exceptions
                  String usedSizeStr = props.getProperty(PARTITION_DEF + (i + 1));
                  if (usedSizeStr == null)
                  {
                     throw new RuntimeException("Partition used size for " + i
                           + " partition is not found");
                  }
                  this.partitionDescriptors.get(i).setUsedSize(Long.parseLong(usedSizeStr));
               }

            }
            catch (final IOException ex)
            {
               _logger.debug("BlockDevice properties file not found");
            }
            finally
            {
               try
               {
                  if (fileInputStream != null)
                  {
                     fileInputStream.close();
                  }
               }
               catch (IOException ignore)
               {
                  _logger.warn("Unable to close file " + fileInputStream);
               }
            }
         }
         finally
         {
            this.sizeLock.writeLock().unlock();
         }
      }
      else
      {
         _logger.warn("File " + constructManagerPropertyFileName()
               + "doesn't exist. Assume first use");
         saveServerProperties();
      }
   }

   @Override
   public SliceStore loadSliceStoreImpl(UUID vaultID) throws SliceStoreIOException,
         SliceStoreNotFoundException
   {
      List<BlockMultiFileSliceStore.FileStorageDefinition> dirDef = loadStoreProperties(vaultID);
      try
      {
         BlockMultiFileSliceStore sliceStore = new BlockMultiFileSliceStore(dirDef);
         if (this.listBlockCount > 0)
         {
            sliceStore.setListBlockCount(this.listBlockCount);
         }
         return sliceStore;
      }
      catch (IOException e)
      {
         throw new SliceStoreIOException("failed to load block multi file definition", e);
      }
   }

   public VaultPermission loadVaultPermissions(UUID vaultID, UUID accountID)
         throws SliceStoreIOException, SliceStoreNotFoundException
   {
      try
      {
         // TODO: remove direct dependency from File implementation. Replace with Resolver
         List<BlockMultiFileSliceStore.FileStorageDefinition> dirDef;
         try
         {
            dirDef = loadStoreProperties(vaultID);
         }
         catch (SliceStoreIOException e)
         {
            throw new VaultIOException("Failed to get Vault definition " + vaultID, e);
         }

         assert dirDef.size() > 0;
         String vaultACLFile = constructACLFileName(dirDef);

         FileInputStream fileInputStream = null;
         try
         {
            fileInputStream = new FileInputStream(vaultACLFile);
            VaultACLFactory factory = new VaultACLFactory();
            VaultACL vaultACL = factory.getInstance(fileInputStream);
            return vaultACL.getPermission(accountID);
         }
         catch (FileNotFoundException ex)
         {
            throw new SliceStoreNotFoundException("Could not file Vault ACL: " + vaultACLFile, ex);
         }
         finally
         {
            if (fileInputStream != null)
            {
               try
               {
                  fileInputStream.close();
               }
               catch (final IOException e)
               {
                  _logger.warn("Cannot close ACL file", e);
               }
            }
         }
      }
      catch (VaultException e)
      {
         throw new SliceStoreIOException("Cannot retrieve permissions for vault " + vaultID, e);
      }
   }

   private String constructACLFileName(List<FileStorageDefinition> dirDef)
   {
      return dirDef.get(0).getDataPath() + File.separator + SliceStoreBase.ACL_FILE_NAME;
   }

   private String constructManagerPropertyFileName()
   {
      return getVaultAccessPath() + File.separator + PROPERTIES_FILE_NAME;
   }

   private String constructSlicePropertyFileName(UUID vaultId)
   {
      return getVaultAccessPath() + File.separator + vaultId.toString() + ".properties";
   }

   /**
    * Size available on this store
    * 
    * @return
    */
   public long getRemainingSize()
   {
      this.sizeLock.readLock().lock();
      long remaining = 0;
      for (PartitionDescriptorState partition : this.partitionDescriptors)
      {
         remaining += partition.getTotalSize() - partition.getUsedSize();
      }
      this.sizeLock.readLock().unlock();

      return remaining;
   }

   /**
    * Creates a SliceStore after wrapping it with an unreliable one
    */
   @Override
   public void createSliceStoreImpl(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         UUID vaultID,
         VaultACL vaultACL,
         byte[] vaultDescriptorBytes,
         Map<String, String> options) throws SliceStoreExistsException, SliceStoreIOException
   {
      List<BlockMultiFileSliceStore.FileStorageDefinition> directoryList = null;
      long totalSliceSize = maxSliceSize + BlockMultiFileSliceStore.getBlockSizeOverhead();

      try
      {
         this.sizeLock.writeLock().lock();

         //How much space do we need
         directoryList = new LinkedList<BlockMultiFileSliceStore.FileStorageDefinition>();
         // Try to fit as many files into smallest index directory
         long numBlocks = sliceStoreSize / maxSliceSize;
         if (numBlocks < 2)
         {
            throw new SliceStoreIOException("Improper slice(" + maxSliceSize + ")/store size("
                  + sliceStoreSize + ") combination");
         }

         long startingBlockNumber = 0;

         // TODO: Randomize it to have a better spread between vaults
         int partionIndex = 0;
         for (; partionIndex < this.partitionDescriptors.size(); partionIndex++)
         {
            PartitionDescriptorState partition = this.partitionDescriptors.get(partionIndex);
            while (startingBlockNumber < numBlocks)
            {
               long remainingSizeToFit = (numBlocks - startingBlockNumber) * totalSliceSize;
               long partitionRemainingSize = partition.getTotalSize() - partition.getUsedSize();
               if (partitionRemainingSize / totalSliceSize < 1)
               { // No records fit, move to another partition
                  break;
               }
               long nextFileSize = 0;
               if (this.maxFileSize < 0)
               {
                  // No limitation for file size
                  nextFileSize = Math.min(partitionRemainingSize, remainingSizeToFit);
               }
               else
               {
                  nextFileSize =
                        Math.min(partitionRemainingSize, Math.min(remainingSizeToFit,
                              this.maxFileSize));
               }
               long nextFileBlocks = nextFileSize / totalSliceSize;
               // Actual next file size should be rounded to a block size
               nextFileSize = nextFileBlocks * totalSliceSize;
               partition.useStorage(nextFileSize);

               BlockMultiFileSliceStore.FileStorageDefinition dirEntry =
                     new BlockMultiFileSliceStore.FileStorageDefinition(partition.getDataPath()
                           + File.separator + vaultID.toString(), nextFileBlocks);
               directoryList.add(dirEntry);
               startingBlockNumber += nextFileBlocks;

            }
            if (startingBlockNumber == numBlocks)
            {
               // space distributed
               break;
            }
         }
         if (partionIndex == this.partitionDescriptors.size())
         {
            // Undo allocation changes
            restoreSize(directoryList, totalSliceSize);
            throw new SliceStoreIOException("Can't fit a given slice store into any partition");
         }
         saveServerProperties();
         _logger.info("Store allocation for " + vaultID + " has been completed");
      }
      finally
      {
         this.sizeLock.writeLock().unlock();
      }
      // Create a new store. Lock is released because it is a potentially long operation
      // If an error occurs a roll-back on changes made up front are to be performed
      try
      {
         saveStoreProperties(vaultID, directoryList);

         SliceStore sliceStore = loadSliceStoreImpl(vaultID);
         sliceStore.createStore(vaultType, maxSliceSize, sliceStoreSize, vaultACL,
               vaultDescriptorBytes, options);

         _logger.info("New store for id " + vaultID + " has been successfully created");
      }
      catch (SliceStoreNotFoundException ex)
      {
         _logger.error("New store for id " + vaultID + " failed due", ex);
         restorePartitionStateOnError(directoryList, totalSliceSize);
         throw new SliceStoreIOException("Error creating new slice store", ex);
      }
      catch (SliceStoreIOException ex)
      {
         _logger.error("New store for id " + vaultID + " failed due", ex);
         restorePartitionStateOnError(directoryList, totalSliceSize);
         throw ex;
      }
      catch (SliceStoreExistsException ex)
      {
         _logger.error("New store for id " + vaultID + " failed due", ex);
         restorePartitionStateOnError(directoryList, totalSliceSize);
         throw ex;
      }
   }

   /**
    * Restore partition state and save.
    * 
    * @param dirList
    * @param totalSliceSize
    */
   private void restorePartitionStateOnError(
         List<FileStorageDefinition> dirList,
         long totalSliceSize)
   {
      this.sizeLock.writeLock().lock();
      try
      {
         restoreSize(dirList, totalSliceSize);
         saveServerProperties();
      }
      catch (SliceStoreIOException ex)
      {
         // Nothing we could, should be repaired
         _logger.error("Error saving properties. Should be repaired", ex);
      }
      finally
      {
         this.sizeLock.writeLock().unlock();
      }
   }

   /**
    * Compensate size in case of an error
    */
   private void restoreSize(List<FileStorageDefinition> dirList, long totalSliceSize)
   {
      for (int dirIndex = 0; dirIndex < dirList.size(); dirIndex++)
      {
         String vaultDirectory = dirList.get(dirIndex).getDataPath();
         int vaultNameIndex = vaultDirectory.lastIndexOf(File.separator);
         String partitionDir = vaultDirectory.substring(0, vaultNameIndex);
         int partitionIndex = getPartitionIndexByDirectory(partitionDir);
         if (partitionIndex == -1)
         {
            _logger.error("Vault directory obtained " + vaultDirectory
                  + " from configuration doesn't correspond any partition");
         }
         else
         {
            // TODO: recalculate in bytes
            this.partitionDescriptors.get(partitionIndex).releaseStorage(
                  dirList.get(dirIndex).getBlockCapacity() * totalSliceSize);
         }
      }
   }

   private void saveStoreProperties(UUID vaultID, List<FileStorageDefinition> directoryList)
         throws SliceStoreIOException
   {
      Properties props = new Properties();
      for (int dirIndex = 0; dirIndex < directoryList.size(); dirIndex++)
      {
         props.setProperty(DIRECTORY_PROPERTY_NAME + (dirIndex + 1),
               directoryList.get(dirIndex).getDataPath());
         props.setProperty(SIZE_PROPERTY_NAME + (dirIndex + 1), Long.toString(directoryList.get(
               dirIndex).getBlockCapacity()));
      }
      File outFile = new File(constructSlicePropertyFileName(vaultID));

      FileOutputStream fileOutputStream = null;
      try
      {
         fileOutputStream = new FileOutputStream(outFile);
         props.store(fileOutputStream, null);
      }
      catch (IOException e)
      {
         throw new SliceStoreIOException("Can't store properties for vault" + vaultID, e);
      }
      finally
      {
         if (fileOutputStream != null)
         {
            try
            {
               fileOutputStream.close();
            }
            catch (IOException e)
            {
            }
         }
      }
   }

   private List<BlockMultiFileSliceStore.FileStorageDefinition> loadStoreProperties(UUID vaultId)
         throws SliceStoreIOException, SliceStoreNotFoundException
   {
      File propFile = new File(constructSlicePropertyFileName(vaultId));
      FileInputStream fileInputStream = null;
      try
      {
         fileInputStream = new FileInputStream(propFile);

         Properties props = new Properties();
         props.load(fileInputStream);

         List<BlockMultiFileSliceStore.FileStorageDefinition> dirList =
               new ArrayList<BlockMultiFileSliceStore.FileStorageDefinition>();

         for (int dirIndex = 1;; dirIndex++)
         {
            String dirName = props.getProperty(DIRECTORY_PROPERTY_NAME + (dirIndex));
            if (dirName == null)
            {
               break;
            }
            String sizeInBlocksStr = props.getProperty(SIZE_PROPERTY_NAME + (dirIndex));
            if (sizeInBlocksStr == null)
            {
               throw new SliceStoreIOException("Error reading slice store properties: " + vaultId
                     + " no size for " + dirIndex);
            }
            long sizeInBlocks = Long.parseLong(sizeInBlocksStr);
            BlockMultiFileSliceStore.FileStorageDefinition dir =
                  new BlockMultiFileSliceStore.FileStorageDefinition(dirName, sizeInBlocks);
            dirList.add(dir);
         }
         if (dirList.size() == 0)
         {
            throw new SliceStoreIOException("No relevant properies for slice store properties: "
                  + vaultId);
         }
         return dirList;
      }
      catch (FileNotFoundException e1)
      {
         throw new SliceStoreNotFoundException("Can't read slice store properties " + vaultId);
      }
      catch (IOException e)
      {
         throw new SliceStoreIOException("Error reading slice store properties " + vaultId);
      }
      finally
      {
         try
         {
            if (fileInputStream != null)
            {
               fileInputStream.close();
            }
         }
         catch (IOException e)
         {
         }
      }
   }

   /**
    * Deletes a slice store after wrapping it with an unreliable one
    * 
    * @throws SliceStoreNotFoundException
    */
   @Override
   public void deleteSliceStore(UUID vaultID, Map<String, String> options)
         throws SliceStoreIOException, SliceStoreNotFoundException
   {
      // Decrease released size
      List<BlockMultiFileSliceStore.FileStorageDefinition> dirList = loadStoreProperties(vaultID);
      try
      {
         this.sizeLock.writeLock().lock();

         SliceStore store = loadSliceStoreImpl(vaultID);
         assert store instanceof BlockMultiFileSliceStore;
         BlockMultiFileSliceStore bmfss = (BlockMultiFileSliceStore) store;
         try
         {
            bmfss.loadProperties();
         }
         catch (IOException e)
         {
            throw new SliceStoreIOException("Can't load properteis", e);
         };

         restoreSize(dirList, bmfss.getBlockSize()
               + BlockMultiFileSliceStore.getBlockSizeOverhead());
         super.deleteSliceStore(vaultID, options);
         File accessFile = new File(constructSlicePropertyFileName(vaultID));
         accessFile.delete();
         saveServerProperties();
      }
      finally
      {
         this.sizeLock.writeLock().unlock();
      }
   }

   /**
    * Checks whether slice store for a given vault is in 'good' shape
    * 
    * @param vailt
    * @return
    */
   public RepairAdvice checkIntegrity(Vault vault, VaultACL acl) throws RepairException
   {
      BlockMultiFileRepairAdvice advice = new BlockMultiFileRepairAdvice();
      // Check ACL exists
      // TODO: remove direct dependency from File implementation. Replace with Resolver

      // Check that access directory exists

      File accessPath = new File(getVaultAccessPath());
      if (!accessPath.exists() || !accessPath.isDirectory())
      {
         advice.addAdvice("Access directory '" + getVaultAccessPath() + "' doesn't exit");
         advice.addAdvice("Likely not enough permissions for a current user");
         advice.setCondition(RepairAdvice.StoreCondition.CORRUPT);
      }
      else
      {
         File partitionUsageFile = new File(constructManagerPropertyFileName());
         if (!partitionUsageFile.canRead())
         {
            advice.addAdvice("Partition usage '" + partitionUsageFile.getAbsolutePath()
                  + "' is not readable");
            advice.addAdvice("Likely not enough permissions for a current user");
            advice.setCondition(RepairAdvice.StoreCondition.CORRUPT);
         }
         // Keep going...
         File sliceDefinitionFile =
               new File(constructSlicePropertyFileName(vault.getVaultIdentifier()));
         if (!sliceDefinitionFile.exists())
         {
            advice.addAdvice("Slice definition '" + sliceDefinitionFile.getAbsolutePath()
                  + "' doesn't exit");

            // Verify whether stores do exist somewhere
            if (!findPartitionsForSliceStore(vault).isEmpty())
            {
               advice.addAdvice("Files for Slice Store vault " + vault.getVaultIdentifier()
                     + " have been found. Delete them first");
               //advice.set
               advice.setCondition(RepairAdvice.StoreCondition.PARTIAL_REPAIR);
            }
            else
            {
               advice.addAdvice("No trace of a slice store, Likely slice store was never created");
               advice.setProbableCondition(RepairAdvice.StoreCondition.FULL_REPAIR);
            }
         }

         List<BlockMultiFileSliceStore.FileStorageDefinition> dirDefs = null;
         try
         {
            dirDefs = loadStoreProperties(vault.getVaultIdentifier());

            if (!dirDefs.isEmpty())
            {
               BlockMultiFileSliceStore.FileStorageDefinition dirDef0 = dirDefs.get(0);

               String baseName =
                     new StringBuffer().append(dirDef0.getDataPath()).append(File.separator).toString();
               String storeProperties = baseName + BlockFileSliceStore.PROPERTIES_FILE_NAME;
               if (!new File(storeProperties).exists())
               {
                  advice.addAdvice("Slice Store Property file '" + storeProperties + "' is missing");
                  advice.setMissingPropertiesFile(storeProperties);
                  advice.setCondition(RepairAdvice.StoreCondition.PARTIAL_REPAIR);
               }

               String aclFileName = constructACLFileName(dirDefs);
               if (!new File(aclFileName).exists())
               {
                  advice.addAdvice("Acl file '" + aclFileName + "' is missing");
                  advice.setMissingAclFile(aclFileName);
                  advice.setProbableCondition(RepairAdvice.StoreCondition.PARTIAL_REPAIR);
               }
               String vaultDescriptorFileName =
                     baseName + vault.getVaultIdentifier()
                           + SliceStoreBase.VAULT_DESCRIPTOR_EXTENSION;
               // Make sure that all slice store files exist
               if (!new File(vaultDescriptorFileName).exists())
               {
                  advice.addAdvice("Descriptor file '" + vaultDescriptorFileName + "' is missing");
                  advice.setMissingDescriptorFile(vaultDescriptorFileName);
                  advice.setProbableCondition(RepairAdvice.StoreCondition.PARTIAL_REPAIR);
               }

               int i = 0;
               for (BlockMultiFileSliceStore.FileStorageDefinition dirDef : dirDefs)
               {
                  StringBuffer sliceFileName = new StringBuffer();
                  sliceFileName.append(dirDef.getDataPath()).append(File.separator).append(
                        BlockMultiFileSliceStore.BLOCK_FILE_BASE).append(new Integer(i).toString()).append(
                        BlockMultiFileSliceStore.BLOCK_FILE_EXT);
                  if (!new File(sliceFileName.toString()).exists())
                  {
                     advice.addAdvice("File '" + sliceFileName + "' is missing");
                     advice.addMissingFile(i, sliceFileName.toString());
                     advice.setProbableCondition(RepairAdvice.StoreCondition.PARTIAL_REPAIR);
                  }
                  i++;
               }
            }
            else
            {
               advice.addAdvice("File store definition is empty. Delete slice store properties file");
               advice.setProbableCondition(RepairAdvice.StoreCondition.CORRUPT);
            }
         }
         catch (SliceStoreIOException e)
         {
            advice.addAdvice("Failed to load vault storage definition for UUID "
                  + vault.getVaultIdentifier() + " Reason: " + e.getMessage());
            advice.addAdvice("Likely the server was wiped out");
            advice.setProbableCondition(RepairAdvice.StoreCondition.FULL_REPAIR);
         }
         catch (SliceStoreNotFoundException e)
         {
            advice.addAdvice("Failed to load vault storage definition for UUID "
                  + vault.getVaultIdentifier() + "Reason: " + e.getMessage());
            advice.addAdvice("Likely the server was wiped out");
            advice.setProbableCondition(RepairAdvice.StoreCondition.FULL_REPAIR);
         }
      }
      return advice;
   }

   private List<String> findPartitionsForSliceStore(Vault vault)
   {
      List<String> partitions = new LinkedList<String>();
      for (PartitionDefinition def : this.partitionDescriptors)
      {
         File vaultDir = new File(def.getDataPath() + File.separator + vault.getVaultIdentifier());
         if (vaultDir.exists())
         {
            partitions.add(vaultDir.getAbsolutePath());
         }
      }
      return partitions;
   }

   /**
    * Perform repair
    * 
    * @param vault
    */
   public void repair(RepairAdvice advice, Vault vault, byte[] vaultDescriptorBytes, VaultACL acl)
         throws RepairException, SliceStoreExistsException
   {
      if (!(advice instanceof BlockMultiFileRepairAdvice))
      {
         throw new IllegalArgumentException(
               "Wrong advice for BlockMultiFileStorage, expected 'MultiFileRepairAdvice' got "
                     + advice.getClass().getName());
      }
      BlockMultiFileRepairAdvice mtAdvice = (BlockMultiFileRepairAdvice) advice;

      // Full repair
      if (mtAdvice.getCondition() == RepairAdvice.StoreCondition.FULL_REPAIR)
      {
         doFullRepair(mtAdvice, vault, vaultDescriptorBytes, acl);
      }
      else if (mtAdvice.getCondition() == RepairAdvice.StoreCondition.PARTIAL_REPAIR)
      {
         doPartialRepair(mtAdvice, vault, vaultDescriptorBytes, acl);
      }
      else
      {
         _logger.info("No repair performed");
      }
   }

   public void doFullRepair(
         RepairAdvice advice,
         Vault vault,
         byte[] vaultDescriptorBytes,
         VaultACL acl) throws RepairException, SliceStoreExistsException
   {
      try
      {
         _logger.info("Recreating a store for vault " + vault.getVaultIdentifier());
         createSliceStore(vault.getType(), vault.getMaxSliceSize(), vault.getSliceStoreSize(0),
               vault.getVaultIdentifier(), acl, vaultDescriptorBytes, new HashMap<String, String>());
      }

      catch (SliceStoreIOException e)
      {
         throw new RepairException("Failed to create a slice store", e);
      }
   }

   public void doPartialRepair(
         RepairAdvice advice,
         Vault vault,
         byte[] vaultDescriptorBytes,
         VaultACL acl) throws RepairException, SliceStoreExistsException
   {
      try
      {
         if (!(advice instanceof BlockMultiFileRepairAdvice))
         {
            throw new IllegalArgumentException(
                  "Wrong advice for BlockMultiFileStorage, expected 'MultiFileRepairAdvice' got "
                        + advice.getClass().getName());
         }
         BlockMultiFileRepairAdvice mtAdvice = (BlockMultiFileRepairAdvice) advice;
         List<BlockMultiFileSliceStore.FileStorageDefinition> dirDefs =
               loadStoreProperties(vault.getVaultIdentifier());
         if (mtAdvice.getMissingAclFile() != null)
         {
            String aclFileName = constructACLFileName(dirDefs);
            _logger.info("Recreating vault ACL file" + aclFileName);

            File f = new File(aclFileName);
            f.getParentFile().mkdirs();
            f.createNewFile();
            FileOutputStream aclStream = new FileOutputStream(aclFileName);
            acl.flush(aclStream);
            aclStream.close();
         }
         if (mtAdvice.getMissingDescriptorFile() != null)
         {
            String baseName =
                  new StringBuffer().append(dirDefs.get(0).getDataPath()).append(File.separator).toString();
            String vaultDescriptorFileName =
                  baseName + vault.getVaultIdentifier() + SliceStoreBase.VAULT_DESCRIPTOR_EXTENSION;

            _logger.info("Recreating vault Descriptor file" + vaultDescriptorFileName);
            File f = new File(vaultDescriptorFileName);
            f.getParentFile().mkdirs();
            f.createNewFile();
            FileOutputStream descriptorStream = new FileOutputStream(vaultDescriptorFileName);
            descriptorStream.write(vaultDescriptorBytes);
            descriptorStream.close();
         }
         if (mtAdvice.getMissingPropertiesFile() != null)
         {
            BlockMultiFileSliceStore store = new BlockMultiFileSliceStore();
            store.setSynchronous(false);
            store.setBlockSize((int) vault.getMaxSliceSize());
            store.setMaxSliceNumber(vault.getSliceStoreSize(0) / vault.getMaxSliceSize());

            Properties props = store.convertToProperties();
            store.saveProperiesToFile(new File(mtAdvice.getMissingPropertiesFile()), props);

         }

         for (Map.Entry<Integer, String> entry : mtAdvice.getMissingStores().entrySet())
         {
            String storeName = entry.getValue();
            _logger.info("Recreating vault storage file" + storeName);

            File f = new File(storeName);
            f.getParentFile().mkdirs();
            f.createNewFile();
         }
      }
      catch (FileNotFoundException e)
      {
         throw new RepairException("File error while repairing", e);
      }
      catch (VaultIOException ex0)
      {
         throw new RepairException("Vault error while repairing", ex0);
      }
      catch (IOException ex1)
      {
         throw new RepairException("Error while repairing", ex1);
      }

      catch (SliceStoreIOException ex2)
      {
         throw new RepairException("Failed to create a slice store", ex2);
      }
      catch (SliceStoreNotFoundException ex3)
      {
         throw new RepairException("Failed to create a slice store", ex3);
      }
   }

   private int getPartitionIndexByDirectory(String partitionDir)
   {
      for (int ind = 0; ind < this.partitionDescriptors.size(); ind++)
      {
         if (this.partitionDescriptors.get(ind).getDataPath().equals(partitionDir))
         {
            return ind;
         }
      }
      return -1;

   }

   /**
    * Helper should be moved to utility
    * 
    * @param capacityStr
    * @return
    * @throws IllegalConfigurationContentException
    */
   public static long getSizeInBytes(String capacityStr)
         throws IllegalConfigurationContentException
   {
      // I hope that something more standard could be used
      capacityStr = capacityStr.trim();
      long baseValue = -1;
      long multiplier = 1;

      int index = 0;
      while (index != capacityStr.length() && capacityStr.charAt(index) >= '0'
            && capacityStr.charAt(index) <= '9')
      {
         index++;
      }
      if (index != 0)
      {
         baseValue = Long.parseLong(capacityStr.substring(0, index));
         while (index != capacityStr.length() && capacityStr.charAt(index) == ' ')
         {
            index++;
         }
         String multiplierStr = index == capacityStr.length() ? "" : capacityStr.substring(index);

         if (!multiplierStr.equals(""))
         {
            if (multiplierStr.equals("K") || multiplierStr.equals("KB"))
            {
               multiplier = 1024L;
            }
            else if (multiplierStr.equals("M") || multiplierStr.equals("MB"))
            {
               multiplier = 1024L * 1024;
            }
            else if (multiplierStr.equals("G") || multiplierStr.equals("GB"))
            {
               multiplier = 1024L * 1024 * 1024;
            }
            else if (multiplierStr.equals("T") || multiplierStr.equals("TB"))
            {
               multiplier = 1024L * 1024 * 1024 * 1024;
            }
            else
            {
               throw new IllegalConfigurationContentException("Bad multiplier " + multiplierStr);
            }
         }
         return multiplier * baseValue;
      }
      else
      {
         throw new IllegalConfigurationContentException("Can't parse " + capacityStr);
      }
   }

   @Override
   public String toString()
   {
      StringBuffer buf = new StringBuffer();
      buf.append("AccessPath:").append(this.vaultAccessPath).append("{");
      for (PartitionDescriptorState partition : this.partitionDescriptors)
      {
         buf.append(" [").append(partition.getDataPath()).append(",").append(
               partition.getTotalSize()).append(",").append(partition.getUsedSize()).append("]");
      }
      buf.append("}");
      return buf.toString();
   }
}
