//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: May 9, 2007
//---------------------

package org.cleversafe.layer.slicestore.unreliable;

import java.lang.reflect.Method;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.slicestore.NotificationHandler;
import org.cleversafe.layer.slicestore.SliceInfo;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;

/**
 * This class wraps a given Slice Store, only it accepts a user specifiable probability for the
 * likelihood that an exception, data corruption, read failure or invalid transaction will occur
 */
public class UnreliableSliceStore extends SliceStoreBase
{
   private static Logger _logger = Logger.getLogger(UnreliableSliceStore.class);

   SliceStore _sliceStore = null;

   // These values should be between 0 and 1
   double _corruptionFailureRate = 0.0;
   double _wrongTransactionFailureRate = 0.0;
   double _readReturnsEmptyDataRate = 0.0;

   long _transactionOffset = -1; // When wrong transaction occurs it will be offset by this amount
   int _dataSizeDifference = 0; // When data is corrupted length will be affected by this value

   long _normalPhaseTime = 0; // Time in milliseconds that we remain in normal phase
   long _delayPhaseTime = 0; // Time in milliseconds that we remain in delay phase

   long _continuousLatency = 0; // Time in milliseconds to delay each request
   long _simulatedBandwidth = 0; // Bits per second of bandwidth to simulate
   Object _networkLock = new Object(); // For bandwidth simulation, synchronize on this lock during send/recv

   private Map<String, Map<Class<?>, Double>> methodExceptionRate =
         new HashMap<String, Map<Class<?>, Double>>();

   /**
    * Default Constructor
    */
   public UnreliableSliceStore()
   {
   }

   /**
    * Constructs an UnreliableSliceStore with 0 probability of failure
    * 
    * @param databaseDir
    */
   public UnreliableSliceStore(SliceStore sliceStore)
   {
      setSliceStore(sliceStore);
      initialize();
   }

   /**
    * Constructs an UnreliableStore taking the slice store in addition to the read and write failure
    * rates
    * 
    * @param databaseDir
    * @param readFailureRate
    * @param writeFailureRate
    */
   public UnreliableSliceStore(
         SliceStore sliceStore,
         double readFailureRate,
         double writeFailureRate) throws NoSuchMethodException
   {
      this._sliceStore = sliceStore;
      this.setMethodExceptionRate("read", SliceStoreIOException.class, readFailureRate);
      this.setMethodExceptionRate("write", SliceStoreIOException.class, writeFailureRate);
      initialize();
   }

   /**
    * Constructs an UneliableStore taking the slice store in addition to the read, write, and data
    * corruption failure rates
    * 
    * @param sliceStore
    * @param readFailureRate
    * @param writeFailureRate
    * @param corruptionFailureRate
    * @throws NoSuchMethodException
    */
   public UnreliableSliceStore(
         SliceStore sliceStore,
         double readFailureRate,
         double writeFailureRate,
         double corruptionFailureRate) throws NoSuchMethodException
   {
      this._sliceStore = sliceStore;
      this.setMethodExceptionRate("read", SliceStoreIOException.class, readFailureRate);
      this.setMethodExceptionRate("write", SliceStoreIOException.class, writeFailureRate);
      this._corruptionFailureRate = corruptionFailureRate;
      initialize();
   }

   /**
    * Constructs an unreliable slice store taking the slice store to be wrapped, read, write,
    * corruption and wrong transaction failure rates
    * 
    * @param sliceStore
    * @param readFailureRate
    * @param writeFailureRate
    * @param corruptionFailureRate
    * @param wrongTransactionFailureRate
    * @throws NoSuchMethodException
    */
   public UnreliableSliceStore(
         SliceStore sliceStore,
         double readFailureRate,
         double writeFailureRate,
         double corruptionFailureRate,
         double wrongTransactionFailureRate) throws NoSuchMethodException
   {
      this._sliceStore = sliceStore;
      this.setMethodExceptionRate("read", SliceStoreIOException.class, readFailureRate);
      this.setMethodExceptionRate("write", SliceStoreIOException.class, writeFailureRate);
      this._corruptionFailureRate = corruptionFailureRate;
      this._wrongTransactionFailureRate = wrongTransactionFailureRate;
      initialize();
   }

   /**
    * Constructs an unreliable store taking the slice store to be wrapped and additionally the read
    * failure rate, write failure rate, data corruption failure rate, wrong transaction failure rate
    * and the rate at which null data is returned.
    * 
    * @param sliceStore
    * @param readFailureRate
    * @param writeFailureRate
    * @param corruptionFailureRate
    * @param wrongTransactionFailureRate
    * @param readReturnsEmptyDataRate
    * @throws NoSuchMethodException
    */
   public UnreliableSliceStore(
         SliceStore sliceStore,
         double readFailureRate,
         double writeFailureRate,
         double corruptionFailureRate,
         double wrongTransactionFailureRate,
         double readReturnsEmptyDataRate) throws NoSuchMethodException
   {
      this._sliceStore = sliceStore;
      this.setMethodExceptionRate("read", SliceStoreIOException.class, readFailureRate);
      this.setMethodExceptionRate("write", SliceStoreIOException.class, writeFailureRate);
      this._corruptionFailureRate = corruptionFailureRate;
      this._wrongTransactionFailureRate = wrongTransactionFailureRate;
      this._readReturnsEmptyDataRate = readReturnsEmptyDataRate;
      initialize();
   }

   /**
    * Setter for the slice store to be wrapped by the unreliable store
    * 
    * @param sliceStore
    *           to be wrapped
    */
   public void setSliceStore(SliceStore sliceStore)
   {
      this._sliceStore = sliceStore;
   }

   /**
    * Setter for the data corruption rate
    * 
    * @param corruptionFailureRate
    */
   public void setCorruptionFailureRate(double corruptionFailureRate)
   {
      this._corruptionFailureRate = corruptionFailureRate;
   }

   /**
    * Setter for the wrong transaction failure rate
    * 
    * @param wrongTransactionFailureRate
    */
   public void setWrongTransactionFailureRate(double wrongTransactionFailureRate)
   {
      this._wrongTransactionFailureRate = wrongTransactionFailureRate;
   }

   /**
    * Setter for empty data failure rate
    * 
    * @param readReturnsEmptyDataRate
    */
   public void setReadReturnsEmptyDataRate(double readReturnsEmptyDataRate)
   {
      this._readReturnsEmptyDataRate = readReturnsEmptyDataRate;
   }

   /**
    * Setter for the transaction offset used when returning an invalid transaction id. Must be
    * non-zero for the transaction failure to have an effect.
    * 
    * @param transactionOffset
    */
   public void setTransactionOffset(long transactionOffset)
   {
      this._transactionOffset = transactionOffset;
   }

   /**
    * Setter for data size difference, controls the size of corrupted data in relation to the
    * original non-corrupted data.
    * 
    * @param dataSizeDifference
    */
   public void setDataSizeDifference(int dataSizeDifference)
   {
      this._dataSizeDifference = dataSizeDifference;
   }

   /**
    * Used by configuration to set the unreliability of different methods
    * 
    * @param value
    */
   public void setUnreliability(String value)
   {
      String[] parameters = value.split(",");

      if (parameters.length != 3)
      {
         _logger.warn("Unable to parse the parameters from string: " + value);
         return;
      }

      double rate = Double.parseDouble(parameters[2]);

      try
      {
         String className = parameters[1];
         Class<?> exceptionClass = Class.forName(className);
         setMethodExceptionRate(parameters[0], exceptionClass, rate);
      }
      catch (NoSuchMethodException e)
      {
         _logger.warn("No such method exception was caught", e);
      }
      catch (ClassNotFoundException e)
      {
         _logger.warn("Class not found exception was caught", e);
      }
   }

   /**
    * Used by configuration to set the frequency and duration of delays
    * 
    * @param value
    */
   public void setDelayParameters(String value)
   {
      String[] parameters = value.split(",");

      if (parameters.length != 2)
      {
         _logger.warn("Unable to parse the parameters from string: " + value);
         return;
      }

      this._normalPhaseTime = Long.parseLong(parameters[0]); // Time to remain in the good phase
      this._delayPhaseTime = Long.parseLong(parameters[1]); // Time to remain in the delay phase
   }

   /**
    * Set the latency of each request
    * @param ms
    */
   public void setLatency(long ms)
   {
      this._continuousLatency = ms;
   }

   /**
    * Set the simulated bandwidth of this store
    * @param bitsPerSecond
    */
   public void setBandwidth(long bitsPerSecond)
   {
      this._simulatedBandwidth = bitsPerSecond;
   }

   /**
    * Determines whether or not we should delay execution of this method, if we are in the delay phase then
    * sleep the current thread until the normal phase begins again.
    */
   public void delay(int bytesTransferred)
   {
      if ((this._normalPhaseTime != 0) && (this._delayPhaseTime != 0))
      {
         long cycleTime = (this._normalPhaseTime + this._delayPhaseTime);
         long phaseTime = System.currentTimeMillis() % cycleTime;

         if (phaseTime < this._normalPhaseTime)
         {
            // We are in a normal phase, do not do anything
         }
         else
         {
            // We need to sleep the remaining time of the delay phase
            long remainingDelayPhaseTime = (cycleTime - phaseTime);
            try
            {
               Thread.sleep(remainingDelayPhaseTime);
            }
            catch (InterruptedException e)
            {
               throw new RuntimeException("Thread was unexpededly interrupted while sleeping");
            }
         }
      }

      if (this._continuousLatency > 0)
      {
         try
         {
            Thread.sleep(this._continuousLatency);
         }
         catch (InterruptedException ex)
         {
            // Ignore
         }
      }

      if (bytesTransferred > 0 && this._simulatedBandwidth > 0)
      {
         // bits / (bits/ms)
         long delayMs = (long) ((bytesTransferred * 8) / (this._simulatedBandwidth / 1000.));
         try
         {
            synchronized (this._networkLock)
            {
               Thread.sleep(delayMs);
            }
         }
         catch (InterruptedException ex)
         {
            // Ignore
         }
      }
   }

   /**
    * This method is used to set the rate at which any wrapped slice store method in will throw an
    * exception of a certain type.
    * 
    * @param methodName
    * @param exception
    * @param rate
    * @throws NoSuchMethodException
    */
   public void setMethodExceptionRate(String methodName, Class<?> exception, double rate)
         throws NoSuchMethodException
   {
      // Assert the error rate is within the proper bounds
      assert (rate >= 0 && rate <= 1) : "The rate must be between 0 and 1";

      // Ensure that the method exists for this class
      Method foundMethod = null;
      for (Method method : this.getClass().getMethods())
      {
         if (method.getName().equals(methodName))
         {
            foundMethod = method;
            break;
         }
      }
      if (foundMethod == null)
      {
         throw new NoSuchMethodException();
      }

      // Ensure the method can thrown the given exception
      List<Class<?>> exceptions = Arrays.asList(foundMethod.getExceptionTypes());
      if (!exceptions.contains(exception))
      {
         throw new IllegalArgumentException("The provided exception is not thrown by the method: "
               + methodName);
      }

      // Get the exceptions thrown map for this method
      Map<Class<?>, Double> exceptionsThrown = methodExceptionRate.get(methodName);
      if (exceptionsThrown == null)
      {
         exceptionsThrown = new HashMap<Class<?>, Double>();
         methodExceptionRate.put(methodName, exceptionsThrown);
      }

      // Save exception and rate in the exceptions thrown map
      exceptionsThrown.put(exception, rate);
   }

   /**
    * Initializes the UnreliableSliceStore, the slice store to be wrapped must be set
    */
   public void initialize()
   {
      assert this._sliceStore != null : "Slice store to be wrapped has not been set";
   }

   /**
    * Returns the identifier of the wrapped slice store, for debuggigng purposes
    */
   public String getIdentification()
   {
      return "Unreliable: " + getWrappedSliceStore().getIdentification();
   }

   /**
    * Throws an exception for a certain method given its name. These exceptions must be registered
    * using the setMethodExceptionRate method.
    * 
    * @param methodName
    * @throws SliceStoreLayerException
    */
   private void conditionalException(String methodName) throws SliceStoreLayerException
   {
      // Get the exceptions thrown map for this method
      Map<Class<?>, Double> exceptionsThrown = methodExceptionRate.get(methodName);

      if (exceptionsThrown != null)
      {
         for (Map.Entry<Class<?>, Double> entry : exceptionsThrown.entrySet())
         {
            if (Math.random() < entry.getValue()) // Throw the exception
            {
               Class<?> exceptionClass = entry.getKey();

               SliceStoreLayerException sliceStoreLayerException = null;
               try
               {
                  sliceStoreLayerException =
                        (SliceStoreLayerException) exceptionClass.getConstructor(String.class).newInstance(
                              "Generated unreliable exception");
               }
               catch (Exception e)
               {
                  throw new RuntimeException("Unable to construct SliceStoreLayerException", e);
               }

               _logger.info("Unreliable store: '" + methodName + "' throwing "
                     + sliceStoreLayerException.getClass().getName());
               throw sliceStoreLayerException;
            }
         }
      }

   }

   /**
    * @see SliceStore
    */
   public void createStore(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes,
         Map<String, String> options) throws SliceStoreIOException, SliceStoreExistsException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("createStore");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      this._sliceStore.createStore(vaultType, maxSliceSize, sliceStoreSize, accessControlList,
            vaultDescriptorBytes, options);
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void deleteStore() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("deleteStore");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      this._sliceStore.deleteStore();
   }

   public void updateStore(
         String vaultType,
         long maxSliceStize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes) throws SliceStoreIOException
   {
      throw new RuntimeException("not yet implemented");
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void beginTransaction(SliceStoreTransaction transaction) throws SliceStoreIOException,
         SliceStoreTransactionException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("beginTransaction");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreTransactionException)
            throw (SliceStoreTransactionException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(0);

      UnreliableSliceStoreTransaction utx = (UnreliableSliceStoreTransaction) transaction;
      _sliceStore.beginTransaction(utx.getChildTransaction());
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void commitTransaction(SliceStoreTransaction transaction) throws SliceStoreIOException,
         SliceStoreTransactionException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("commitTransaction");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreTransactionException)
            throw (SliceStoreTransactionException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(0);

      UnreliableSliceStoreTransaction utx = (UnreliableSliceStoreTransaction) transaction;
      _sliceStore.commitTransaction(utx.getChildTransaction());
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public SliceStoreTransaction createTransaction(long transactionId) throws SliceStoreIOException,
         SliceStoreTransactionException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("createTransaction");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreTransactionException)
            throw (SliceStoreTransactionException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(0);

      return new UnreliableSliceStoreTransaction(_sliceStore.createTransaction(transactionId));
   }

   /**
    * @see SliceStore
    */
   public void endSession() throws SliceStoreIOException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("endSession");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      _sliceStore.endSession();
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public boolean exists(SliceName name) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("exists");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof IllegalSourceNameException)
            throw (IllegalSourceNameException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(0);

      return _sliceStore.exists(name);
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public SliceStoreTransaction getTransaction(long transactionId) throws SliceStoreIOException,
         SliceStoreTransactionException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("getTransaction");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreTransactionException)
            throw (SliceStoreTransactionException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      return new UnreliableSliceStoreTransaction(_sliceStore.getTransaction(transactionId));
   }

   /**
    * @see SliceStore
    */
   public boolean isOperational()
   {
      return _sliceStore.isOperational();
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void listBegin() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("listBegin");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(0);

      _sliceStore.listBegin();
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void listBegin(SliceName name) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("listBegin");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof IllegalSourceNameException)
            throw (IllegalSourceNameException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(0);

      _sliceStore.listBegin(name);
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public List<SliceInfo> listContinue() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("listContinue");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(0);

      return _sliceStore.listContinue();
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void listStop() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("listStop");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(0);

      _sliceStore.listStop();
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public DataSlice readImpl(SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      List<SliceName> nameList = new ArrayList<SliceName>(1);
      nameList.add(name);
      List<DataSlice> dataSlices = readImpl(nameList);
      return dataSlices.get(0); // Should be only one
   }

   @Override
   public List<DataSlice> readImpl(List<SliceName> names) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      if (names.size() == 0)
      {
         return new ArrayList<DataSlice>();
      }

      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("read");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof IllegalSourceNameException)
            throw (IllegalSourceNameException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      List<DataSlice> retVal = _sliceStore.read(names);

      // Delay execution if we are in a delay phase
      int bytesRead = 0;
      for (DataSlice slice : retVal)
      {
         byte[] data = slice.getData();
         bytesRead += (data != null) ? data.length : 0;
      }
      delay(bytesRead);

      // Replace the returned data with random data
      if (Math.random() < _corruptionFailureRate)
      {
         for (int sourceIdx = 0; sourceIdx < names.size(); ++sourceIdx)
         {
            DataSlice corruptedSliceCandidate = retVal.get(sourceIdx);
            SecureRandom generator = new SecureRandom();
            byte[] originalBytes = corruptedSliceCandidate.getData();
            if (originalBytes != null)
            {
               byte[] corruptedBytes = new byte[originalBytes.length + _dataSizeDifference];
               generator.nextBytes(corruptedBytes);
               retVal.set(sourceIdx, new DataSlice(corruptedSliceCandidate.getSliceName(),
                     corruptedSliceCandidate.getTransactionId(), corruptedBytes));
            }
         }
      }

      // Replace returned data with null
      if (Math.random() < _readReturnsEmptyDataRate)
      {
         for (int sourceIdx = 0; sourceIdx < names.size(); ++sourceIdx)
         {
            DataSlice corruptedSliceCandidate = retVal.get(sourceIdx);
            retVal.set(sourceIdx, new DataSlice(corruptedSliceCandidate.getSliceName(),
                  corruptedSliceCandidate.getTransactionId(), null));
         }
      }

      // Replace the returned transaction with a random transaction ID
      if (Math.random() < _wrongTransactionFailureRate)
      {
         for (int sourceIdx = 0; sourceIdx < names.size(); ++sourceIdx)
         {
            DataSlice corruptedSliceCandidate = retVal.get(sourceIdx);
            retVal.set(sourceIdx, new DataSlice(corruptedSliceCandidate.getSliceName(),
                  corruptedSliceCandidate.getTransactionId() + _transactionOffset,
                  corruptedSliceCandidate.getData()));
         }
      }

      return retVal;
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void registerForNorNotification(NotificationHandler handler)
         throws SliceStoreIOException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("registerForNorNotification");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      _sliceStore.registerForNorNotification(handler);
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public boolean remove(SliceName name) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("remove");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof IllegalSourceNameException)
            throw (IllegalSourceNameException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(0);

      return _sliceStore.remove(name);
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void rollbackTransaction(SliceStoreTransaction transaction) throws SliceStoreIOException,
         SliceStoreTransactionException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("rollbackTransaction");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreTransactionException)
            throw (SliceStoreTransactionException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(0);

      UnreliableSliceStoreTransaction utx = (UnreliableSliceStoreTransaction) transaction;
      _sliceStore.rollbackTransaction(utx.getChildTransaction());
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void startSession() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("startSession");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      _sliceStore.startSession();
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void writeImpl(DataSlice data, boolean allowOverwriteNewer) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreTransactionException, SliceStoreQuotaException,
         SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("write");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreTransactionException)
            throw (SliceStoreTransactionException) ex;
         if (ex instanceof IllegalSourceNameException)
            throw (IllegalSourceNameException) ex;
         if (ex instanceof SliceStoreQuotaException)
            throw (SliceStoreQuotaException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      delay(data.getData().length);

      _sliceStore.write(data, allowOverwriteNewer);
   }

   @Override
   public void writeImpl(List<DataSlice> dataSlices, boolean overwriteNewer)
         throws SliceStoreIOException, IllegalSourceNameException, SliceStoreTransactionException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("write");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof IllegalSourceNameException)
            throw (IllegalSourceNameException) ex;
         if (ex instanceof SliceStoreTransactionException)
            throw (SliceStoreTransactionException) ex;
         if (ex instanceof SliceStoreQuotaException)
            throw (SliceStoreQuotaException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      // Delay execution if we are in a delay phase
      int bytesWritten = 0;
      for (DataSlice slice : dataSlices)
      {
         bytesWritten += slice.getData().length;
      }
      delay(bytesWritten);

      this._sliceStore.write(dataSlices, overwriteNewer);
   }

   /**
    * Returns the slice store which this unreliable slice store wraps
    * 
    * @return
    */
   public SliceStore getWrappedSliceStore()
   {
      return this._sliceStore;
   }

   public VaultDescriptor getVaultDescriptor() throws SliceStoreIOException, VaultIOException,
         VaultSecurityException, VaultDescriptorException, SliceStoreNotFoundException
   {
      return getWrappedSliceStore().getVaultDescriptor();
   }

   /**
    * @see SliceStore
    */
   public boolean isActiveTransaction(SliceStoreTransaction transaction)
   {
      // No exceptions
      UnreliableSliceStoreTransaction utx = (UnreliableSliceStoreTransaction) transaction;
      return this._sliceStore.isActiveTransaction(utx.getChildTransaction());
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public boolean listInProgress() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      // ////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      try
      {
         conditionalException("listInProgress");
      }
      catch (SliceStoreLayerException ex)
      {
         if (ex instanceof SliceStoreIOException)
            throw (SliceStoreIOException) ex;
         if (ex instanceof SliceStoreNotFoundException)
            throw (SliceStoreNotFoundException) ex;
      }
      // End throw of conditional exception
      // ////////////////////////////////////////////////////////////////////////////////////////////

      return _sliceStore.listInProgress();
   }

   private class UnreliableSliceStoreTransaction implements SliceStoreTransaction
   {
      private UnreliableSliceStore unreliableStore;
      private SliceStoreTransaction childTransaction;

      public UnreliableSliceStoreTransaction(SliceStoreTransaction childTransaction)
      {
         this.unreliableStore = UnreliableSliceStore.this;
         this.childTransaction = childTransaction;
      }

      public long getID()
      {
         return childTransaction.getID();
      }

      public SliceStore getSliceStore()
      {
         return unreliableStore;
      }

      public SliceStoreTransaction getChildTransaction()
      {
         return childTransaction;
      }
   }
}
