//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Nov 27, 2007
//---------------------

package org.cleversafe.layer.slicestore.unreliable;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreWrapperBase;

// TODO: Describe class or interface
public class UnreliableWrapper extends SliceStoreWrapperBase
{
   private static Logger _logger = Logger.getLogger(UnreliableWrapper.class);

   // These values should be between 0 and 1
   private double corruptionFailureRate = 0.0;
   private double wrongTransactionFailureRate = 0.0;
   private double readReturnsEmptyDataRate = 0.0;

   private long transactionOffset = -1;
   private int dataSizeDifference = 0;

   private List<String> unreliableMethods = new ArrayList<String>();
   
   private String delayParameters = "0,0";

   public SliceStore wrap(SliceStore store)
   {
      SliceStore sliceStore = super.wrap(store);
      if (this.unreliableMethods.size() == 0)
         return sliceStore;

      _logger.debug("Creating unreliable wrapper for the slice store");
      UnreliableSliceStore unreliableStore = new UnreliableSliceStore(sliceStore);

      // Set unreliability measures
      unreliableStore.setCorruptionFailureRate(corruptionFailureRate);
      unreliableStore.setWrongTransactionFailureRate(wrongTransactionFailureRate);
      unreliableStore.setReadReturnsEmptyDataRate(readReturnsEmptyDataRate);
      unreliableStore.setTransactionOffset(transactionOffset);
      unreliableStore.setDataSizeDifference(dataSizeDifference);
      unreliableStore.setDelayParameters(delayParameters);

      for (String value : unreliableMethods)
      {
         unreliableStore.setUnreliability(value);
      }

      return unreliableStore;
   }

   /**
    * Setter for the probability that a data corruption occurs
    * 
    * @param corruptionFailureRate
    */
   public void setCorruptionFailureRate(double corruptionFailureRate)
   {
      this.corruptionFailureRate = corruptionFailureRate;
   }

   /**
    * Setter for the probability that a wrong transaction id is set in the returned DataSlice
    * 
    * @param wrongTransactionFailureRate
    */
   public void setWrongTransactionFailureRate(double wrongTransactionFailureRate)
   {
      this.wrongTransactionFailureRate = wrongTransactionFailureRate;
   }

   /**
    * Setter for the probability that a read returns null data in the returned DataSlice
    * 
    * @param readReturnsEmptyDataRate
    */
   public void setReadReturnsEmptyDataRate(double readReturnsEmptyDataRate)
   {
      this.readReturnsEmptyDataRate = readReturnsEmptyDataRate;
   }

   /**
    * Setter for the offset of a transaction id when a wrong transaction id is set
    * 
    * @param transactionOffset
    */
   public void setTransactionOffset(long transactionOffset)
   {
      this.transactionOffset = transactionOffset;
   }

   /**
    * Setter for the change in data size that occurs when data is corrupted
    * 
    * @param dataSizeDifference
    */
   public void setDataSizeDifference(int dataSizeDifference)
   {
      this.dataSizeDifference = dataSizeDifference;
   }

   /**
    * Setter which sets the probability that a certain method will fail throwing a certain
    * exception.
    * 
    * @param unreliability
    *           Must be a comma separated string containing a method name, exception, and a double
    *           failure rate between 0 and 1
    */
   public void setUnreliability(String unreliability)
   {
      this.unreliableMethods.add(unreliability);
   }
   
   /**
    * Setter which sets the amount of time the unreliable slice store should operate in a 
    * normal phase vs. a delayed phase where all requests are blocked until the delay phase ends
    * 
    * @param delayParameters
    */
   public void setDelayParameters(String delayParameters)
   {
      this.delayParameters = delayParameters;
   }
}
