//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: Apr 30, 2007
//---------------------

package org.cleversafe.serialization;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.cleversafe.layer.protocol.BeginSessionRequest;
import org.cleversafe.layer.protocol.BeginSessionResponse;
import org.cleversafe.layer.protocol.BeginTransactionRequest;
import org.cleversafe.layer.protocol.BeginTransactionResponse;
import org.cleversafe.layer.protocol.CommitTransactionRequest;
import org.cleversafe.layer.protocol.CommitTransactionResponse;
import org.cleversafe.layer.protocol.CreateStoreRequest;
import org.cleversafe.layer.protocol.CreateStoreResponse;
import org.cleversafe.layer.protocol.EndSessionRequest;
import org.cleversafe.layer.protocol.EndSessionResponse;
import org.cleversafe.layer.protocol.ErrorResponse;
import org.cleversafe.layer.protocol.ErrorUnsolicited;
import org.cleversafe.layer.protocol.ExistsRequest;
import org.cleversafe.layer.protocol.ExistsResponse;
import org.cleversafe.layer.protocol.GetSupportedProtocolVersionsRequest;
import org.cleversafe.layer.protocol.GetSupportedProtocolVersionsResponse;
import org.cleversafe.layer.protocol.GetSupportedProtocolsRequest;
import org.cleversafe.layer.protocol.GetSupportedProtocolsResponse;
import org.cleversafe.layer.protocol.GridProtocolMessage;
import org.cleversafe.layer.protocol.GridProtocolOperation;
import org.cleversafe.layer.protocol.IntegrityVerificationRequest;
import org.cleversafe.layer.protocol.IntegrityVerificationResponse;
import org.cleversafe.layer.protocol.ListBeginRequest;
import org.cleversafe.layer.protocol.ListBeginResponse;
import org.cleversafe.layer.protocol.ListContinueRequest;
import org.cleversafe.layer.protocol.ListContinueResponse;
import org.cleversafe.layer.protocol.ListInProgressRequest;
import org.cleversafe.layer.protocol.ListInProgressResponse;
import org.cleversafe.layer.protocol.ListStopRequest;
import org.cleversafe.layer.protocol.ListStopResponse;
import org.cleversafe.layer.protocol.MultipleReadRequest;
import org.cleversafe.layer.protocol.MultipleReadResponse;
import org.cleversafe.layer.protocol.MultipleRemoveRequest;
import org.cleversafe.layer.protocol.MultipleRemoveResponse;
import org.cleversafe.layer.protocol.MultipleWriteRequest;
import org.cleversafe.layer.protocol.MultipleWriteResponse;
import org.cleversafe.layer.protocol.NoopRequest;
import org.cleversafe.layer.protocol.NoopResponse;
import org.cleversafe.layer.protocol.PasswordAuthenticationRequest;
import org.cleversafe.layer.protocol.PasswordAuthenticationResponse;
import org.cleversafe.layer.protocol.ProtocolOperation;
import org.cleversafe.layer.protocol.ProtocolSerializable;
import org.cleversafe.layer.protocol.RegisterForNotificationsRequest;
import org.cleversafe.layer.protocol.RegisterForNotificationsResponse;
import org.cleversafe.layer.protocol.RemoveStoreRequest;
import org.cleversafe.layer.protocol.RemoveStoreResponse;
import org.cleversafe.layer.protocol.RollbackTransactionRequest;
import org.cleversafe.layer.protocol.RollbackTransactionResponse;
import org.cleversafe.layer.protocol.VaultBindRequest;
import org.cleversafe.layer.protocol.VaultBindResponse;
import org.cleversafe.layer.protocol.WriteNotifyUnsolicited;
import org.cleversafe.layer.protocol.exceptions.OperationAlreadyRegisteredException;
import org.cleversafe.layer.protocol.exceptions.OperationNotRegisteredException;

/**
 * This class serializes and deserializes {@link GridProtocolMessage} objects to and from byte
 * arrays. It is required that an external registry registers all the protocol operation codes with
 * their corresponding classes using the static register() method, prior to calling deserialize.
 * 
 * @author Manish Motwani
 */
public abstract class GridProtocolMessageFactory implements ProtocolMessageFactory
{
   // Maps protocol message operation code onto the class of that message
   private static Map<Integer, Class<? extends ProtocolSerializable>> protocolMessageMap =
         new HashMap<Integer, Class<? extends ProtocolSerializable>>();

   public static final short GRID_PROTOCOL_VERSION = 1;
   public static final short GRID_PROTOCOL_ID = 1;

   // Register all protocol messages
   static
   {
      try
      {
         registerMessages();
      }
      catch (OperationAlreadyRegisteredException ignore)
      {
      }
   }

   private static void registerMessages() throws OperationAlreadyRegisteredException
   {
      // Noop
      GridProtocolMessageFactory.register(GridProtocolOperation.NOOP_REQUEST, NoopRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.NOOP_RESPONSE, NoopResponse.class);

      // Sessions
      GridProtocolMessageFactory.register(GridProtocolOperation.BEGIN_SESSION_REQUEST,
            BeginSessionRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.BEGIN_SESSION_RESPONSE,
            BeginSessionResponse.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.END_SESSION_REQUEST,
            EndSessionRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.END_SESSION_RESPONSE,
            EndSessionResponse.class);

      // Notifications
      GridProtocolMessageFactory.register(GridProtocolOperation.REGISTER_FOR_NOTIFICATIONS_REQUEST,
            RegisterForNotificationsRequest.class);
      GridProtocolMessageFactory.register(
            GridProtocolOperation.REGISTER_FOR_NOTIFICATIONS_RESPONSE,
            RegisterForNotificationsResponse.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.WRITE_NOTIFY_UNSOLICITED,
            WriteNotifyUnsolicited.class);

      // Errors
      GridProtocolMessageFactory.register(GridProtocolOperation.ERROR_UNSOLICITED,
            ErrorUnsolicited.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.ERROR_RESPONSE, ErrorResponse.class);

      // Listing
      GridProtocolMessageFactory.register(GridProtocolOperation.LIST_BEGIN_REQUEST,
            ListBeginRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.LIST_BEGIN_RESPONSE,
            ListBeginResponse.class);

      GridProtocolMessageFactory.register(GridProtocolOperation.LIST_CONTINUE_REQUEST,
            ListContinueRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.LIST_CONTINUE_RESPONSE,
            ListContinueResponse.class);

      GridProtocolMessageFactory.register(GridProtocolOperation.LIST_INPROGRESS_REQUEST,
            ListInProgressRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.LIST_INPROGRESS_RESPONSE,
            ListInProgressResponse.class);

      GridProtocolMessageFactory.register(GridProtocolOperation.LIST_STOP_REQUEST,
            ListStopRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.LIST_STOP_RESPONSE,
            ListStopResponse.class);

      // Transactions
      GridProtocolMessageFactory.register(GridProtocolOperation.BEGIN_TRANSACTION_REQUEST,
            BeginTransactionRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.BEGIN_TRANSACTION_RESPONSE,
            BeginTransactionResponse.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.COMMIT_TRANSACTION_REQUEST,
            CommitTransactionRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.COMMIT_TRANSACTION_RESPONSE,
            CommitTransactionResponse.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.ROLLBACK_TRANSACTION_REQUEST,
            RollbackTransactionRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.ROLLBACK_TRANSACTION_RESPONSE,
            RollbackTransactionResponse.class);

      // Slice Operations
      GridProtocolMessageFactory.register(GridProtocolOperation.EXISTS_REQUEST, ExistsRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.EXISTS_RESPONSE,
            ExistsResponse.class);

      // Authentication
      GridProtocolMessageFactory.register(GridProtocolOperation.PASSWORD_AUTHENTICATION_REQUEST,
            PasswordAuthenticationRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.PASSWORD_AUTHENTICATION_RESPONSE,
            PasswordAuthenticationResponse.class);

      // Authorization operations
      GridProtocolMessageFactory.register(GridProtocolOperation.VAULT_BIND_REQUEST,
            VaultBindRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.VAULT_BIND_RESPONSE,
            VaultBindResponse.class);

      // Store creation and removal operations
      GridProtocolMessageFactory.register(GridProtocolOperation.CREATE_STORE_REQUEST,
            CreateStoreRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.CREATE_STORE_RESPONSE,
            CreateStoreResponse.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.REMOVE_STORE_REQUEST,
            RemoveStoreRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.REMOVE_STORE_RESPONSE,
            RemoveStoreResponse.class);

      // Multiple read/write
      GridProtocolMessageFactory.register(GridProtocolOperation.MULTIPLE_READ_REQUEST,
            MultipleReadRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.MULTIPLE_READ_RESPONSE,
            MultipleReadResponse.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.MULTIPLE_WRITE_REQUEST,
            MultipleWriteRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.MULTIPLE_WRITE_RESPONSE,
            MultipleWriteResponse.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.MULTIPLE_REMOVE_REQUEST,
            MultipleRemoveRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.MULTIPLE_REMOVE_RESPONSE,
            MultipleRemoveResponse.class);

      // Versioning
      GridProtocolMessageFactory.register(GridProtocolOperation.GET_PROTOCOL_VERSIONS_REQUEST,
            GetSupportedProtocolVersionsRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.GET_PROTOCOL_VERSIONS_RESPONSE,
            GetSupportedProtocolVersionsResponse.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.GET_PROTOCOLS_REQUEST,
            GetSupportedProtocolsRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.GET_PROTOCOLS_RESPONSE,
            GetSupportedProtocolsResponse.class);

      GridProtocolMessageFactory.register(GridProtocolOperation.INTEGRITY_VERIFICATION_REQUEST,
            IntegrityVerificationRequest.class);
      GridProtocolMessageFactory.register(GridProtocolOperation.INTEGRITY_VERIFICATION_RESPONSE,
            IntegrityVerificationResponse.class);

   }

   /**
    * Registers the Operation with a class derived from ProtocolMessage.
    * 
    * @param operation
    *           The ProtocolOperation.
    * @param operationImpl
    *           The protocol message implementation class.
    * @throws OperationAlreadyRegisteredException
    */
   public static void register(
         ProtocolOperation operation,
         Class<? extends ProtocolSerializable> operationImpl)
         throws OperationAlreadyRegisteredException
   {
      if (protocolMessageMap.containsKey(operation.getOperationCode()))
      {
         throw new OperationAlreadyRegisteredException(
               "ProtocolOperation registration attempt occurred for an operation that is already registered",
               operation);
      }

      protocolMessageMap.put(operation.getOperationCode(), operationImpl);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.communication.protocol.ProtocolMessageFactory#getMessageTypes()
    */
   public Set<Class<?>> getMessageTypes()
   {
      return new HashSet<Class<?>>(protocolMessageMap.values());
   }

   public Class<? extends ProtocolSerializable> getMessageType(int operationCode)
         throws OperationNotRegisteredException
   {
      Class<? extends ProtocolSerializable> operationClass = protocolMessageMap.get(operationCode);
      if (operationClass == null)
      {
         throw new OperationNotRegisteredException("No operation with code " + operationCode
               + " has been registered");
      }
      return operationClass;
   }

   public static short getGridProtocolVersion()
   {
      return GRID_PROTOCOL_VERSION;
   }

   public static short getGridProtocolID()
   {
      return GRID_PROTOCOL_ID;
   }
}
