//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: May 10, 2007
//---------------------

package org.cleversafe.serialization.asn1;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.DERGeneralString;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.serialization.GridSerializable;

// Base class for exception ASN1 serialization
// For each class that needs to be serialized into ASN1 
// a derived class should be created that is capable of serializing the original one

public class ASN1ExceptionWrapper implements GridSerializable
{
   private final static Logger _logger = Logger.getLogger(ASN1ExceptionWrapper.class);

   // Each derived class must add itself with a type of exception it wraps
   // Each derived class should have a default public constructor
   /**
    * Map of all registered Exceptions
    */
   private final static Map<Class<?>, String> wrappedNames = new HashMap<Class<?>, String>();
   private final static Map<String, Class<?>> reversedWrappedNames =
         new HashMap<String, Class<?>>();

   static
   {
      // Load and register exceptions from exception configuration
      ExceptionRegistrator.loadAndRegister();

      // Register methods to be invoked for WrappedNetworkException
      // serialization
      try
      {
         ASN1.DERConverter.registerDefaultEncodeDecode(ASN1ExceptionWrapper.class,
               ASN1ExceptionWrapper.class.getMethod("encodeException", ASN1ExceptionWrapper.class),
               ASN1ExceptionWrapper.class.getMethod("decodeException", ASN1ExceptionWrapper.class,
                     ASN1Object.class));
      }
      catch (final Exception registrationException)
      {
         _logger.error("Failed to register WrapperNetworkException encoding/decoding methods",
               registrationException);
      }
   }

   // for testing only
   static void clearRegistry()
   {
      wrappedNames.clear();
      reversedWrappedNames.clear();
   }

   /**
    * Each Exception that is supposed to be serialized needs to register
    * 
    * @param type
    *           name under which exception known on the wire
    * @param exceptionClass
    *           to be serialized
    */
   public static void registerException(final String type, final Class<?> exceptionClass)
   {
      final String oldType = wrappedNames.get(exceptionClass);
      final Class<?> oldClass = reversedWrappedNames.get(type);
      if (oldType == null && oldClass == null)
      {
         wrappedNames.put(exceptionClass, type);
         reversedWrappedNames.put(type, exceptionClass);
      }
      else if (oldType != null)
      {
         throw new RuntimeException("Duplicate exception of class" + exceptionClass.getName());
      }
      else
      {
         throw new RuntimeException("Type " + type + " is registered more than once");
      }
   }

   /**
    * Special encoding for wrapped exception
    * 
    * @param wrapperExc
    *           wrapped exception
    * @return DERSequence that contains: SEQUENCE: ASN1Object (if there additional data), DERNull -
    *         if not SEQUENCE: Exception type (empty if not known) Cause
    */
   public static DERSequence encodeException(final ASN1ExceptionWrapper wrapperExc)
   {
      final Exception wrapped = wrapperExc.getOriginalException();
      assert wrapped != null;
      return encodeSingleException(wrapped);
   }

   public static DERSequence encodeSingleException(final Exception wrapped)
   {
      final Method[] codec =
            ASN1.DERConverter.getExternalCodecs(wrapped.getClass(), ASN1ExceptionWrapper.class);
      // Part that is serialized by a special codec
      ASN1Object nonDefaultPart = null;
      // Part serialized into string if nothing better is available
      ASN1Object nonDefaultPartString = null;

      // Is codec defined?
      if (codec != null)
      {
         try
         {
            nonDefaultPart = (ASN1Object) codec[0].invoke(null, wrapped);
         }
         catch (final Exception e)
         {
            _logger.error("Exception during exception encoding of type "
                  + wrapped.getClass().getName(), e);
            nonDefaultPart = new DERNull();
         }
      }
      else
      {
         nonDefaultPart = new DERNull();
      }
      if (wrapped.getMessage() != null)
      {
         // Just stringify it through message
         nonDefaultPartString = new DERGeneralString(wrapped.getMessage());
      }
      else
      {
         nonDefaultPartString = new DERNull();
      }

      final String exceptionType = wrappedNames.get(wrapped.getClass());
      DERGeneralString type = null;
      if (exceptionType != null)
      {
         type = new DERGeneralString(exceptionType);
      }
      else
      {
         _logger.warn("Unknown to serialization exception type " + wrapped);
         type = new DERGeneralString(""); // type unknown
      }

      Throwable cause = wrapped.getCause();
      ASN1Object derCause = null;
      if (cause != null && cause instanceof Exception)
      {
         derCause = encodeSingleException((Exception) cause);
      }
      else
      {
         derCause = new DERNull();
      }
      // this dance is because DERSequence has a lousy interface
      // Creating sequence: type, cause1, cause2...

      // Creating final sequence of 4 elements
      return new DERSequence(new ASN1Object[]{
            nonDefaultPart, nonDefaultPartString, type, derCause
      });

   }

   /**
    * Decoding wrapped exception Protocol definition see in encode() Exception will not be restored
    * in its entirety: only the most outer is, all causes are combined into list of strings.
    * 
    * @param wrappedExc
    *           wrapped exception to be resored
    * @param asnObj
    *           ASN.1 from which it is restored
    */
   public static void decodeException(final ASN1ExceptionWrapper wrappedExc, final ASN1Object asnObj)
   {
      wrappedExc.setOriginalException(decodeSingleException(asnObj));

   }

   public static Exception decodeSingleException(final ASN1Object asnObj)
   {
      // We expect sequence: 1-st type
      final DERSequence quartet = (DERSequence) asnObj;
      if (quartet.size() != 4)
      {
         throw new RuntimeException("Incorrect sequence. Should contain 4 parts");
      }
      final ASN1Object nonDefaultPart = (ASN1Object) quartet.getObjectAt(0);
      final ASN1Object nonDefaultStringPart = (ASN1Object) quartet.getObjectAt(1);
      String message = null;
      if (nonDefaultStringPart instanceof DERGeneralString)
      {
         message = ((DERGeneralString) nonDefaultStringPart).getString();
      }

      Exception wrapped = null;
      // Obtain type from ASN.1 message
      final DERGeneralString asnType = (DERGeneralString) quartet.getObjectAt(2);
      final String type = asnType.getString();
      if (type.equals("")) // None is provided, make it generic
      {
         wrapped = new Exception(message);
      }
      else
      {
         // Create an instance
         final Class<?> origExcClass = reversedWrappedNames.get(type);
         if (origExcClass != null)
         {
            // does it have a constructor with String?
            try
            {
               Constructor<?> ctr = origExcClass.getConstructor(String.class);
               wrapped = (Exception) ctr.newInstance(message);
            }
            catch (Exception e1)
            {
               try
               {
                  wrapped = (Exception) origExcClass.newInstance();
               }
               catch (final Exception creationException)
               {
                  _logger.error("Exception creation for '" + origExcClass.getName() + "' failed",
                        creationException);
               }
            }
         }
         if (wrapped == null)
         {
            wrapped = new Exception(message); // Better something than nothing
            _logger.error("Failed to create object of type '" + origExcClass.getName()
                  + "'. Creating a plain Exception instead");
         }
      }
      // Accurate or not we've got an original exception
      assert wrapped != null;

      final Method[] codec =
            ASN1.DERConverter.getExternalCodecs(wrapped.getClass(), ASN1ExceptionWrapper.class);

      // Does a wrapped class requiers specialized decoding?
      if (codec != null && !(nonDefaultPart instanceof DERNull))
      {
         try
         {
            codec[1].invoke(null, wrapped, nonDefaultPart);
         }
         catch (final Exception decodeException)
         {
            _logger.error("Exception during exception decoding for type "
                  + wrapped.getClass().getName(), decodeException);
         }
      }
      final ASN1Object cause = (ASN1Object) quartet.getObjectAt(3);
      if (!(cause instanceof DERNull))
      {
         wrapped.initCause(decodeSingleException(cause));
      }
      return wrapped;
   }

   private Exception originalException;

   public ASN1ExceptionWrapper()
   {
      this.originalException = null;
   }

   public ASN1ExceptionWrapper(final Exception originalException)
   {
      this.originalException = originalException;
   }

   /**
    * @return the exception
    */
   public Exception getOriginalException()
   {
      return originalException;
   }

   /**
    * @param exception
    *           the exception to set
    */
   public void setOriginalException(final Exception exception)
   {
      this.originalException = exception;
   }

   @Override
   public String toString()
   {
      if (originalException != null)
      {
         return "Wraps: " + originalException.toString();
      }
      else
      {
         return "Wraps: no exception";
      }
   }
}
