//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Aug 16, 2007
//---------------------

package org.cleversafe.serialization.asn1.custom;

import java.lang.reflect.Method;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.protocol.MultipleReadRequest;

/**
 * Implements custom ASN.1 <code>encode</code> and <code>decode</code> methods for
 * {@link MultipleReadRequest}.
 * 
 * @see MultipleReadRequest
 * @author Manish Motwani
 */
public class MultipleReadRequestSerialization implements CustomSerialization
{
   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getSerializableClass()
    */
   public Class<MultipleReadRequest> getSerializableClass()
   {
      return MultipleReadRequest.class;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getEncodeMethod()
    */
   public Method getEncodeMethod()
   {
      try
      {
         return MultipleReadRequestSerialization.class.getMethod("encodeMultipleReadRequest",
               new Class[]{
                  getSerializableClass()
               });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getDecodeMethod()
    */
   public Method getDecodeMethod()
   {
      try
      {
         return MultipleReadRequestSerialization.class.getMethod("decodeMultipleReadRequest",
               new Class[]{
                     getSerializableClass(), ASN1Object.class
               });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /**
    * Encode method for {@link MultipleReadRequest}.
    * 
    * @param multipleReadRequest
    * @return
    */
   public static ASN1Object encodeMultipleReadRequest(MultipleReadRequest multipleReadRequest)
   {
      SliceName[] sliceNames = multipleReadRequest.getSliceNames();

      if (sliceNames == null)
      {
         return new DERNull();
      }

      ASN1Encodable[] sourceNameArray = new ASN1Encodable[sliceNames.length];
      for (int i = 0; i < sliceNames.length; ++i)
      {
         if (sliceNames[i] == null)
         {
            sourceNameArray[i] = new DERNull();
         }
         else
         {
            sourceNameArray[i] = SliceNameSerialization.encodeSliceName(sliceNames[i]);
         }
      }

      return new DERSequence(sourceNameArray);
   }

   /**
    * Decode method for {@link MultipleReadRequest}.
    * 
    * @param multipleReadRequest
    * @param asnObj
    */
   public static void decodeMultipleReadRequest(
         MultipleReadRequest multipleReadRequest,
         ASN1Object asnObj)
   {
      if (asnObj instanceof DERNull)
      {
         multipleReadRequest.setSliceNames(new SliceName[0]);
      }
      else
      {
         assert asnObj instanceof DERSequence;
         DERSequence derMultipleReadRequest = (DERSequence) asnObj;

         if (derMultipleReadRequest.size() == 0)
         {
            multipleReadRequest.setSliceNames(new SliceName[0]);
         }
         else
         {
            SliceName[] sliceNames = new SliceName[derMultipleReadRequest.size()];
            for (int i = 0; i < derMultipleReadRequest.size(); ++i)
            {
               if (!(derMultipleReadRequest.getObjectAt(i) instanceof DERNull))
               {
                  assert derMultipleReadRequest.getObjectAt(i) instanceof DERSequence;
                  sliceNames[i] = new SliceName();
                  SliceNameSerialization.decodeSliceName(sliceNames[i],
                        (DERSequence) derMultipleReadRequest.getObjectAt(i));
               }
            }
            multipleReadRequest.setSliceNames(sliceNames);
         }
      }
   }

}
