//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Aug 22, 2007
//---------------------

package org.cleversafe.serialization.asn1.custom;

import java.lang.reflect.Method;

import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.DERGeneralString;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.layer.protocol.NoopResponse;

/**
 * Implements custom ASN.1 <code>encode</code> and <code>decode</code> methods for
 * {@link NoopResponse}.
 * 
 * @see NoopResponse
 * @author Manish Motwani
 */
public class NoopResponseSerialization implements CustomSerialization
{
   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getSerializableClass()
    */
   public Class<NoopResponse> getSerializableClass()
   {
      return NoopResponse.class;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getEncodeMethod()
    */
   public Method getEncodeMethod()
   {
      try
      {
         return NoopResponseSerialization.class.getMethod("encodeNoopResponse", new Class[]{
            getSerializableClass()
         });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getDecodeMethod()
    */
   public Method getDecodeMethod()
   {
      try
      {
         return NoopResponseSerialization.class.getMethod("decodeNoopResponse", new Class[]{
               getSerializableClass(), ASN1Object.class
         });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /**
    * Encode method for {@link NoopResponse}.
    * 
    * @param noopResponse
    * @return
    */
   public static ASN1Object encodeNoopResponse(NoopResponse noopResponse)
   {
      ASN1Object derMessage = null;

      if (noopResponse.getMessage() != null)
      {
         derMessage = new DERGeneralString(noopResponse.getMessage());
      }
      else
      {
         derMessage = new DERNull();
      }

      ASN1Object derPayload = null;
      if (noopResponse.getPayload() != null)
      {
         derPayload = new DEROctetString(noopResponse.getPayload());
      }
      else
      {
         derMessage = new DERNull();
      }

      ASN1Object exception =
            ResponseExceptionSerializationUtil.encodeResponseException(noopResponse);

      return new DERSequence(new ASN1Object[]{
            derMessage, derPayload, exception
      });
   }

   /**
    * Decode method for {@link NoopResponse}.
    * 
    * @param noopResponse
    * @param asnObj
    */
   public static void decodeNoopResponse(NoopResponse noopResponse, ASN1Object asnObj)
   {
      assert asnObj instanceof DERSequence;

      DERSequence derNoopResponse = (DERSequence) asnObj;

      assert derNoopResponse.size() == 3;
      assert derNoopResponse.getObjectAt(0) instanceof ASN1Object;
      ASN1Object derMessage = (ASN1Object) derNoopResponse.getObjectAt(0);

      if (derMessage instanceof DERGeneralString)
      {
         noopResponse.setMessage(((DERGeneralString) derMessage).getString());
      }
      else
      {
         assert derMessage instanceof DERNull;
         noopResponse.setMessage(null);
      }

      assert derNoopResponse.getObjectAt(1) instanceof ASN1Object;
      ASN1Object derPayload = (ASN1Object) derNoopResponse.getObjectAt(1);

      if (derPayload instanceof DEROctetString)
      {
         noopResponse.setPayload(((DEROctetString) derPayload).getOctets());
      }
      else
      {
         assert derPayload instanceof DERNull;
         noopResponse.setPayload(null);
      }

      assert derNoopResponse.getObjectAt(2) instanceof ASN1Object;

      ResponseExceptionSerializationUtil.decodeResponseException(noopResponse,
            (ASN1Object) derNoopResponse.getObjectAt(2));
   }
}
