//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Aug 14, 2007
//---------------------

package org.cleversafe.serialization.asn1.custom;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.DERGeneralString;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.layer.grid.SourceName;

/**
 * Implements custom ASN.1 <code>encode</code> and <code>decode</code> methods for
 * {@link SourceName}.
 * 
 * @see SourceName
 * @author Manish Motwani
 */
public class SourceNameSerialization implements CustomSerialization
{
   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getSerializableClass()
    */
   public Class<SourceName> getSerializableClass()
   {
      return SourceName.class;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getEncodeMethod()
    */
   public Method getEncodeMethod()
   {
      try
      {
         return SourceNameSerialization.class.getMethod("encodeSourceName", new Class[]{
            getSerializableClass()
         });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getDecodeMethod()
    */
   public Method getDecodeMethod()
   {
      try
      {
         return SourceNameSerialization.class.getMethod("decodeSourceName", new Class[]{
               getSerializableClass(), ASN1Object.class
         });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /**
    * Encode method for {@link SourceName}.
    * 
    * @param sourceName
    * @return
    */
   public static DERSequence encodeSourceName(SourceName sourceName)
   {
      DERGeneralString derName = new DERGeneralString(sourceName.getName());

      if (sourceName.getNamedExtensions() == null)
      {
         return new DERSequence(new DERObject[]{
               derName, new DERNull()
         });
      }

      DERSequence derMap = encodeMap(sourceName.getNamedExtensions());

      return new DERSequence(new DERObject[]{
            derName, derMap
      });
   }

   /**
    * Decode method for {@link SourceName}.
    * 
    * @param sourceName
    * @param asnObj
    */
   public static void decodeSourceName(SourceName sourceName, ASN1Object asnObj)
   {
      assert asnObj instanceof DERSequence;
      DERSequence derSourceName = (DERSequence) asnObj;
      assert derSourceName.size() == 2;

      assert derSourceName.getObjectAt(0) instanceof DERGeneralString;
      sourceName.setName(((DERGeneralString) derSourceName.getObjectAt(0)).getString());

      if (derSourceName.getObjectAt(1) instanceof DERNull)
      {
         sourceName.setNamedExtensions(null);
      }
      else
      {
         assert derSourceName.getObjectAt(1) instanceof DERSequence;

         Map<String, String> namedExtensions = new HashMap<String, String>();
         decodeMap(namedExtensions, (DERSequence) derSourceName.getObjectAt(1));
         sourceName.setNamedExtensions(namedExtensions);
      }
   }

   // pre-condition: map != null;
   private static DERSequence encodeMap(Map<String, String> map)
   {
      DERSequence[] seq = new DERSequence[map.size()];
      Set<String> keys = map.keySet();
      int i = 0;
      for (String key : keys)
      {

         seq[i++] = new DERSequence(new DERGeneralString[]{
               new DERGeneralString(key), new DERGeneralString(map.get(key))
         });
      }
      return new DERSequence(seq);
   }

   // pre-condition: map != null;
   private static void decodeMap(Map<String, String> map, ASN1Object asnObj)
   {
      assert asnObj instanceof DERSequence;

      DERSequence seq = (DERSequence) asnObj;
      map.clear();
      for (int i = 0; i < seq.size(); i++)
      {
         assert seq.getObjectAt(i) instanceof DERSequence;
         DERSequence pair = (DERSequence) seq.getObjectAt(i);
         String key = ((DERGeneralString) pair.getObjectAt(0)).getString();
         String value = ((DERGeneralString) pair.getObjectAt(1)).getString();
         map.put(key, value);
      }
   }

}
