//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: bcilfone
//
// Date: Jan 9, 2008
//---------------------

package org.cleversafe.test.grid;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.CorruptedSliceObserver;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.DeleteController;
import org.cleversafe.layer.grid.ReadController;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.WriteController;
import org.cleversafe.layer.grid.exceptions.ControllerDataTransformationException;
import org.cleversafe.layer.grid.exceptions.ControllerException;
import org.cleversafe.layer.grid.exceptions.ControllerGridStateUnknownException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerInformationDispersalException;
import org.cleversafe.layer.grid.exceptions.ControllerTransactionException;
import org.cleversafe.vault.Vault;

public class MockControllers
{
   private Map<String, byte[]> grid = new HashMap<String, byte[]>();

   private static Logger _logger = Logger.getLogger(MockControllers.class);

   public WriteController getWriteController()
   {
      return new MockWriteController();
   }

   public ReadController getReadController()
   {
      return new MockReadController();
   }

   public DeleteController getDeleteController()
   {
      return new MockDeleteController();
   }

   public void dumpInfo()
   {
      if (_logger.isDebugEnabled())
      {
         for (Map.Entry<String, byte[]> entry : grid.entrySet())
         {
            _logger.debug(entry.getKey() + ": " + (entry.getValue() == null ? "null" : entry.getValue().length + " bytes"));
         }
      }
   }

   public Map<String, byte[]> getData()
   {
      return grid;
   }

   public class MockWriteController implements WriteController
   {
      public void commit() throws ControllerTransactionException, ControllerIOException
      {
      }

      public long getTransactionId()
      {
         return 0;
      }

      public void rollback() throws ControllerTransactionException, ControllerIOException
      {
      }

      public void write(DataSource data) throws ControllerDataTransformationException, ControllerInformationDispersalException, ControllerIOException, ControllerIllegalSourceNameException, ControllerTransactionException, ControllerGridStateUnknownException
      {
         byte[] bytes = data.getData();

         byte[] dataCopy = new byte[bytes.length];
         System.arraycopy(bytes, 0, dataCopy, 0, dataCopy.length);

         MockControllers.this.grid.put(data.getName().getName(), dataCopy);

         if (_logger.isDebugEnabled())
         {
            _logger.debug("Wrote data to mock vault " + data.getName() + ": " + (bytes == null ? "null" : bytes.length + " bytes") + ", vault size: " + MockControllers.this.grid.size());

            if (_logger.isTraceEnabled())
            {
               _logger.trace("---- data: " + Arrays.toString(bytes));
            }
         }
      }

      public void write(List<DataSource> sources) throws ControllerDataTransformationException, ControllerInformationDispersalException, ControllerIOException, ControllerIllegalSourceNameException, ControllerTransactionException, ControllerGridStateUnknownException
      {
         for (DataSource source : sources)
         {
            write(source);
         }
      }

      public void addCorruptedSliceObserver(CorruptedSliceObserver corruptedSliceObserver)
      {
      }

      public Vault getVault()
      {
         return null;
      }

      public boolean isOperational()
      {
         return false;
      }

      public void setVault(Vault vault)
      {
      }

      public void shutdown() throws ControllerException
      {
      }
   }

   public class MockReadController implements ReadController
   {
      public DataSource read(SourceName name) throws ControllerDataTransformationException, ControllerInformationDispersalException, ControllerIOException, ControllerGridStateUnknownException, ControllerIllegalSourceNameException
      {
         byte[] data = MockControllers.this.grid.get(name.getName());

         if (_logger.isDebugEnabled())
         {
            _logger.debug("Read data from mock vault " + name + ": " + (data == null ? "null" : String.valueOf(data.length) + " bytes") + ", vault size: " + MockControllers.this.grid.size());

            if (_logger.isTraceEnabled())
            {
               _logger.trace("---- data: " + Arrays.toString(data));
            }
         }

         return new DataSource(name, 0, data);
      }

      public List<DataSource> read(List<SourceName> names) throws ControllerDataTransformationException, ControllerInformationDispersalException, ControllerIOException, ControllerGridStateUnknownException, ControllerIllegalSourceNameException
      {
         List<DataSource> sources = new ArrayList<DataSource>();

         for (SourceName name : names)
         {
            sources.add(read(name));
         }

         return sources;
      }

      public void addCorruptedSliceObserver(CorruptedSliceObserver corruptedSliceObserver)
      {
      }

      public Vault getVault()
      {
         return null;
      }

      public boolean isOperational()
      {
         return false;
      }

      public void setVault(Vault vault)
      {
      }

      public void shutdown() throws ControllerException
      {
      }
   }

   public class MockDeleteController implements DeleteController
   {

      public void delete(SourceName data) throws ControllerIOException, ControllerIllegalSourceNameException
      {
         MockControllers.this.grid.remove(data.getName());

         if (_logger.isDebugEnabled())
         {
            _logger.debug("Removed data from mock vault: " + data + ", vault size: " + MockControllers.this.grid.size());
         }
      }

      public void delete(List<SourceName> sources) throws ControllerIOException, ControllerIllegalSourceNameException
      {
         for (SourceName source : sources)
         {
            delete(source);
         }
      }

      public void addCorruptedSliceObserver(CorruptedSliceObserver corruptedSliceObserver)
      {
      }

      public Vault getVault()
      {
         return null;
      }

      public boolean isOperational()
      {
         return false;
      }

      public void setVault(Vault vault)
      {
      }

      public void shutdown() throws ControllerException
      {
      }
   }
}
