//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Zach
//
// Date: Jun 19, 2007
//---------------------

package org.cleversafe.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.apache.log4j.Logger;
import org.cleversafe.util.exceptions.FileResolutionException;
import org.cleversafe.util.exceptions.ResolutionException;

/**
 * File-based implementation of Resolver interface.
 *
 */
public class FileResolver implements Resolver
{
   private static Logger _logger = Logger.getLogger(FileResolver.class);
   
   // The resolved path; null if the path was not resolved yet.
   private File resolvedPath;
   
   
   /**
    * Constructs a file resolver. 
    * @param propertyName The java system property used to load path information.
    * @param defaultPath The default path used if a property is not set.
    * @param overridePath A path that should be used above preset values; null if regular
    *    file resolution should be used.
    * @throws IOException If file could not be resolved.
    */
   public FileResolver( String property, File defaultPath, File overridePath ) throws ResolutionException
   {
      _logger.debug("Resolving file = " + property );
      
      File propertyValue = 
         System.getProperty(property) == null ? null : new File(System.getProperty(property));
      
      if ( overridePath != null )
      {
         this.resolvedPath = overridePath;
         _logger.debug( "Resolved file (using override value) = " + overridePath );
      }
      else if ( propertyValue != null )
      {
         this.resolvedPath = propertyValue;
         _logger.debug( "Resolved file (using system property) = " + propertyValue );
      }
      else if ( defaultPath != null )
      {
         this.resolvedPath = defaultPath;
         _logger.debug( "Resolved file (using default value) = " + defaultPath );
      }
      else
      {
         throw new FileResolutionException( "could not resolve file." );
      }
   }
   

   /**
    * Resolves and opens the target file.
    * @return InputStream corresponding to the opened resource.
    */
   public InputStream getInputStream() throws FileResolutionException
   {
      FileInputStream retval = null;
      try
      {
         retval = new FileInputStream(this.resolvedPath);
      }
      catch (FileNotFoundException e)
      {
         throw new FileResolutionException(e);
      }
      
      return retval;
   }
   
   /**
    * Resolves and opens the target file.
    * @param resourceName Name of the target file, excluding path.
    * @param resourceLocation Location of the resource path in system properties.  
    * If not found, default to user's current home directory.
    * @return OutputStream corresponding to the opened resource.
    */
   public OutputStream getOutputStream() throws FileResolutionException
   {
      FileOutputStream retval = null;
      try
      {
         retval = new FileOutputStream(this.resolvedPath);
      }
      catch (FileNotFoundException e)
      {
         throw new FileResolutionException(e);
      }
      
      return retval;
   }

   /**
    * @throws ResolutionException
    * @see File#mkdirs()
    */
   public void mkdirs() throws FileResolutionException
   {
      _logger.trace("Creating directory tree for file: " + this.resolvedPath);
      if ( (!this.resolvedPath.getParentFile().exists()) &&
           (!this.resolvedPath.getParentFile().mkdirs()) )
      {
         throw new FileResolutionException("Could not create directory tree");
      } 
   }
   
   /**
    * Returns the resolved path.
    */
   public File getFile()
   {
      return this.resolvedPath;
   }
   
}
