//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: Ilya Volvovski
//
// Date: Jul 24, 2007
//---------------------

package org.cleversafe.util;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import org.apache.log4j.Logger;

import com.martiansoftware.jsap.JSAP;
import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Switch;
import com.martiansoftware.jsap.defaultsources.PropertyDefaultSource;

// This class provides a simple extension for regular JSAP by adding Help and version capabilities
public class JSAPCommandLineParser extends JSAP
{
   private static Logger _logger = Logger.getLogger(JSAPCommandLineParser.class);
   private static class JarVersionInfo
   {
      private String jarName;
      private String version = "XX";
      private Date buildDate;

      public JarVersionInfo(String jarName)
      {
         this.jarName = jarName;
         Calendar cal = Calendar.getInstance();
         // Set a fake date
         cal.set(2004, 1, 1); // Alleged cleversafe birthdate
         this.buildDate = cal.getTime();
      }

      /**
       * @return the buldDate
       */
      public Date getBuldDate()
      {
         return buildDate;
      }

      /**
       * @return the version
       */
      public String getVersion()
      {
         return version;
      }

      /**
       * @param buldDate
       *           the buldDate to set
       */
      protected void setBuildDate(Date buldDate)
      {
         this.buildDate = buldDate;
      }

      /**
       * @param version
       *           the version to set
       */
      protected void setVersion(String version)
      {
         this.version = version;
      }

      public String getJarName()
      {
         return jarName;
      }

      public void setJarName(String jarName)
      {
         this.jarName = jarName;
      }

      public String toString()
      {
         return this.jarName
               + " "
               + this.version
               + " built "
               + DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT).format(
                     this.buildDate);
      }
   }

   public static final String VERSION_SWITCH = "VERSION";
   public static final String HELP_SWITCH = "HELP";
   
   /**
    * If present in the JSAP XML resource either the default value or the user specified value
    * will be used to load a property default source file.
    */
   public static final String CONFIGURATION_OPTION = "configuration_file";

   public static String TITLE_ATTRIBUTE = "Cleversafe-Title";
   public static String VERSION_ATTRIBUTE = "Cleversafe-Version";
   public static String DATE_ATTRIBUTE = "Cleversafe-BuildDate";
   
   public static String COPYRIGHT_HOLDER = "Copyright (C) 2007-2008 Cleversafe, Inc.";
   public static String SOFTWARE_DESCRIPTION = "Cleversafe Dispersed Storage";

   private String commandFormat;
   private String commandDescription;

   /**
    * Logs environment and command line arguments to debug level.
    * <p>
    * Also logs version information if available.
    * @param args Application arguments.
    */
   public static void logEnvironment( String[] args )
   {
      _logger.trace("Starting application with libraries:");
      for ( String library : JSAPCommandLineParser.getLibraryVersionInfo() )
      {
         _logger.trace(library);
      }
      
      _logger.trace("Starting application with arguments: " + Arrays.toString(args));
      
      // Log environment variables
      Map<String,String> variables = System.getenv();
      Set<String> variableNames = variables.keySet();
      Iterator<String> nameIterator = variableNames.iterator();

      for (int index = 0; index < variableNames.size(); index++)
      {
           String name = (String) nameIterator.next();
           String value = (String) variables.get(name);
           _logger.trace( "Environment variable: " + name + " = " + value );
      }
   }
   
   /**
    * Creates a object based on convention where config file must reside
    * 
    * @param mainClass
    *           class for which this command line is used
    * @param jsapResource
    *           resource used to define command line
    * @param commandFormat
    *           command itseldf as presented in help
    * @param commandDescription
    *           command description as presented in help
    * @throws IOException
    *            error reading config
    * @throws JSAPException
    *            error parsing config
    */
   public JSAPCommandLineParser(
         Class<?> mainClass,
         String jsapResource,
         String commandFormat,
         String commandDescription) throws IOException, JSAPException
   {
      // This constructor throws NullPointerException when resource is not found. 
      // super(NamingHelper.getPackagePathFromClass(mainClass) + "/" + jsapResource);
      // Do it hard way
      super(convertResourceToURL(mainClass, jsapResource));
      init(commandFormat, commandDescription);
   }

   private static URL convertResourceToURL(Class<?> mainClass, String jsapResource) throws IOException
   {
      String xmlDefinition = NamingHelper.getPackagePathFromClass(mainClass) + "/" + jsapResource;
      URL url = mainClass.getClassLoader().getResource(xmlDefinition);
      if (url == null)
      {
         throw new IOException(xmlDefinition + ": JSAP resource not found");
      }
      return url;
   }

   /**
    * 
    * @param commandFormat
    *           command itself as presented in help
    * @param commandDescription
    *           command description as presented in help
    * @throws JSAPException
    *            error parsing config
    */
   public JSAPCommandLineParser(String commandFormat, String commandDescription)
         throws JSAPException
   {
      init(commandFormat, commandDescription);
   }

   private void init(String commandFormat, String commandDescription) throws JSAPException
   {
      this.commandFormat = commandFormat;
      this.commandDescription = commandDescription;

      Switch versionSwitch = new Switch(VERSION_SWITCH, 'v', "version");
      versionSwitch.setHelp("Outputs version information  and exits");

      Switch helpSwitch = new Switch(HELP_SWITCH, 'h', "help");
      helpSwitch.setHelp("Outputs usage information and exits");

      this.registerParameter(versionSwitch);
      this.registerParameter(helpSwitch);
   }

   /**
    * Convenience method
    */
   @Override
   public JSAPResult parse(String arg)
   {
      return super.parse(new String[]{
         arg
      });
   }

   @Override
   /**
    * returns JSAPResult if further processing is possible, null otherwise
    */
   public JSAPResult parse(String[] args)
   {
      JSAPResult parsingResults = super.parse(args);
      // Some boilerplate processing is encapsulated here

      // --help -h
      if (parsingResults.getBoolean(HELP_SWITCH))
      {
         showUsage();
         return null;
      }
      // --version -v
      else if (parsingResults.getBoolean(VERSION_SWITCH))
      {
         System.err.println(JSAPCommandLineParser.getVersionInfo());
         return null;
      }
      // if configuration file is specified in CONF_PATH,
      // re-parse after loading it as a default source
      else if (parsingResults.contains(CONFIGURATION_OPTION))
      {
         _logger.debug("Loading default data source: " + parsingResults.getString(CONFIGURATION_OPTION));
         this.registerDefaultSource( 
               new PropertyDefaultSource(parsingResults.getString(CONFIGURATION_OPTION), true) );
         parsingResults = super.parse(args);
      }
      
      // Print errors and show usage
      if (!parsingResults.success())
      {
         System.err.println();
         // Print all errors
         for (Iterator<?> errorIter = parsingResults.getErrorMessageIterator(); errorIter.hasNext();)
         {
            System.err.println(errorIter.next());
         }
         System.err.println();
         showUsage();
         return null;
      }

      // Leave it to the application to deal with it
      else
      {
         return parsingResults;
      }
   }

   private void showUsage()
   {
      System.err.println("Usage: " + this.commandFormat);
      System.err.println(this.commandDescription);
      System.err.println();
      System.err.println(this.getHelp());
   }

   
   /**
    * Extracts Cleversafe specific jar information from all jars and creates string version
    * lines for each one.
    */
   public static List<String> getLibraryVersionInfo()
   {
      List<String> versionList = new LinkedList<String>();

      String classPathString = System.getProperty("java.class.path");
      String[] classPaths = classPathString.split(File.pathSeparator);
      for (String classPath : classPaths)
      {
         _logger.trace("Checking classpath for version information: " + classPath); 
         if (!classPath.endsWith(".jar"))
         {
            continue;
         }

         File f = new File(classPath);
         try
         {
            JarFile jar = new JarFile(f);
            // _logger.debug(classPath);

            Manifest m = jar.getManifest();
            if (m != null)
            {
               _logger.trace("Checking manifest file");
               Attributes attrs = m.getMainAttributes();
               _logger.trace("Manifest file attributes: " + attrs.keySet().toString());
               String title = attrs.getValue(TITLE_ATTRIBUTE);
               if (title != null)
               {
                  JarVersionInfo jarVersion = new JarVersionInfo(title);

                  jarVersion.setVersion(attrs.getValue(VERSION_ATTRIBUTE));
                  String dateStr = attrs.getValue(DATE_ATTRIBUTE);
                  Date buildDate;
                  try
                  {
                     buildDate =
                           DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT).parse(
                                 dateStr);
                     jarVersion.setBuildDate(buildDate);
                  }
                  catch (ParseException e)
                  {
                     _logger.error("Could not parse date '" + dateStr, e);
                  }
                  versionList.add(jarVersion.toString());
               }
            }
            else
            {
               _logger.trace("Class path does not have manifest file");
            }
         }

         catch (IOException ignore)
         {
         }
      }
      return versionList;
   }
   
   
   /**
    * Extracts Cleversafe specific jar information from all jars and creates a string with proper
    * designation TODO: may be someone would be interested in not stringified jar info. Doubt it :(
    * 
    * @return
    */
   public static String getVersionInfo()
   {
      StringBuffer versionInformation = new StringBuffer("");
      versionInformation.append( SOFTWARE_DESCRIPTION + "\n\n" );
      
      for (String library : JSAPCommandLineParser.getLibraryVersionInfo())
      {
         versionInformation.append(library).append("\n");
      }
      
      versionInformation.append( "\n" + COPYRIGHT_HOLDER + "\n" );
      
      return versionInformation.toString();
   }
}
