//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Apr 27, 2008
//---------------------

package org.cleversafe.util;

import java.util.HashSet;
import java.util.Set;

/**
 * Utility class to iterate over all ways to choose R elements from a set of N
 */
public class NChooseR
{
   //private static Logger _logger = Logger.getLogger(NChooseR.class);

   // Size of element set
   private int n;

   // Number of elements to choose
   private int r;

   // First element this instance can use for iteration
   private int pivot;

   // Child for recursive definition
   private NChooseR child;

   /**
    * Construct an iterator for nCr
    * @param n
    * @param r
    */
   public NChooseR(int n, int r)
   {
      this(n, r, 0);
   }

   /**
    * Internal constructor
    * @param n
    * @param r
    * @param pivot
    */
   private NChooseR(int n, int r, int pivot)
   {
      assert n > 0 && n >= r && pivot <= n - r;

      this.n = n;
      this.r = r;
      this.pivot = pivot;

      this.child = (r > 1) ? new NChooseR(n, r - 1, pivot + 1) : null;
   }

   /**
    * Returns the next choice.  Returns null once all permutations have been returned
    * @return
    */
   public Set<Integer> next()
   {
      if (this.pivot <= this.n - this.r)
      {
         if (this.r > 1)
         {
            // For r > 1, iteration of index with child
            Set<Integer> childList = this.child.next();
            if (childList == null)
            {
               // When child is done, move to next pivot
               this.pivot++;
               if (this.pivot <= this.n - this.r)
               {
                  this.child = new NChooseR(this.n, this.r - 1, this.pivot + 1);
                  childList = this.child.next();
               }
            }

            if (childList != null)
            {
               childList.add(this.pivot);
               return childList;
            }
         }
         else if (this.r == 0)
         {
            // Special case -- For r == 0, return empty set
            this.pivot = Integer.MAX_VALUE;
            return new HashSet<Integer>();
         }
         else
         {
            // Base case -- For r == 1, straight iteration of pivot
            Set<Integer> l = new HashSet<Integer>();
            l.add(this.pivot++);
            return l;
         }
      }

      // Done with iteration
      return null;
   }

   public String toString()
   {
      return String.format("NCR(%d,%d) - %d elements", this.n, this.r, this.size());
   }

   /**
    * Returns the number of elements this iterator will produce
    * @return
    */
   public long size()
   {
      return calculateSize(this.n, this.r);
   }

   /**
    * Calculate size for a hypothetical nCr
    * @param n
    * @param r
    */
   public static long calculateSize(int n, int r)
   {
      return fact(n) / (fact(r) * fact(n - r));
   }

   /**
    * Factorial
    * @param n
    * @return
    */
   private static long fact(int n)
   {
      assert n >= 0;

      if (n <= 1)
      {
         return 1;
      }
      else
      {
         return n * fact(n - 1);
      }
   }
}
