//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Aug 28, 2007
//---------------------

package org.cleversafe.util;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * Helper package to aggregate a collection of objects descending from T
 * 
 * @param <T>
 */
public class TypeAggregator<T>
{
   // For each type in the collection, store the number of occurrences and an example
   private Map<Class<? extends T>, Tuple2<Integer, T>> map;

   /**
    * Create an empty aggregator
    */
   public TypeAggregator()
   {
      this.map = new HashMap<Class<? extends T>, Tuple2<Integer, T>>();
   }

   /**
    * Create an aggregator populated with objects
    * 
    * @param objects
    */
   public TypeAggregator(Collection<? extends T> objects)
   {
      super();
      for (T obj : objects)
      {
         this.add(obj);
      }
   }

   /**
    * Add an object to the aggregator
    * 
    * @param object
    */
   public void add(T object)
   {
      Class<? extends T> c = (Class<? extends T>) object.getClass();

      Tuple2<Integer, T> tuple = this.map.get(c);
      if (tuple == null)
      {
         this.map.put(c, new Tuple2<Integer, T>(1, object));
      }
      else
      {
         this.map.put(c, new Tuple2<Integer, T>(tuple.getFirst() + 1, tuple.getSecond()));
      }
   }

   /**
    * Get a mapping of class -> # of occurrences
    * 
    * @return
    */
   public Map<Class<? extends T>, Tuple2<Integer, T>> getMap()
   {
      return this.map;
   }

   /**
    * Get the most common class and the number of occurrences
    * 
    * @return
    */
   public Map.Entry<Class<? extends T>, Tuple2<Integer, T>> getMaximum()
   {
      Map.Entry<Class<? extends T>, Tuple2<Integer, T>> max = null;

      for (Map.Entry<Class<? extends T>, Tuple2<Integer, T>> entry : this.map.entrySet())
      {
         if (max == null || entry.getValue().getFirst() > max.getValue().getFirst())
         {
            max = entry;
         }
      }

      return max;
   }

   /**
    * Get the number of occurrences for the given class
    * 
    * @param type
    * @return
    */
   public int getCount(Class<? extends T> type)
   {
      Tuple2<Integer, T> tuple = this.map.get(type);
      return tuple != null ? tuple.getFirst() : 0;
   }

   /**
    * Get the total number of occurrences in the aggregator
    * 
    * @return
    */
   public int size()
   {
      int size = 0;

      for (Map.Entry<Class<? extends T>, Tuple2<Integer, T>> entry : this.map.entrySet())
      {
         size += entry.getValue().getFirst();
      }

      return size;
   }
}
