//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Aug 14, 2007
//---------------------

package org.cleversafe.util;

import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.PriorityBlockingQueue;

import org.cleversafe.exceptions.BaseException;

import com.evelopers.common.exception.CommonException;
import com.evelopers.common.exception.SystemException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.runtime.ControlledObject;
import com.evelopers.unimod.runtime.EventManager;
import com.evelopers.unimod.runtime.EventProvider;
import com.evelopers.unimod.runtime.ExceptionHandler;
import com.evelopers.unimod.runtime.ModelEngine;
import com.evelopers.unimod.runtime.context.Parameter;
import com.evelopers.unimod.runtime.context.StateMachineContext;
import com.evelopers.unimod.runtime.context.StateMachineContextImpl;

/**
 * Abstract base for Unimod state machine "worlds". Handles priority event dispatching, concurrent
 * tasks, exception events, and basic common world state
 */
public abstract class UnimodStateMachine
      implements
         EventProvider,
         ControlledObject,
         ExceptionHandler
{
   // private static Logger _logger = Logger.getLogger(UnimodStateMachine.class);

   /************************************************************************************************
    * Events
    ***********************************************************************************************/

   /** @unimod.event.descr Yes */
   public static final String EV_START = "start";

   /** @unimod.event.descr Yes */
   public static final String EV_YES = "yes";

   /** @unimod.event.descr No */
   public static final String EV_NO = "no";

   /** @unimod.event.descr OK */
   public static final String EV_OK = "ok";

   /** @unimod.event.descr Not OK */
   public static final String EV_NOT_OK = "not_ok";

   /** @unimod.event.descr Exception */
   public static final String EV_EXCEPTION = "exception";

   /** @unimod.event.descr An operation successfully completed */
   public static final String EV_OPERATION_COMPLETED = "operation_completed";

   /************************************************************************************************
    * State machine bookkeeping
    ***********************************************************************************************/

   // Is this state machine in a final state?
   protected boolean stateMachineFinished = false;

   // State machine context
   protected UnimodStateMachineContext stateMachineContext = new UnimodStateMachineContext();

   // Model engine
   protected ModelEngine modelEngine;

   // Model event manager - process events
   protected EventManager modelEventManager;

   // Executor for concurrent tasks
   protected ExecutorService taskExecutor = new ThreadlessExecutor();

   // Pending or executing tasks
   protected Set<ConcurrentTask> pendingTasks = new HashSet<ConcurrentTask>();

   // Queue of pending events
   protected BlockingQueue<Event> eventQueue = new PriorityBlockingQueue<Event>();

   /**
    * Set an alternative executor for this machine's tasks
    * 
    * @param executor
    */
   public void setExecutor(ExecutorService executor)
   {
      this.taskExecutor = executor;
   }

   /**
    * Function with ignorable events 
    * 
    * TODO: Consider introducing a flag for subclasses to check
    * mid-operation to see if they have been cancelled
    */
   protected abstract class ConcurrentFunction implements Runnable
   {
      // Any events generated by this function should be ignored
      boolean isIgnored = false;

      /**
       * Ignore any events generated by this task
       */
      public void ignore()
      {
         this.isIgnored = true;
      }

      /**
       * Used by a function to fire an event to the state machine
       * 
       * @param event
       */
      protected void fireEvent(UnimodEvent event)
      {
         // Fire event only if this task has not been ignored
         if (!this.isIgnored)
         {
            UnimodStateMachine.this.fireEvent(event, false);
         }
      }
   }

   /**
    * Cancellable concurrent task
    * 
    * @param <T>
    */
   protected static class ConcurrentTask extends FutureTask<Void>
   {
      ConcurrentFunction callable;

      /**
       * Create a new task from a ConcurrentFunction
       * 
       * @param callable
       */
      public ConcurrentTask(ConcurrentFunction callable)
      {
         super(callable, null);
         this.callable = callable;
      }

      /**
       * Ignore any events generated by this task
       */
      public void ignore()
      {
         this.callable.ignore();
      }

      /**
       * Cancel this task and ignore any events generated by its function if already running
       */
      @Override
      public boolean cancel(boolean mayInterruptIfRunning)
      {
         this.ignore();
         return super.cancel(mayInterruptIfRunning);
      }
   }

   /**
    * Prioritizable world event
    */
   protected static class UnimodEvent extends Event implements Comparable<UnimodEvent>
   {
      private static final long serialVersionUID = -6108309120346982477L;

      // Event priority between 0-Long.MAX_VALUE
      protected long priority;

      public UnimodEvent(String name)
      {
         super(name);
      }

      public UnimodEvent(String name, Parameter parameter)
      {
         super(name, parameter);
      }

      public UnimodEvent(String name, Parameter[] parameters)
      {
         super(name, parameters);
      }

      /**
       * Set event priority (0-1)
       * 
       * @param priority
       */
      public void setPriority(long priority)
      {
         this.priority = priority;
      }

      public int compareTo(UnimodEvent arg0)
      {
         if (arg0.priority == this.priority)
         {
            return 0;
         }
         return (this.priority < arg0.priority) ? -1 : 1;
      }
   }

   /**
    * WorldEvent to fire when an exception is thrown
    */
   protected static class ExceptionEvent extends UnimodEvent
   {
      private static final long serialVersionUID = 5046153655686390520L;

      protected Exception exception;

      public ExceptionEvent(BaseException e)
      {
         this(EV_EXCEPTION, e);
      }

      public ExceptionEvent(String event, Exception e)
      {
         super(event, new Parameter("exception", e));
         this.exception = e;
      }

      public Exception getException()
      {
         return this.exception;
      }
   }

   /**
    * Simple delegating class for our specific context
    */
   protected static class UnimodStateMachineContext implements StateMachineContext
   {
      private StateMachineContext context;
      private Exception exception;
      private Object result;
      private Event currentEvent;

      public UnimodStateMachineContext()
      {
         this.context = StateMachineContextImpl.create();
         this.exception = null;
         this.result = null;
      }

      /**
       * Set the current world event
       * 
       * @param event
       */
      public void setCurrentEvent(Event event)
      {
         this.currentEvent = event;
      }

      /**
       * Get the current world event
       * 
       * @return
       */
      public Event getCurrentEvent()
      {
         return this.currentEvent;
      }

      public void setException(Exception ex)
      {
         this.exception = ex;
      }

      public void setResult(Object o)
      {
         this.result = o;
      }

      public Object getResult() throws Exception
      {
         if (this.exception != null)
         {
            throw this.exception;
         }
         else
         {
            return this.result;
         }
      }

      public Context getApplicationContext()
      {
         return this.context.getApplicationContext();
      }

      public Context getEventContext()
      {
         return this.context.getEventContext();
      }

      public Context getUserContext()
      {
         return this.context.getUserContext();
      }
   }

   /**
    * EventProvider
    */
   public void dispose()
   {
      // Do nothing
   }

   /**
    * EventProvider
    */
   public void init(ModelEngine modelEngine) throws CommonException
   {
      this.stateMachineFinished = false;
      this.modelEngine = modelEngine;
      this.modelEventManager = this.modelEngine.getEventManager();

      // Set up exception handling for this machine
      this.modelEngine.getEventProcessor().addExceptionHandler(this);
   }

   /**
    * Fire an event to the state machine
    * 
    * @param event
    *           Event to fire
    * @param immediate
    *           If immediate, event will be placed on the front of the queue, otherwise it will be
    *           placed at the end. This can be used to trigger immediate state transitions
    */
   protected void fireEvent(UnimodEvent event, boolean immediate)
   {
      // Low priority first is the natural ordering
      long priority = immediate ? 0 : System.currentTimeMillis();
      event.setPriority(priority);
      this.eventQueue.add(event);
   }
   
   // FIXME: Add another version of fireEvent that respects original event priority

   /**
    * Fire an event to the state machine (non-immediate)
    * 
    * @param event
    */
   protected void fireEvent(UnimodEvent event)
   {
      this.fireEvent(event, false);
   }

   /**
    * Run a concurrent task
    * 
    * @param function
    * @return
    */
   protected Future<Void> runTask(ConcurrentFunction function)
   {
      ConcurrentTask task = new ConcurrentTask(function);
      this.pendingTasks.add(task);
      return (Future<Void>) this.taskExecutor.submit(task);
   }

   /**
    * Clear pending events and ignore events from all pending tasks
    * FIXME: Make it explicit that this should only be called from the controlling thread
    */
   protected void ignorePendingEvents()
   {
      for (ConcurrentTask task : this.pendingTasks)
      {
         task.ignore();
      }

      this.eventQueue.clear();
   }

   /**
    * Cancel all pending tasks and clear event queue
    * FIXME: Make it explicit that this should only be called from the controlling thread
    */
   protected void clearPendingEvents()
   {
      // Cancel pending tasks
      for (ConcurrentTask task : this.pendingTasks)
      {
         task.cancel(false);
      }

      this.pendingTasks.clear();
      this.eventQueue.clear();
   }

   /************************************************************************************************
    * Permissible world operations
    ***********************************************************************************************/

   /**
    * @unimod.action.descr Is the current status acceptable by policy?
    */
   public void fireOk(StateMachineContext context)
   {
      // Fire immediately
      this.fireEvent(new UnimodEvent(EV_OK), true);
   }

   /**
    * @unimod.action.descr Is the current status acceptable by policy?
    */
   public void fireNotOk(StateMachineContext context)
   {
      // Fire immediately
      this.fireEvent(new UnimodEvent(EV_NOT_OK), true);
   }

   /**
    * @unimod.action.descr Finish machine and return result or exception
    * 
    * FIXME: This action would be unnecessary if we figure out how to tell if the machine is in a
    * final state
    */
   public void exitMachine(StateMachineContext context)
   {
      this.stateMachineFinished = true;
   }

   /************************************************************************************************
    * State machine operation
    ***********************************************************************************************/

   /**
    * Handles exceptions from within the state machine
    * FIXME: Is this used?
    */
   public void handleException(StateMachineContext context, SystemException e)
   {
      UnimodStateMachine.this.exitMachine(context);

      // If a SystemException has a cause, that is our exception. Otherwise,
      // the exception is internal
      if (e.getParentException() instanceof BaseException)
      {
         ((UnimodStateMachineContext) context).setException((BaseException) e.getParentException());
      }
      else
      {
         ((UnimodStateMachineContext) context).setException(e);
      }
   }

   /**
    * Helper method dispatch prioritized events to the state machine. This is the single thread
    * driving event processing and machine execution
    */
   public Object runStateMachine(UnimodStateMachineContext context) throws Exception
   {
      // Fire start event
      this.fireEvent(new UnimodEvent(EV_START));

      // Dispatch events to machine as they arrive
      while (!this.stateMachineFinished)
      {
         Event event = this.eventQueue.take();
         context.setCurrentEvent(event);
         this.modelEventManager.handleAndWait(event, context);
      }

      // Determine state machine result, may throw an exception
      return context.getResult();
   }
}
