//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Feb 6, 2008
//---------------------

package org.cleversafe.util;

// TODO: Describe class or interface
public class Version implements Comparable<Version>
{
   private final String major;
   private final String minor;
   private final String build;

   public Version(String major, String minor, String build)
   {
      this.major = major;
      this.minor = minor;
      this.build = build;
   }

   @Override
   public String toString()
   {
      StringBuilder builder = new StringBuilder();
      builder.append(this.major);

      if (!this.minor.equals(""))
      {
         builder.append(".");
         builder.append(this.minor);
         if (!this.build.equals(""))
         {
            builder.append(".");
            builder.append(this.build);
         }
      }

      return builder.toString();
   }

   public static Version fromString(String versionString)
   {
      // Version string cannot end with a dot or contain two dots in a row
      if (versionString.endsWith(".") || versionString.indexOf("..") >= 0)
      {
         throw new IllegalArgumentException("Invalid version string: " + versionString);
      }

      String[] versions = versionString.split("\\.");

      for (String version : versions)
      {
         try
         {
            Integer.parseInt(version);
         }
         catch (NumberFormatException e)
         {
            throw new IllegalArgumentException("Invalid version string: " + versionString, e);
         }
      }

      String major = "";
      String minor = "";
      String build = "";

      if (versions.length == 1)
      {
         major = versionString;
      }
      else if (versions.length == 2)
      {
         major = versions[0];
         minor = versions[1];
      }
      else if (versions.length == 3)
      {
         major = versions[0];
         minor = versions[1];
         build = versions[2];
      }
      else
      {
         throw new IllegalArgumentException("Invalid version string: " + versionString);
      }

      return new Version(major, minor, build);
   }

   // This method normalizes the shorter string, so a String compare matches the numerical compare
   private int compareValues(String first, String second)
   {
      if (first.length() > second.length())
      {
         while (second.length() != first.length())
         {
            second = "0" + second;
         }
      }
      if (first.length() < second.length())
      {
         while (first.length() != second.length())
         {
            first = "0" + first;
         }
      }

      return first.compareTo(second);
   }

   public int compareTo(Version o)
   {
      if (this == o)
         return 0;

      if (!this.major.equals(o.major))
      {
         int comparison = compareValues(this.major, o.major);

         if (comparison != 0)
            return comparison;
      }

      if (!this.minor.equals(o.minor))
      {
         int comparison = compareValues(this.minor, o.minor);

         if (comparison != 0)
            return comparison;
      }

      return compareValues(this.build, o.build);
   }

   public String getMajor()
   {
      return this.major;
   }

   public String getMinor()
   {
      return this.minor;
   }

   public String getBuild()
   {
      return this.build;
   }

   @Override
   public int hashCode()
   {
      final int prime = 31;
      int result = 1;
      result = prime * result + ((this.build == null) ? 0 : this.build.hashCode());
      result = prime * result + ((this.major == null) ? 0 : this.major.hashCode());
      result = prime * result + ((this.minor == null) ? 0 : this.minor.hashCode());
      return result;
   }

   @Override
   public boolean equals(Object obj)
   {
      if (this == obj)
         return true;
      if (obj == null)
         return false;
      if (getClass() != obj.getClass())
         return false;
      final Version other = (Version) obj;
      if (this.build == null)
      {
         if (other.build != null)
            return false;
      }
      else if (!this.build.equals(other.build))
         return false;
      if (this.major == null)
      {
         if (other.major != null)
            return false;
      }
      else if (!this.major.equals(other.major))
         return false;
      if (this.minor == null)
      {
         if (other.minor != null)
            return false;
      }
      else if (!this.minor.equals(other.minor))
         return false;
      return true;
   }
}
