//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: Jun 8, 2007
//---------------------

package org.cleversafe.vault;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.codec.Codec;
import org.cleversafe.codec.Decoder;
import org.cleversafe.codec.Encoder;
import org.cleversafe.codec.exceptions.CodecUnknownSizeChangeException;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.InformationDispersalDecoder;
import org.cleversafe.ida.InformationDispersalEncoder;
import org.cleversafe.ida.exceptions.IDADecodeException;
import org.cleversafe.ida.exceptions.IDAEncodeException;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.ida.exceptions.IDANotInitializedException;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.GridTransaction;
import org.cleversafe.layer.grid.OperationScorecard;
import org.cleversafe.layer.grid.PlacedDataSlice;
import org.cleversafe.layer.grid.PlacedSliceName;
import org.cleversafe.layer.grid.PlacedSourceSliceSet;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.SourceSliceSet;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.exceptions.DataTransformationException;
import org.cleversafe.vault.exceptions.InformationDispersalException;

/**
 * Base abstract vault class that implements the common part of the {@link Vault} interface.
 * 
 * @author Manish Motwani
 */
public abstract class BaseVault implements Vault
{
   private static final Logger _logger = Logger.getLogger(BaseVault.class);

   protected InformationDispersalCodec idaCodec = null;
   protected List<Codec> datasourceCodecs = Collections.emptyList();
   protected List<Codec> reverseDatasouceCodecs = Collections.emptyList();

   /**
    * Read and write thresholds -- see notes in the Vault interface
    */
   protected int readThreshold = Integer.MIN_VALUE;
   protected int writeThreshold = Integer.MIN_VALUE;

   protected List<Codec> sliceCodecs = Collections.emptyList();
   protected List<Codec> reverseSliceCodecs = Collections.emptyList();

   protected List<SliceStore> sliceStores = Collections.emptyList();

   protected UUID vaultIdentifier = null;

   private String type = null;

   /**
    * 
    * @param type
    */
   protected BaseVault(String type)
   {
      this.type = type;
   }

   /**
    * 
    * @param type
    * @param idaCodec
    * @param codecs
    * @param sliceStores
    */
   public BaseVault(
         String type,
         InformationDispersalCodec idaCodec,
         List<Codec> codecs,
         List<Codec> sliceCodecs,
         List<SliceStore> sliceStores)
   {
      this.type = type;
      setInformationDispersalCodec(idaCodec);
      setDatasourceCodecs(codecs);
      setSliceCodecs(sliceCodecs);
      setSliceStores(sliceStores);
   }

   public void initialize() throws IDAInvalidParametersException
   {
      // Check read and write thresholds and assign reasonable defaults if necessary
      int t = this.idaCodec.getThreshold();
      int n = this.idaCodec.getNumSlices();
      if (this.writeThreshold == Integer.MIN_VALUE && this.readThreshold == Integer.MIN_VALUE)
      {
         // Default favors read
         this.readThreshold = t;
         this.writeThreshold = Math.max(t, n - this.readThreshold + 1);
      }
      else if (this.writeThreshold == Integer.MIN_VALUE)
      {
         this.writeThreshold = Math.max(t, n - this.readThreshold + 1);
      }
      else if (this.readThreshold == Integer.MIN_VALUE)
      {
         this.readThreshold = Math.max(t, n - this.writeThreshold + 1);
      }

      if (this.writeThreshold < t)
      {
         String err =
               String.format("Write threshold %d less than IDA threshold %d for vault %s",
                     this.writeThreshold, t, this.toString());
         _logger.error(err);
         throw new IDAInvalidParametersException(err);
      }
      if (this.readThreshold < t)
      {
         String err =
               String.format("Read threshold %d less than IDA threshold %d for vault %s",
                     this.readThreshold, t, this.toString());
         _logger.error(err);
         throw new IDAInvalidParametersException(err);
      }

      // Warn only if condition 3 (see Vault) is violated
      if (this.readThreshold + this.writeThreshold <= n)
      {
         _logger.warn(String.format(
               "Vault %s using a dangerous read/write threshold configuration: read(%d) + write(%d) <= width(%d)",
               this.toString(), this.readThreshold, this.writeThreshold, n));
      }
   }

   public UUID getVaultIdentifier()
   {
      return this.vaultIdentifier;
   }

   public void setVaultIdentifier(UUID vaultIdentifier)
   {
      this.vaultIdentifier = vaultIdentifier;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#getCodecs()
    */
   public List<Codec> getDatasourceCodecs()
   {
      return this.datasourceCodecs;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#getSliceCodecs()
    */
   public List<Codec> getSliceCodecs()
   {
      return this.sliceCodecs;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#getThreshold()
    */
   public int getThreshold()
   {
      return this.idaCodec.getThreshold();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#getInformationDispersalCodec()
    */
   public InformationDispersalCodec getInformationDispersalCodec()
   {
      return this.idaCodec;
   }

   protected long getEncodedDatasourceSize(long inputSize)
   {
      long retVal = inputSize;

      try
      {
         // Get data source size after codecs
         for (Codec codec : this.getDatasourceCodecs())
         {
            retVal = codec.getEncodedSize(retVal);
         }

      }
      catch (CodecUnknownSizeChangeException e)
      {
         inputSize = -1;
      }

      return retVal;
   }

   protected long getSliceSize(long inputSize)
   {
      long retVal = inputSize;

      try
      {
         // Get size of data after datasource codecs are applied
         retVal = getEncodedDatasourceSize(retVal);

         // Determine size of individual slices after dispersal
         retVal = this.getInformationDispersalCodec().getDispersedSize(retVal);
         retVal /= this.getInformationDispersalCodec().getNumSlices();

         // Get slice size after codecs
         for (Codec codec : this.getSliceCodecs())
         {
            retVal = codec.getEncodedSize(retVal);
         }

      }
      catch (CodecUnknownSizeChangeException e)
      {
         inputSize = -1;
      }

      return retVal;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#getStorage()
    */
   public List<SliceStore> getSliceStores()
   {
      return this.sliceStores;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#getSliceStoreSize(int)
    */
   public long getSliceStoreSize(int sliceStoreIndex)
   {
      // FIXME: implement
      return SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#getWidth()
    */
   public int getWidth()
   {
      return this.sliceStores.size();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#getType()
    */
   public String getType()
   {
      return this.type;
   }

   protected void setDatasourceCodecs(List<Codec> datasourceCodecs)
   {
      if (datasourceCodecs != null)
      {
         this.datasourceCodecs = datasourceCodecs;
      }
      else
      {
         this.datasourceCodecs = new ArrayList<Codec>();
      }
      this.reverseDatasouceCodecs = new ArrayList<Codec>(this.datasourceCodecs);
      Collections.reverse(this.reverseDatasouceCodecs);
   }

   protected void setSliceCodecs(List<Codec> sliceCodecs)
   {
      if (sliceCodecs != null)
      {
         this.sliceCodecs = sliceCodecs;
      }
      else
      {
         this.sliceCodecs = new ArrayList<Codec>();
      }
      this.reverseSliceCodecs = new ArrayList<Codec>(this.sliceCodecs);
      Collections.reverse(this.reverseSliceCodecs);
   }

   protected void setInformationDispersalCodec(InformationDispersalCodec idaCodec)
   {
      assert idaCodec != null;
      this.idaCodec = idaCodec;
   }

   protected void setSliceStores(List<SliceStore> sliceStores)
   {
      assert sliceStores != null;
      assert sliceStores.size() > 0;
      this.sliceStores = sliceStores;
   }

   /*
    * (non-Javadoc)
    * @see org.cleversafe.vault.Vault#setWriteThreshold(int)
    */
   public void setWriteThreshold(int writeThreshold)
   {
      this.writeThreshold = writeThreshold;
   }

   /*
    * (non-Javadoc)
    * @see org.cleversafe.vault.Vault#getWriteThreshold()
    */
   public int getWriteThreshold()
   {
      return this.writeThreshold;
   }

   /*
    * (non-Javadoc)
    * @see org.cleversafe.vault.Vault#setReadThreshold(int)
    */
   public void setReadThreshold(int readThreshold)
   {
      this.readThreshold = readThreshold;
   }

   /*
    * (non-Javadoc)
    * @see org.cleversafe.vault.Vault#getReadThreshold()
    */
   public int getReadThreshold()
   {
      return this.readThreshold;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#startSessions()
    */
   public void startSessions()
   {
      assert this.sliceStores != null;

      for (SliceStore sliceStore : this.sliceStores)
      {
         try
         {
            sliceStore.startSession();
         }
         catch (SliceStoreLayerException ex)
         {
            // Log but ignore error, the grid may still be in an acceptable state
            if (_logger.isDebugEnabled())
               _logger.debug("(recoverable) Could not start session for store " + sliceStore + ": "
                     + ex);
         }
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#endSessions()
    */
   public void endSessions()
   {
      assert this.sliceStores != null;

      for (SliceStore sliceStore : this.sliceStores)
      {
         try
         {
            sliceStore.endSession();
         }
         catch (SliceStoreLayerException ex)
         {
            // Log but ignore error, the grid may still be in an acceptable state
            if (_logger.isDebugEnabled())
               _logger.debug("Could not end session for store (recoverable): " + sliceStore);
         }
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#encodeSource(org.cleversafe.layer.grid.DataSource, long)
    */
   public DataSource encodeSource(DataSource source) throws DataTransformationException
   {
      byte[] data =
            this.encodeDataImpl(this.datasourceCodecs, source.getName(), source.getTransactionId(),
                  source.getData());
      return new DataSource(source.getName(), source.getTransactionId(), data);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#decodeSource(org.cleversafe.layer.grid.DataSource, long)
    */
   public DataSource decodeSource(DataSource source) throws DataTransformationException
   {
      byte[] data =
            this.decodeDataImpl(this.reverseDatasouceCodecs, source.getName(),
                  source.getTransactionId(), source.getData());
      return new DataSource(source.getName(), source.getTransactionId(), data);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#encodeSlice(org.cleversafe.layer.grid.DataSlice)
    */
   public DataSlice encodeSlice(DataSlice slice) throws DataTransformationException
   {
      byte[] data =
            this.encodeDataImpl(this.sliceCodecs, slice.getSliceName().getSourceName(),
                  slice.getTransactionId(), slice.getData());
      return new DataSlice(slice.getSliceName(), slice.getTransactionId(), data);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#decodeSlice(org.cleversafe.layer.grid.DataSlice)
    */
   public DataSlice decodeSlice(DataSlice slice) throws DataTransformationException
   {
      byte[] data =
            this.decodeDataImpl(this.reverseSliceCodecs, slice.getSliceName().getSourceName(),
                  slice.getTransactionId(), slice.getData());
      return new DataSlice(slice.getSliceName(), slice.getTransactionId(), data);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#assembleSource(org.cleversafe.layer.grid.SourceSliceSet)
    */
   public DataSource assembleSource(SourceSliceSet sliceSet) throws InformationDispersalException
   {
      long transactionId = Long.MAX_VALUE;
      InformationDispersalDecoder decoder;
      try
      {
         decoder = this.idaCodec.getDecoder();
      }
      catch (IDANotInitializedException e)
      {
         throw new InformationDispersalException("Error crearing decoder", e);
      }

      assert sliceSet.size() > 0 : "No slices provided";

      List<? extends DataSlice> slices = sliceSet.getSlices();
      List<byte[]> buffers = Arrays.asList(new byte[this.getWidth()][]);

      for (DataSlice slice : slices)
      {
         // Ensure that all slices have the same transaction id
         if (transactionId == Long.MAX_VALUE)
         {
            transactionId = slice.getTransactionId();
         }
         assert slice.getTransactionId() == transactionId : "Not all slices in the provided set have the same transaction ID";

         // NOTE: Assumes slices are indexed 0 <= x < P
         assert slice.getSliceName().getSliceIndex() < this.getWidth() : "Slice index "
               + slice.getSliceName().getSliceIndex() + " exceeds max value " + this.getWidth();
         buffers.set(slice.getSliceName().getSliceIndex(), slice.getData());
      }

      // Assemble dispersed slices
      byte[] data;
      try
      {
         data = decoder.finish(buffers);
      }
      catch (final IDADecodeException ex)
      {
         throw new InformationDispersalException("Error assembling source", ex);
      }
      catch (final IDANotInitializedException ex)
      {
         throw new RuntimeException("IDA not initialized");
      }

      return new DataSource(sliceSet.getName(), transactionId, data);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#disperseSource(org.cleversafe.layer.grid.DataSource)
    */
   public SourceSliceSet disperseSource(DataSource source) throws InformationDispersalException
   {
      InformationDispersalEncoder encoder;
      try
      {
         encoder = this.idaCodec.getEncoder();
      }
      catch (IDANotInitializedException e)
      {
         throw new InformationDispersalException("Error crearing encoder", e);
      }

      // Disperse a datasource into slices
      List<byte[]> sliceData;
      try
      {
         sliceData = encoder.finish(source.getData());
      }
      catch (final IDAEncodeException ex)
      {
         throw new InformationDispersalException("Error dispersing source", ex);
      }
      catch (final IDANotInitializedException ex)
      {
         throw new RuntimeException("IDA not initialized");
      }

      // Build a SourceSliceSet
      SourceSliceSet sourceSet = new SourceSliceSet(source.getName());
      for (int i = 0; i < sliceData.size(); ++i)
      {
         sourceSet.addSlice(new DataSlice(new SliceName(source.getName(), i),
               source.getTransactionId(), sliceData.get(i)));
      }

      return sourceSet;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#mapSliceNameToStore(org.cleversafe.layer.grid.SliceName)
    */
   public PlacedSliceName mapSliceNameToStore(SliceName name)
   {
      // Naive implementation, assigns slice N to the Nth SliceStore in the vault
      // NOTE: Assumes slices are indexed 0 <= x < P
      assert this.sliceStores.size() > name.getSliceIndex() : "Vault is not wide enough for slice: "
            + name;
      return new PlacedSliceName(name.getSourceName(), this.sliceStores.get(name.getSliceIndex()),
            name.getSliceIndex());
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#mapSourceNameToSliceNames(org.cleversafe.layer.slicestore.SourceName)
    */
   public List<PlacedSliceName> mapSourceNameToSliceNames(SourceName name)
   {
      List<PlacedSliceName> slices = new ArrayList<PlacedSliceName>(this.idaCodec.getNumSlices());
      for (int sliceIndex = 0; sliceIndex < this.idaCodec.getNumSlices(); ++sliceIndex)
      {
         slices.add(this.mapSliceNameToStore(new SliceName(name, sliceIndex)));
      }
      return slices;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#mapSliceToStore(org.cleversafe.layer.grid.DataSlice)
    */
   public PlacedDataSlice mapSliceToStore(DataSlice slice)
   {
      PlacedSliceName name = this.mapSliceNameToStore(slice.getSliceName());
      return new PlacedDataSlice(name, slice.getTransactionId(), slice.getData());
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#mapSliceSetToStore(org.cleversafe.layer.grid.SourceSliceSet)
    */
   public PlacedSourceSliceSet mapSliceSetToStore(SourceSliceSet slices)
   {
      PlacedSourceSliceSet set = new PlacedSourceSliceSet(slices.getName());

      for (DataSlice slice : slices.getSlices())
      {
         set.addSlice(this.mapSliceToStore(slice));
      }

      return set;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#applyAssemblyStack(org.cleversafe.layer.grid.SourceSliceSet)
    */
   public DataSource applyAssemblyStack(SourceSliceSet slices)
         throws InformationDispersalException, DataTransformationException
   {
      return this.applyAssemblyStack(slices, true);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#applyAssemblyStack(org.cleversafe.layer.grid.SourceSliceSet)
    */
   public DataSource applyAssemblyStack(SourceSliceSet slices, boolean applySliceCodecs)
         throws InformationDispersalException, DataTransformationException
   {
      // Apply post-dispersal decoding
      Exception lastException = null;
      SourceSliceSet decodedSlices;
      if (applySliceCodecs)
      {
         decodedSlices = new SourceSliceSet(slices.getName());
         for (DataSlice slice : slices.getSlices())
         {
            try
            {
               decodedSlices.addSlice(this.decodeSlice(slice));
            }
            catch (final DataTransformationException ex)
            {
               // Even if a slice fails decoding the source may still be restorable
               _logger.warn("(recoverable) Slice failed decoding", ex);
               lastException = ex;
            }
         }
      }
      else
      {
         // Skip slice codecs, it is assumed that they have already been applied
         decodedSlices = slices;
      }

      // Check that enough slices were decoded
      if (lastException != null && decodedSlices.size() < this.getThreshold())
      {
         assert lastException != null;
         throw new DataTransformationException("Only " + decodedSlices.size() + " of "
               + slices.size() + " could be decoded", lastException);
      }

      // Assemble data
      DataSource data = this.assembleSource(decodedSlices);

      // Apply pre-dispersal decoding
      DataSource decodedData = this.decodeSource(data);

      return decodedData;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#applyDispersalStack(org.cleversafe.layer.grid.DataSource)
    */
   public SourceSliceSet applyDispersalStack(DataSource data) throws InformationDispersalException,
         DataTransformationException
   {
      // Apply pre-dispersal codecs
      DataSource encodedData = this.encodeSource(data);

      // Disperse data
      SourceSliceSet slices = this.disperseSource(encodedData);

      // Apply post-dispersal codecs
      SourceSliceSet encodedSlices = new SourceSliceSet(data.getName());
      for (DataSlice slice : slices.getSlices())
      {
         // While it may be possible to write a datasource even if one or more slices fails
         // encoding, an encoding failure is usually indicative of a larger problem that should not
         // be ignored
         encodedSlices.addSlice(this.encodeSlice(slice));
      }

      return encodedSlices;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#rebuildSources(org.cleversafe.layer.grid.DataTransformationMatrix)
    */
   public List<DataSource> rebuildSources(OperationScorecard matrix)
         throws InformationDispersalException, DataTransformationException
   {
      return this.rebuildSources(matrix, true);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.Vault#rebuildSources(org.cleversafe.layer.grid.DataTransformationMatrix)
    */
   public List<DataSource> rebuildSources(OperationScorecard matrix, boolean applySliceCodecs)
         throws InformationDispersalException, DataTransformationException
   {
      List<DataSource> dataSources = new LinkedList<DataSource>();

      for (SourceName source : matrix.getSources())
      {
         // Attempt to rebuild this source
         Map<Long, PlacedSourceSliceSet> txMap = matrix.getRevisions(source);
         List<Tuple2<PlacedSourceSliceSet, Long>> rankedRevisions = this.rankRevisions(txMap);

         // Try each revision in order
         boolean rebuilt = false;
         Exception lastException = null;
         for (Tuple2<PlacedSourceSliceSet, Long> revisionTuple : rankedRevisions)
         {
            PlacedSourceSliceSet revision = revisionTuple.getFirst();
            long revisionNumber = revisionTuple.getSecond();

            // Handle the special case of datasource not found
            if (revisionNumber == GridTransaction.NOT_FOUND_TRANSACTION_ID)
            {
               dataSources.add(new DataSource(source, GridTransaction.NOT_FOUND_TRANSACTION_ID,
                     null));
               if (_logger.isDebugEnabled())
                  _logger.debug("Datasource not found (not an error): " + source);

               // Datasource not found, continue to the next source
               rebuilt = true;
               break;
            }
            else
            {
               try
               {
                  SourceSliceSet slices = revision.getSourceSliceSet();

                  // Rebuild revision
                  dataSources.add(this.applyAssemblyStack(slices, applySliceCodecs));

                  // Success, continue to the next source
                  rebuilt = true;
                  break;
               }
               catch (final InformationDispersalException ex)
               {
                  lastException = ex;
                  _logger.warn("(recoverable) IDA failure assembling revision", ex);
               }
               catch (final DataTransformationException ex)
               {
                  lastException = ex;
                  _logger.warn("(recoverable) Decode failure assembling revision", ex);
               }
            }
         }

         // If this source cannot be rebuilt, throw an exception
         if (!rebuilt)
         {
            // Marshal to known exception types
            if (lastException instanceof InformationDispersalException)
            {
               throw (InformationDispersalException) lastException;
            }
            else if (lastException instanceof DataTransformationException)
            {
               throw (DataTransformationException) lastException;
            }
            else
            {
               throw new RuntimeException("Internal error - unexpected exception", lastException);
            }
         }
      }

      assert dataSources.size() == matrix.getSources().size() : "Failed to report a source that could not be rebuilt";
      return dataSources;
   }

   /*
    * DEVELOPER'S NOTE: This function is deprecated, but still used by the old SimpleReadController
    */
   @Deprecated
   public List<Tuple2<SourceName, Integer>> getIncompleteSources(OperationScorecard matrix)
   {
      List<Tuple2<SourceName, Integer>> incompleteSources =
            new LinkedList<Tuple2<SourceName, Integer>>();

      for (SourceName source : matrix.getSources())
      {
         int maxSlices = 0;
         Map<Long, PlacedSourceSliceSet> txMap = matrix.getRevisions(source);

         for (PlacedSourceSliceSet txSlices : txMap.values())
         {
            maxSlices = Math.max(txSlices.size(), maxSlices);
         }

         if (maxSlices < this.getThreshold())
         {
            incompleteSources.add(new Tuple2<SourceName, Integer>(source, this.getThreshold()
                  - maxSlices));
         }
      }

      return incompleteSources;
   }

   /**
    * Helper function to rank a list of datasources revisions. This implementation chooses the
    * highest-numbered revision that satisfies the assembly threshold
    * 
    * @param transactionMap
    * @return
    */
   private List<Tuple2<PlacedSourceSliceSet, Long>> rankRevisions(
         Map<Long, PlacedSourceSliceSet> transactionMap)
   {
      List<Long> orderedTransactionIds = new LinkedList<Long>();

      // Find the Ids of all viable transactions
      for (Map.Entry<Long, PlacedSourceSliceSet> entry : transactionMap.entrySet())
      {
         // Any winning transaction must meet the restoration criteria
         if (entry.getValue().size() >= this.getThreshold())
         {
            orderedTransactionIds.add(entry.getKey());
         }
      }

      // Return an ordered list of transaction sets
      Collections.sort(orderedTransactionIds, Collections.reverseOrder());
      List<Tuple2<PlacedSourceSliceSet, Long>> transactions =
            new LinkedList<Tuple2<PlacedSourceSliceSet, Long>>();
      for (Long txid : orderedTransactionIds)
      {
         transactions.add(new Tuple2<PlacedSourceSliceSet, Long>(transactionMap.get(txid), txid));
      }

      return transactions;
   }

   /**
    * Helper for encoding a byte[] using a codec stack
    * 
    * @param codecs
    * @param sourceName
    * @param transactionId
    * @param data
    * @return
    * @throws DataTransformationException
    */
   protected byte[] encodeDataImpl(
         List<Codec> codecs,
         SourceName sourceName,
         long transactionId,
         byte[] data) throws DataTransformationException
   {
      for (Codec codec : codecs)
      {
         try
         {
            Encoder encoder = codec.getEncoder();

            if (_logger.isTraceEnabled())
               _logger.trace("Encoding data using codec " + codec.getName());
            encoder.reset(sourceName, transactionId);
            data = encoder.finish(data);
            if (_logger.isTraceEnabled())
               _logger.trace("Encoded data using codec " + codec.getName());
         }
         catch (Exception e)
         {
            throw new DataTransformationException("Could not encode data using codec '"
                  + codec.getName() + "' of type '" + codec.getType() + "'", e);
         }
      }
      return data;
   }

   /**
    * Helper for decoding a byte[] using a codec stack
    * 
    * @param codecs
    * @param sourceName
    * @param transactionId
    * @param data
    * @return
    * @throws DataTransformationException
    */
   protected byte[] decodeDataImpl(
         List<Codec> codecs,
         SourceName sourceName,
         long transactionId,
         byte[] data) throws DataTransformationException
   {
      for (Codec codec : codecs)
      {
         try
         {
            Decoder decoder = codec.getDecoder();

            if (_logger.isTraceEnabled())
               _logger.trace("Decoding sllice using codec " + codec.getName());
            decoder.reset(sourceName, transactionId);
            data = decoder.finish(data);

            if (_logger.isTraceEnabled())
               _logger.trace("Decoded slice using codec " + codec.getName());
         }
         catch (Exception e)
         {
            throw new DataTransformationException("Could not decode slice using codec '"
                  + codec.getName() + "' of type '" + codec.getType() + "'", e);
         }
      }
      return data;
   }

   public String toString()
   {
      StringBuffer buf = new StringBuffer();
      buf.append("type:").append(this.getType()).append("[").append(this.getWidth()).append("/").append(
            this.getThreshold()).append("]");
      buf.append(" IDA:").append(this.getInformationDispersalCodec().getName());
      buf.append(" Codecs [");
      for (Codec codec : this.getDatasourceCodecs())
      {
         buf.append(codec.getName()).append(" ");
      }
      buf.append("]");

      buf.append(" Slice Codecs [");
      for (Codec codec : this.getSliceCodecs())
      {
         buf.append(codec.getName()).append(" ");
      }
      buf.append("]");

      buf.append(" Slice Stores[");
      int i = 0;
      for (SliceStore store : this.getSliceStores())
      {
         buf.append(++i).append(". ").append(store);
      }
      return buf.toString();
   }
}
