//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: May 30, 2007
//---------------------

package org.cleversafe.vault;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.security.Key;
import java.security.PrivateKey;
import java.util.Map;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.util.ASN1Dump;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultKeyException;
import org.cleversafe.vault.exceptions.VaultKeyLookupException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.cleversafe.vault.storage.ASN1VaultACL;
import org.cleversafe.vault.storage.asn1.EncryptedKeyInfo;
import org.cleversafe.vault.storage.asn1.PlainKeyInfo;
import org.cleversafe.vault.storage.asn1.VaultACLEntry;
import org.cleversafe.vault.storage.asn1.VaultPermissionEntry;


/**
 * Stores a vault access control list in a binary file.
 * <p>
 * ASN.1 is used for the actual encoding.
 *
 */
public class FileVaultACL extends VaultACL
{
   private static Logger _logger = Logger.getLogger(FileVaultACL.class);

   private static final String SCHEME = "file";
   
   private static String VAULT_SUBDIR = VaultACL.VAULT_SUBDIR_DEFAULT;
   
   public static final String FILE_EXTENSION = ".acl";
   
   private ASN1VaultACL acl;
   private File file;
   
   static
   {
      VAULT_SUBDIR = System.getProperty( 
            VaultACL.VAULT_SUBDIR_PROPERTY, VaultACL.VAULT_SUBDIR_DEFAULT );
   }
   
   
   private static File getFilePath( URI vaultLocation, UUID vaultIdentifier )
   {
      return new File(
            vaultLocation.getPath() + File.separator + FileVaultACL.VAULT_SUBDIR +
            File.separator + vaultIdentifier.toString() + FileVaultACL.FILE_EXTENSION );
   }
   
   private void decode( InputStream input ) throws IOException
   {
      // Note: Any IllegalArgumentException or ClassCastException thrown by the ASN.1
      // system needs to be caught and wrapped in an IOException.
      
      ASN1InputStream in = new ASN1InputStream( input );
      
      DERObject obj = in.readObject();
      
      try
      {
         this.acl = ASN1VaultACL.getInstance(obj);
      }
      catch (IllegalArgumentException e)
      {
         _logger.debug( e );
         if ( obj != null )
         {
            _logger.debug( ASN1Dump.dumpAsString(obj) );
         }
         throw new IOException("Error reading Vault ACL: " + e.getMessage());
      }
      catch (ClassCastException e)
      {
         _logger.debug( e );
         if ( obj != null )
         {
            _logger.debug( ASN1Dump.dumpAsString(obj) );
         }
         throw new IOException("Error reading Vault ACL: " + e.getMessage());
      }
   }
   
   
   // Constructor for use with VaultACLFactory
   // This used to be protected, but had to be made public to allow for configuration framework compatibility.
   public FileVaultACL()
   {
      this.file = null;
      this.acl = new ASN1VaultACL();
   }
   
   
   /**
    * Reads in a vault access control list from an input stream. The ACL cannot be written
    * back out unless {@link #flush(OutputStream)} is called.
    * 
    * @throws IOException If any storage errors occur.
    */
   public FileVaultACL( InputStream input ) throws IOException
   {
       this.file = null;
       this.decode(input);
   }
   
   /**
    * Reads in a vault access control list from a file. The ACL can be written back out
    * using {@link #flush()}.
    * 
    * @throws IOException If any storage errors occur.
    * @throws FileNotFoundException The indicated file was not found.
    */
   public FileVaultACL( File file ) throws IOException, FileNotFoundException
   {
      this.file = file;
      FileInputStream in = new FileInputStream( this.file );
      this.decode(in);
      in.close();
   }
   
   public byte[] getEncoded()
   {
      try
      {
         return this.acl.getEncoded();
      }
      catch (IOException ex)
      {
         throw new RuntimeException("Failed to get encoded bytes");
      }
   }
   
   @Override
   protected void loadFromURI( URI vaultLocation, UUID vaultIdentifier ) 
         throws VaultIOException
   {
      if ( ! vaultLocation.getScheme().equals(FileVaultACL.SCHEME) )
      {
         throw new VaultIOException("This vault implementation expects '" + 
               FileVaultACL.SCHEME + "' vaults, URI indicates '" +
               vaultLocation.getScheme() + "'." );
      }
      
      this.file = FileVaultACL.getFilePath(vaultLocation, vaultIdentifier);
      
      try
      {
         this.decode( new FileInputStream(this.file) );
      }
      catch (IOException e)
      {
         throw new VaultIOException("Error reading vault from file: " + this.file, e);
      }
   }
   
   
   @Override
   public void loadFromStream(InputStream in) throws VaultIOException
   {
      this.file = null;
      try
      {
         this.decode(in);     
      }
      catch(IOException ex)
      {
         throw new VaultIOException("Error reading vault input stream", ex);
      }
   }

   @Override
   public void flush() throws VaultIOException
   {
      if ( this.file == null )
      {
         throw new VaultIOException("not initialized with file object, no output stream");
      }
      try
      {
         FileOutputStream out = new FileOutputStream( this.file );
         this.flush( out );
         out.close();
      }
      catch (IOException e)
      {
         throw new VaultIOException("error writing vault to file", e);
      }
   }
   
   @Override
   public void flush( OutputStream output ) throws VaultIOException
   {
      try
      {
         output.write( this.acl.getDEREncoded() );
      }
      catch (IOException e)
      {
         throw new VaultIOException("error writing vault to output stream", e);
      }
   }
   
   @Override
   public void flush( URI vaultLocation ) throws VaultIOException
   {
      try
      {
         this.file = FileVaultACL.getFilePath(vaultLocation, this.acl.getVaultIdentifier());
         OutputStream out = new FileOutputStream( this.file );
         out.write( this.acl.getDEREncoded() );
      }
      catch (IOException e)
      {
         throw new VaultIOException("error writing vault to file: " + this.file.toString(), e);
      }
   }
   
   


   @Override
   protected void sign(UUID adminAccount, PrivateKey adminPrivateEncryptionKey) 
         throws VaultSecurityException
   {
      // vault acl signing not implemented
      return;
   }



   @Override
   public boolean verify()
   {
//    vault acl signing not implemented
      return true;
   }



   @Override
   protected VaultACLEntry getACLEntry(UUID account)
   {
      return acl.getACLEntry(account);
   }





   @Override
   public UUID getOwner()
   {
      return acl.getOwner();
   }



   @Override
   public UUID getVaultIdentifier()
   {
      return acl.getVaultIdentifier();
   }



   @Override
   protected void storeACLEntry(UUID account, VaultPermissionEntry permission, Map<Integer, EncryptedKeyInfo> vaultKeys)
   {
      acl.storeACLEntry(account, permission, vaultKeys);
   }



   @Override
   protected void storeOwner(UUID account)
   {
      acl.storeOwner(account);     
   }



   @Override
   protected void storeVaultIdentifier(UUID vault)
   {
      acl.storeVaultIdentifier(vault);
   }


   @Override
   protected void storePublicKeys(Map<Integer, PlainKeyInfo> publicKeys)
   {
      acl.storePublicKeys(publicKeys);
   }


   @Override
   protected Key getKey(UUID account, int index, PrivateKey privateKey) throws VaultACLException,
         VaultKeyException, VaultKeyLookupException, VaultSecurityException
   {
      return this.acl.getKey(account, index, privateKey);
   }


   @Override
   protected Key getPublicKey(UUID account, int index, PrivateKey privateKey)
         throws VaultACLException, VaultKeyException, VaultKeyLookupException,
         VaultSecurityException
   {
      return this.acl.getPublicKey(account, index, privateKey);
   }



   
   
   

   
   
   
   
}


