//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: Jul 6, 2007
//---------------------

package org.cleversafe.vault;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.util.List;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.config.ExecutionContext;
import org.cleversafe.layer.communication.exceptions.CommunicationException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.cleversafe.vault.storage.VaultKeyInfo;
import org.cleversafe.vault.storage.VaultKeyInfoGenerator;

/**
 * A full account creator object for testing purposes. It creates a vault ACL object, loads a vault
 * from a vault descriptor, creates the stores described in the vault descriptor and begins their
 * sessions.
 * 
 * @author Manish Motwani
 */
public class FullAccountCreator
{
   private static Logger _logger = Logger.getLogger(FullAccountCreator.class);
   private static int number_accounts_created = 0;

   private UUID vaultIdentifier;
   private Vault vault;
   private VaultACL acl;
   private Credentials credentials;
   private KeyPair keyPair;

   /**
    * 
    * @param vaultDescriptorFileName
    * @throws Exception
    */
   public FullAccountCreator(String vaultDescriptorFileName) throws Exception
   {
      try
      {
         createAccountAndVault(vaultDescriptorFileName);

         _logger.debug("Creating Vault for account: "
               + this.credentials.getAccountIdentifier().toString());

         // Load the vault
         this.vault =
               new XMLVaultLoader().loadVaultFromXML(vaultDescriptorFileName,
                     createExecutionContext());
         this.vault.setVaultIdentifier(this.vaultIdentifier);

         // Note: the following code is going to be removed once vault descriptor is no
         // longer stored on the slice servers.
         FileInputStream vaultDescriptorInputStream = new FileInputStream(vaultDescriptorFileName);
         ByteBuffer vaultDescriptorByteBuffer =
               ByteBuffer.allocate(vaultDescriptorInputStream.available());
         while (vaultDescriptorInputStream.available() > 0)
         {
            byte[] bytes = new byte[vaultDescriptorInputStream.available()];
            vaultDescriptorInputStream.read(bytes);
            vaultDescriptorByteBuffer.put(bytes);
         }
         vaultDescriptorInputStream.close();

         // Create slice stores
         for (int i = 0; i < this.vault.getSliceStores().size(); ++i)
         {
            SliceStore store = this.vault.getSliceStores().get(i);
            store.createStore(this.vault.getType(), this.vault.getMaxSliceSize(),
                  this.vault.getSliceStoreSize(i), getACL(), vaultDescriptorByteBuffer.array());
         }

         _logger.debug("Vault created for account: "
               + this.credentials.getAccountIdentifier().toString());
      }
      catch (Exception e)
      {
         --number_accounts_created;
         throw e;
      }

   }

   private void createAccountAndVault(String vaultDescriptorFileName) throws Exception
   {
      this.vaultIdentifier = UUID.randomUUID();

      // Creating the the credentials
      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("account_creator_" + (++number_accounts_created));
      credentials.setPassword("password");
      this.credentials = credentials;

      _logger.debug("Creating Vault ACL for account: "
            + credentials.getAccountIdentifier().toString());

      KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
      keygen.initialize(2048);
      this.keyPair = keygen.generateKeyPair();

      List<VaultKeyInfo> keyInfoList = createVaultKeyList(vaultDescriptorFileName);

      // Creating the Vault ACL
      VaultACLFactory fact = new VaultACLFactory();
      ByteArrayOutputStream out = new ByteArrayOutputStream();
      fact.create(out, this.vaultIdentifier, this.credentials, this.keyPair.getPublic(),
            keyInfoList);
      ByteArrayInputStream in = new ByteArrayInputStream(out.toByteArray());
      this.acl = fact.getInstance(in);

      // Clear memory
      out.reset();
      in.reset();
   }

   private List<VaultKeyInfo> createVaultKeyList(String vaultDescriptorFileName) throws Exception
   {
      FileInputStream din;
      try
      {
         din = new FileInputStream(new File(vaultDescriptorFileName));
      }
      catch (FileNotFoundException e)
      {
         throw new IOException("Could not find XML vault descriptor file");
      }

      List<VaultKeyInfo> keyInfoList;
      try
      {
         keyInfoList = VaultKeyInfoGenerator.generate(din);
      }
      catch (XmlException e)
      {
         throw new IOException("Could not parse XML vault descriptor");
      }
      catch (VaultDescriptorException e)
      {
         throw new IOException("Encountered a Vault Desciptor exception");
      }
      catch (IOException e)
      {
         throw new IOException("IO exception parsing XML vault descriptor");
      }
      din.close();

      return keyInfoList;
   }

   private ExecutionContext createExecutionContext() throws VaultSecurityException
   {
      // Create the execution context for vault loading
      ExecutionContext ctx = new ExecutionContext();
      ctx.add(VaultDescriptor.ACL_CTX_STRING, this.acl);
      ctx.add(VaultDescriptor.ACL_ENTRY_CTX_STRING, this.acl.getEntry(this.acl.getOwner(),
            this.keyPair.getPrivate()));
      ctx.add(VaultDescriptor.CREDENTIALS_CTX_STRING, this.credentials);
      return ctx;
   }

   /**
    * 
    * @return
    */
   public List<SliceStore> getSliceStores()
   {
      return this.vault.getSliceStores();
   }

   /**
    * 
    * @return
    */
   public UUID getOwner()
   {
      return this.acl.getOwner();
   }

   /**
    * 
    * @return
    */
   public UUID getVaultIdentifier()
   {
      return this.acl.getVaultIdentifier();
   }

   /**
    * 
    * @return
    */
   public Vault getVault()
   {
      return this.vault;
   }

   /**
    * 
    * @return
    */
   public VaultACL getACL()
   {
      return this.acl;
   }

   /**
    * 
    * @return
    */
   public Credentials getCredentials()
   {
      return this.credentials;
   }

   /**
    * 
    * @throws CommunicationException
    */
   public void cleanup() throws SliceStoreLayerException
   {
      for (SliceStore ss : this.vault.getSliceStores())
      {
         try
         {
            ss.deleteStore();
         }
         catch (Exception ex)
         {
            _logger.error("Error deleting store:" + ss);
         }
      }
   }

}
