//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: May 25, 2007
//---------------------

package org.cleversafe.vault;

import java.util.List;
import java.util.UUID;

import org.cleversafe.codec.Codec;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.OperationScorecard;
import org.cleversafe.layer.grid.PlacedDataSlice;
import org.cleversafe.layer.grid.PlacedSliceName;
import org.cleversafe.layer.grid.PlacedSourceSliceSet;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.SourceSliceSet;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.exceptions.DataTransformationException;
import org.cleversafe.vault.exceptions.InformationDispersalException;

/**
 * Provides codecs and slice stores used to store data in an opened vault.
 * <p>
 * After a vault is loaded this interface is presented to the slice store controller. Vaults are
 * opened using the {@link VaultFactory}.
 * 
 * @see VaultFactory
 * 
 */
public interface Vault
{

   /**
    * @return The vault identifier.
    */
   public UUID getVaultIdentifier();

   /**
    * Sets the vault identifier.
    */
   public void setVaultIdentifier(UUID vaultIdentifier);

   /**
    * The vault is split between a number of slice stores. Data stored in the vault must be sliced
    * accordingly. This number is fixed per vault.
    * 
    * @return Width in slice stores for the vault.
    * @see #getSliceStores()
    */
   public int getWidth();

   /**
    * The total size of the vault in bytes.
    * 
    * @return
    */
   public long getSize();

   /**
    * The vault's IDA configuration allows up to a given number of slices to be lost before recovery
    * of a given piece of data is not possible.
    * 
    * This method returns number of slices that are REQUIRED for data recovery.
    * 
    * @return The number of slices that are REQUIRED for data recovery determined by the IDA.
    */
   public int getThreshold();

   /**
    * Data in a vault is stored in a number of slices equal to the width of the vault.
    * 
    * @return An array of initialized slice stores.
    * @see #getWidth()
    */
   public List<SliceStore> getSliceStores();

   /**
    * Returns the size of the slice store identified by the input <code>sliceStoreIndex</code>
    * 
    * @param sliceStoreIndex
    * @return
    */
   public long getSliceStoreSize(int sliceStoreIndex);

   /**
    * Data stored in a vault is passed through a number of codecs before being sent to the IDA.
    * <p>
    * When encoding the codecs must be applied in first to last order. When decoding the codecs must
    * be applied in last to first order.
    * 
    * @return A list of codecs in the order they should be used.
    */
   public List<Codec> getDatasourceCodecs();

   /**
    * 
    * @return A list of codecs applied to the slices after dispersal in the order they should be
    *         used.
    */
   public List<Codec> getSliceCodecs();

   /**
    * Data is sent through the IDA after the all codecs are applied. The slices output from the IDA
    * are stored on the vault's slice stores.
    * 
    * @return An Information Dispersal Algorithm (IDA) codec.
    */
   public InformationDispersalCodec getInformationDispersalCodec();

   /**
    * This function is used to calculate the maximum size a slice will have after all data source
    * codecs, informational dispersal and slice codecs have been applied.
    * 
    * @param inputDataSize
    * @return
    */
   public long getMaxSliceSize();

   /**
    * Type of vault
    * 
    * @return
    */
   public String getType();
   
   /**
    * Read and write thresholds
    * 
    * To guarantee a single consistent revision, the following conditions must hold:
    *  1. Read threshold >= IDA threshold
    *  2. Write threshold >= IDA threshold
    *  3. Read threshold + Write threshold > IDA width
    *  
    *  It is legal for advanced users to break condition 3 for specialized vaults, but 
    *  not advised unless the vault owner fully understands the consequences.
    */
   public void setWriteThreshold(int writeThreshold);
   public int getWriteThreshold();

   public void setReadThreshold(int readThreshold);
   public int getReadThreshold();   

   /**
    * Attempts to starts the session for each {@link SliceStore} the vault is stored on. Does not
    * guarantee success or report failure.
    */
   public void startSessions();

   /**
    * Attempts to ends all the sessions for each {@link SliceStore} the vault is stored on. Does not
    * guarantee success or report failure.
    */
   public void endSessions();

   /**
    * Maps a slice name to the correct SliceStore on the grid
    * 
    * @param name
    * @return
    */
   public PlacedSliceName mapSliceNameToStore(SliceName name);

   /**
    * Maps a datasource to some number of slices and their grid locations
    * 
    * @return
    */
   public List<PlacedSliceName> mapSourceNameToSliceNames(SourceName name);

   /**
    * Maps a slice of data to the correct SliceStore on the grid
    * 
    * @param slice
    * @return
    */
   public PlacedDataSlice mapSliceToStore(DataSlice slice);

   /**
    * Maps a set of slices for a given datasource to SliceStores on the grid
    * 
    * @param slices
    * @return
    */
   public PlacedSourceSliceSet mapSliceSetToStore(SourceSliceSet slices);

   /**
    * Apply encoding transformation stack based on codecs defined in Vault, returning a new
    * DataSource
    * 
    * @param source
    * @param transactionId
    * @return
    * @throws DataTransformationException
    */
   public DataSource encodeSource(DataSource source) throws DataTransformationException;

   /**
    * Apply decoding transformation stack based on codecs defined in Vault, returning a new
    * DataSource
    * 
    * @param source
    * @param transactionId
    * @return
    * @throws DataTransformationException
    */
   public DataSource decodeSource(DataSource source) throws DataTransformationException;

   /**
    * Apply encoding transformation stack based on codecs defined in Vault to a single slice,
    * returning a new DataSlice
    * 
    * @param slice
    * @return
    * @throws DataTransformationException
    */
   public DataSlice encodeSlice(DataSlice slice) throws DataTransformationException;

   /**
    * Apply decoding transformation stack based on codecs defined in Vault fro a single slice,
    * returning a new DataSlice
    * 
    * @param slice
    * @return
    * @throws DataTransformationException
    */
   public DataSlice decodeSlice(DataSlice slice) throws DataTransformationException;

   /**
    * Dispersal based on Vault definition
    * 
    * @param source
    * @return
    * @throws InformationDispersalException
    */
   public SourceSliceSet disperseSource(DataSource source) throws InformationDispersalException;

   /**
    * Assemble source from slices. At least the threshold number of slices should be present. It is
    * assumed that all slices in the provided set have the same transaction ID
    * 
    * @param slices
    * @return
    * @throws InformationDispersalException
    */
   public DataSource assembleSource(SourceSliceSet slices) throws InformationDispersalException;

   /**
    * Apply pre-dispersal coding stack, dispersal, and post-dispersal coding stack to the given
    * datasource
    * 
    * @param data
    * @param transactionId
    * @return
    * @throws InformationDispersalException
    * @throws DataTransformationException
    */
   public SourceSliceSet applyDispersalStack(DataSource data) throws InformationDispersalException,
         DataTransformationException;

   /**
    * Reassemble datasources from slices by applying the post-dispersal decoding stack,
    * de-dispersal, and pre-dispersal decoding stack. It is assumed that all slices in the set are
    * of the same revision.
    * 
    * @param slices
    * @return
    * @throws InformationDispersalException
    * @throws DataTransformationException
    */
   public DataSource applyAssemblyStack(SourceSliceSet slices)
         throws InformationDispersalException, DataTransformationException;

   /**
    * Reassembles datasources from slices, optionally applying post-dispersal slice codecs, allowing
    * slice pre-verification. It is assumed that all slices in the set are of the same revision.
    * 
    * @see applyAssemblyStack(SourceSliceSet slices)
    * 
    * @param slices
    * @param applySliceCodecs
    * @return
    * @throws InformationDispersalException
    * @throws DataTransformationException
    */
   public DataSource applyAssemblyStack(SourceSliceSet slices, boolean applySliceCodecs)
         throws InformationDispersalException, DataTransformationException;

   /**
    * Attempts to rebuild all sources from a slice matrix. If any source cannot be rebuilt, an
    * exception is thrown.
    * 
    * @param matrix
    * @return
    */
   public List<DataSource> rebuildSources(OperationScorecard matrix)
         throws InformationDispersalException, DataTransformationException;

   /**
    * Attempts to rebuild all sources from a slice matrix. If any source cannot be rebuilt, an
    * exception is thrown. Optionally applies slice codecs, allowing slice pre-verification.
    * 
    * @see rebuildSources(DataTransformationMatrix matrix)
    * 
    * @param matrix
    * @param applySliceCodecs
    * @return
    */
   public List<DataSource> rebuildSources(OperationScorecard matrix, boolean applySliceCodecs)
         throws InformationDispersalException, DataTransformationException;

   /**
    * Analyzes a slice matrix and determines which sources do not have the required number of slices
    * to be rebuilt for at least one transaction revision. Note that this does not check slices for
    * corruption or make any guarantees about which revision can be rebuilt
    * 
    * @param matrix
    * @return
    */
   @Deprecated
   public List<Tuple2<SourceName, Integer>> getIncompleteSources(OperationScorecard matrix);
   
   /**
    * Analyzes properties of the vault such as the codecs and IDA used to optionally optimize 
    * properties of the vault and other codecs such as the IDA chunk size for block device vaults.
    */
   public void optimizeVault();
}
