//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: May 25, 2007
//---------------------

package org.cleversafe.vault;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.security.Key;
import java.security.KeyException;
import java.security.PrivateKey;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Map;
import java.util.UUID;

import org.cleversafe.exceptions.NotImplementedException;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultKeyException;
import org.cleversafe.vault.exceptions.VaultKeyLookupException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.cleversafe.vault.storage.asn1.EncryptedKeyInfo;
import org.cleversafe.vault.storage.asn1.PlainKeyInfo;
import org.cleversafe.vault.storage.asn1.VaultACLEntry;
import org.cleversafe.vault.storage.asn1.VaultPermissionEntry;

/**
 * Vault access control list. Provides JAAS Permission objects and controls access to vault secret
 * (symmetric) keys.
 * <p>
 * This abstract class implements the ACL security operations. Concrete implementations provide
 * actual ACL storage and retrieval.
 * <p>
 * This is the main interface provided to slice servers to check client permissions. Management
 * applications should also use this interface to modify the ACL.
 * <p>
 * This is a secondary interface provided to clients to retrieve vault secret keys. The
 * {@link Vault} object generally abstracts usage of the ACL.
 */
public abstract class VaultACL
{

   public static final String VAULT_SUBDIR_PROPERTY = "org.cleversafe.vault.dir";
   public static final String VAULT_SUBDIR_DEFAULT = "vaults";

   // TODO: Add logging for security and other events.

   /**
    * Creates an empty VaultACL object. This constructor is used by the VaultACLFactory, which then
    * generates appropriate data and calls the store methods to populate the ACL.
    */
   protected VaultACL()
   {
   }

   /**
    * Get an encoded representation of this VaultACL.
    * 
    * @return Encoded representation of the VaultACL.
    */
   public abstract byte[] getEncoded();

   // Abstract storage methods.

   /**
    * Stores a new ACL entry or overwrites an existing one.
    */
   protected abstract void storeACLEntry(
         UUID account,
         VaultPermissionEntry permission,
         Map<Integer, EncryptedKeyInfo> vaultKeys);

   /**
    * Stores a set of vault public keys.
    * 
    * @param publicKeys
    */
   protected abstract void storePublicKeys(Map<Integer, PlainKeyInfo> publicKeys);

   /**
    * Changes the vault owner.
    */
   protected abstract void storeOwner(UUID account);

   /**
    * Changes or sets the vault identifier. This method must only be called by the
    * {@link VaultACLFactory} when the VaultACL is initially created.
    */
   protected abstract void storeVaultIdentifier(UUID vault);

   // Abstract protected getters.

   /**
    * Returns the ACL entry assigned to the given grid account. Returns <code>null</code> if the
    * given account does not have an entry in this vault.
    */
   protected abstract VaultACLEntry getACLEntry(UUID account);

   /**
    * Returns a vault key which the indicated grid account has access to. Specific keys are
    * referenced by an integer index. All keys returned by this method are either symmetric secret
    * keys or asymmetric private keys. For asymmetric entries call
    * {@link #getPublicKey(UUID, int, PrivateKey)} for the matching public key.
    * 
    * @param account
    *           The account whose entry the key will be read from.
    * @param index
    *           The key storage index.
    * @param privateKey
    *           The account private key.
    * @return The secret or private key.
    * @throws VaultACLException
    *            If the ACL is malformed. This error is unrecoverable.
    * @throws VaultKeyException
    *            If the account private key is not valid.
    * @throws VaultKeyLookupException
    *            If no key is stored under the given storage index.
    * @throws VaultSecurityException
    *            If the account does not have access to the key stored in the given storage index.
    */
   protected abstract Key getKey(UUID account, int index, PrivateKey privateKey)
         throws VaultACLException, VaultKeyException, VaultKeyLookupException,
         VaultSecurityException;

   /**
    * Returns a vault public key. Specific keys are referenced by an integer index. This method only
    * returns public keys for those indices where asymmetric keys are stored. Call
    * {@link #getKey(UUID, int, PrivateKey)} for the matching private key.
    * 
    * @param account
    *           The account whose entry the key will be read from.
    * @param index
    *           The key storage index.
    * @param privateKey
    *           The account private key.
    * @return The public key.
    * @throws VaultACLException
    *            If the ACL is malformed. This error is unrecoverable.
    * @throws VaultKeyException
    *            If the account private key is not valid.
    * @throws ValueKeyLookupException
    *            If no key is stored under the given storage index.
    * @throws VaultSecurityException
    *            If the account does not have access to the key stored in the given storage index.
    */
   protected abstract Key getPublicKey(UUID account, int index, PrivateKey privateKey)
         throws VaultACLException, VaultKeyException, VaultKeyLookupException,
         VaultSecurityException;

   // Abstract public getters.

   /**
    * The vault this ACL controls access to.
    */
   public abstract UUID getVaultIdentifier();

   /**
    * Returns the identifier of the grid account which currently owns the vault.
    */
   public abstract UUID getOwner();

   // Abstract public ACL management methods.

   /**
    * Loads an ACL from a given vault location. Loading always happens in a type specific manner.
    * 
    * @param vaultLocation
    *           The URI where Vault ACLs are stored. Indicates the Vault ACL loading protocol and
    *           the location where any Vault ACL can be loaded from. The specific location of a
    *           given ACL is dependent on the vault storage type.
    * @param vaultIdentifier
    *           The identifier of the Vault whose ACL is to be loaded.
    * @throws VaultIOException
    *            If the Vault ACL cannot be found or a loading error occurs.
    */
   protected abstract void loadFromURI(URI vaultLocation, UUID vaultIdentifier)
         throws VaultIOException;

   /**
    * Loads an ACL from an input stream.
    * 
    * @throws VaultIOException
    *            if ACL cannot be read.
    */
   protected abstract void loadFromStream(InputStream in) throws VaultIOException;

   /**
    * Flushes any changes to the access control list to storage. The exact nature of an ACL flush is
    * dependent on the storage implementation.
    * 
    * @throws IOException
    *            If any storage errors occur.
    */
   public abstract void flush() throws VaultIOException;

   /**
    * Flushes any changes to the access control list to storage. The exact nature of an ACL flush is
    * dependent on the storage implementation.
    * 
    * @param out
    *           OutputStream to which ACL changes are written.
    * @throws IOException
    *            If any storage errors occur.
    */
   public abstract void flush(OutputStream out) throws VaultIOException;

   /**
    * Flushes any changes to the access control list to storage. The exact nature of an ACL flush is
    * dependent on the storage implementation.
    * 
    * @param vaultLocation
    *           The URI where vaults are stored.
    * @throws VaultIOException
    *            If any storage errors occur.
    */
   public abstract void flush(URI vaultLocation) throws VaultIOException;

   /**
    * Verifies the authenticity of the access control list.
    * 
    * Unless another means of verifying authenticity is available, storage interfaces which do not
    * implement signatures should always return true.
    * 
    * @return True if the access control list is authentic; False otherwise.
    */
   public abstract boolean verify();

   /**
    * Provides the ACL storage with enough information to sign the ACL. Storage implementations
    * which do not sign the ACL should provide an empty implementation.
    * 
    * Assuming the permission objects provided by the storage implementation are valid, the given
    * "administrative" account is guaranteed to have modifyACL permission.
    * 
    * A SecurityException must be thrown instead of a SignatureException to avoid the calling code
    * having to be responsible for signature exceptions. Some ACL storage implementations may not
    * implement signatures.
    * 
    * @param adminAccount
    *           An account with modifyACL permission.
    * @param adminPrivateEncryptionKey
    *           The private encryption key belonging to the admin account.
    * @throws SecurityException
    *            If the private encryption key was not usable or a signature error occured.
    */
   protected abstract void sign(UUID adminAccount, PrivateKey adminPrivateEncryptionKey)
         throws VaultSecurityException;

   // Base class methods

   /**
    * Returns the permission assigned to the given account identifier.
    * 
    * @param account
    *           The account to retrieve permissions for.
    * @return Null if this account does not have access to the vault; otherwise the permission
    *         object for the vault applied to this account.
    */
   public VaultPermission getPermission(UUID account) throws VaultSecurityException
   {
      VaultACLEntry entry = getACLEntry(account);
      if (entry == null)
      {
         throw new SecurityException("account " + account + " does not have access to vault "
               + getVaultIdentifier());
      }
      else
      {
         return new VaultPermission(getVaultIdentifier(), entry.getPermission());
      }
   }

   /**
    * Sets new permissions for the target grid account. The target account may be the an existing
    * authorized account or a new account.
    * 
    * The private encryption key of an existing account with modifyACL permission must be used to
    * set the new permission. The existing "administrative" account may be the same as the target
    * account.
    * 
    * A grid account may be removed from the access control list by setting a permission object with
    * no actions.
    * 
    * @param adminAccount
    *           An account with existing modifyACL permission.
    * @param adminPrivateEncryptionKey
    *           The private encryption key belonging to the admin account.
    * @param targetAccount
    *           An encryption certificate of the account whose permissions are being set.
    * @param permission
    *           The desired permission for the target account.
    * @throws CertificateException
    *            If the grid account identifier is not properly specified in a uid field or the
    *            certificate is otherwise malformed.
    * @throws SecurityException
    *            If the indicated admin account does not have modifyACL permissions or the given key
    *            did not successfully decrypt the symmetric vault key.
    * @throws KeyException
    *            If the symmetric vault key could not be decrypted because of improper configuration
    *            or unsupported algorithms.
    */
   public void setPermission(
         UUID adminAccount,
         PrivateKey adminPrivateEncryptionKey,
         X509Certificate targetAccount,
         VaultPermission permission) throws VaultSecurityException, VaultKeyException
   {
      throw new NotImplementedException("Not yet implemented");
   }

   /**
    * Sets a new vault owner and associated permissions. It is possible to set a vault owner without
    * access or modification permissions.
    * 
    * {@link #setPermission(UUID, PrivateKey, X509Certificate, VaultPermission)}
    */
   public void setOwner(
         UUID adminAccount,
         PrivateKey adminPrivateEncryptionKey,
         X509Certificate targetAccount,
         VaultPermission permission) throws VaultSecurityException, VaultKeyException
   {
      throw new NotImplementedException("Not yet implemented");
      //      UUID account = parseAccountIdentifier( targetAccount );
      //      checkACLPermission( account, getPermission(adminAccount) );
      //      storeOwner( account );
      //      setPermission( adminAccount, adminPrivateEncryptionKey, targetAccount, permission );
   }

   // Methods used to set and access the "current" ACL Entry.

   /**
    * Returns the entry belonging to the indicated grid account.
    * 
    * @throws SecurityException
    *            If the keys provided by the entry could not be decrypted using the given private
    *            key.
    */
   public Entry getEntry(UUID account, PrivateKey privateKey) throws VaultSecurityException
   {
      return new Entry(this, account, privateKey);
   }

   /**
    * Provides access to a specific ACL entry. Combines entry specific information with data common
    * to all entries.
    * <p>
    * This class should be used by the {@link Vault} object to access only information for a single
    * entry. It may be used only for reading an Entry. {@link VaultACL} methods must be used
    * directly to modify the ACL.
    */
   public static class Entry
   {
      private final VaultACL acl;
      private final UUID account;
      private final PrivateKey accountPrivateKey;

      protected Entry(VaultACL acl, UUID account, PrivateKey privateKey)
      {
         this.acl = acl;
         this.account = account;
         this.accountPrivateKey = privateKey;
      }

      public VaultPermission getPermission() throws VaultSecurityException
      {
         return this.acl.getPermission(this.account);
      }

      /**
       * Retrieves a vault key by index.
       * 
       * @param index
       *           The key storage index.
       * @throws VaultACLException
       *            If the ACL is malformed. This error is unrecoverable.
       * @throws VaultKeyException
       *            If the account private key is not valid.
       * @throws VaultKeyLookupException
       *            If no key is stored under the given storage index.
       * @throws VaultSecurityException
       *            If the account does not have access to the key stored in the given storage
       *            index.
       */
      public Key getKey(int index) throws VaultACLException, VaultKeyException,
            VaultKeyLookupException, VaultSecurityException
      {
         return this.acl.getKey(this.account, index, this.accountPrivateKey);
      }

      /**
       * Retrieves a vault key by index. The key returned by this method will be the public key of a
       * key pair.
       * 
       * @param index
       *           The key storage index.
       * @throws VaultACLException
       *            If the ACL is malformed. This error is unrecoverable.
       * @throws VaultKeyException
       *            If the account private key is not valid.
       * @throws VaultKeyLookupException
       *            If no key is stored under the given storage index.
       * @throws VaultSecurityException
       *            If the account does not have access to the key stored in the given storage
       *            index.
       */
      public Key getPublicKey(int index) throws VaultACLException, VaultKeyException,
            VaultKeyLookupException, VaultSecurityException
      {
         return this.acl.getPublicKey(this.account, index, this.accountPrivateKey);
      }

   }

}
