//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 4, 2007
//---------------------

package org.cleversafe.vault;

import java.net.URI;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;

import org.cleversafe.authentication.CredentialsManager;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;

/**
 * Interface for vault management implementations. Allows the creation, deletion, and management of
 * a user-visible vault object.
 * <p>
 * When loaded, a vault manager will need to be provided with the location of the "grid resources"
 * directory. Grid resources are centrally stored configuration elements (e.g., vault information
 * and account lists).
 * <p>
 * The current interface does not require grid account information. These interfaces will be revised
 * once the application-level access control and authorization system is more complete.
 */
public interface VaultManager extends Iterable<UUID>
{
   /**
    * Loads configuration information from the indicated resource location.
    */
   VaultManager load(URI directory) throws VaultIOException;
   
   /**
    * Closes any connections the vault manager has open.
    */
   void close() throws VaultIOException;

   /**
    * Performs a lookup or conversion from vault name to vault identifier. A vault identifier is a
    * UUID or GUID.
    * 
    * @param vaultName
    *           The human-readable name associated with the vault.
    * @return The vault identifier associated with the given vault name.
    * @throws VaultIOException
    *            If the name to identifier lookup fails.
    */
   UUID getVaultIdentifier(String vaultName) throws VaultIOException;

   /**
    * Performs a reverse lookup or conversion from vault identifier to vault name. A vault name is
    * an undefined text string, usually human readable.
    * 
    * @param vaultIdentifier
    *           The vault identifier associated with the vault.
    * @return The human-readable name associated with the given vault identifier.
    * @throws VaultException
    *            If the identifier to name reverse lookup fails.
    */
   String getVaultName(UUID vaultIdentifier) throws VaultIOException;

   /**
    * Returns the vault identifiers of all currently existing vaults.
    * 
    * @throws VaultIOException
    *            If the vault lookup operation fails.
    */
   List<UUID> getVaults() throws VaultIOException;

   /**
    * Returns an iterator over all vault identifiers.
    * <p>
    * The iteration will fail as soon as a vault lookup operation fails. No {@link VaultIOException}
    * will be thrown.
    */
   Iterator<UUID> iterator();

   /**
    * Loads the vault associated with a given vault identifier.
    * 
    * @param vaultIdentifier
    *           The vault identifier of the vault to load.
    * @param accountIdentifier
    *           The account with which to load the vault.
    * @param manager
    *           A credentials manager used to obtain credentials for the given vault.
    * @return A loaded vault.
    * @throws VaultException
    *            If the vault could not be loaded.
    * @throws VaultSecurityException
    *            If the vault could not be loaded in the specified mode because if insufficient
    *            vault permissions for the given grid account.
    */
   Vault loadVault(UUID vaultIdentifier, UUID accountIdentifier, CredentialsManager manager)
         throws VaultException, VaultSecurityException;

   /**
    * Loads the ACL associated with the given vault identifier.
    * 
    * @param vaultIdentifier
    *           The vault identifier of the ACL to load.
    * @return A loaded VaultACL.
    * @throws VaultException
    *            If the vault ACL could not be loaded.
    */
   VaultACL getVaultACL(UUID vaultIdentifier) throws VaultIOException;

   /**
    * Loads the vault descriptor associated with the given vault identifier.
    * 
    * @param vaultIdentifier
    *           The vault identifier of the descriptor to load.
    * @return The serialized vault descriptor.
    * @throws VaultIOException
    *            If the vault descriptor could not be loaded.
    */
   String getVaultDescriptor(UUID vaultIdentifier) throws VaultIOException;
   
   /**
    * Creates a new vault.
    * 
    * @param vaultName
    *           A human readable name to associate with the vault.
    * @param vaultDescriptor
    *           A vault descriptor.
    * @param accountIdentifier
    *           The account with which to create the vault.
    * @param manager
    *           A credentials manager used to obtain credentials for the given vault.
    * @return The vault identifier of the newly created vault.
    * @throws VaultException
    *            If the vault could not be created.
    * @throws VaultSecurityException
    *            If the vault could not be created because of insufficient network permissions for
    *            the given grid account.
    */
   UUID createVault(
         String vaultName,
         String vaultDescriptor,
         UUID accountIdentifier,
         CredentialsManager manager) throws VaultException, VaultSecurityException;

   /**
    * Changes the human-readable vault name for the given vault.
    * 
    * @param vaultIdentifier
    *           The vault identifier of the vault to rename.
    * @param newVaultName
    *           The new vault name.
    * @param accountIdentifier
    *           An account authorized to modify the vault.
    * @param manager
    *           A credentials manager used to obtain credentials for the given vault.
    * @throws VaultException
    *            If the vault could not be renamed.
    * @throws VaultSecurityException
    *            If the vault could not be renamed because of insufficient vault permissions.
    */
   void renameVault(
         UUID vaultIdentifier,
         String newVaultName,
         UUID accountIdentifier,
         CredentialsManager manager) throws VaultException, VaultSecurityException;

   /**
    * Deletes a vault from the network. Deletes all vault data from the grid.
    * 
    * @param vaultIdentifier
    *           The vault identifier of the vault to delete.
    * @param accountIdentifier
    *           An account authorized to modify the vault.
    * @param manager
    *           A credentials manager used to obtain credentials for the given vault.
    * @throws VaultException
    *            If the vault could not be deleted.
    * @throws VaultSecurityException
    *            If the vault could not be deleted because of insufficient vault or network
    *            permissions.
    */
   void deleteVault(UUID vaultIdentifier, UUID accountIdentifier, CredentialsManager manager)
         throws VaultException, VaultSecurityException;

   /**
    * Verifies that vault configuration information is consistent across the grid. Will be removed
    * once central configuration storage is implemented.
    * 
    * @deprecated
    */
   void checkVault(UUID vaultIdentifier, UUID accountIdentifier, CredentialsManager manager)
         throws VaultException, VaultSecurityException;

}
