//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: Jun 7, 2007
//---------------------

package org.cleversafe.vault;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import org.apache.xmlbeans.XmlException;
import org.cleversafe.config.ExecutionContext;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;

/**
 * An XML implementation of a that loads a vault from a vault descriptor XML file.
 * 
 * @author Manish Motwani
 */
public class XMLVaultLoader extends Object implements VaultLoader
{
   /**
    * Default constructor.
    * 
    */
   public XMLVaultLoader()
   {

   }

   /**
    * Loads vault from a given XML input stream.
    * 
    * @param xmlInputStream
    *           XML input stream.
    * @param ctx
    *           {@link ExecutionContext} object that contains a map of named objects as
    *           string-object pairs.
    * @return The {@link Vault} object.
    * @throws IOException
    * @throws XmlException
    * @throws VaultDescriptorException
    * @throws VaultIOException
    */
   public Vault loadVaultFromXML(final InputStream xmlInputStream, final ExecutionContext ctx)
         throws IOException, XmlException, VaultDescriptorException, VaultACLException, VaultIOException
   {
      final XMLVaultDescriptor descriptor = new XMLVaultDescriptor(xmlInputStream, ctx);
      return loadVaultFromDescriptor(descriptor);
   }

   /**
    * Loads vault from a given file.
    * 
    * @param xmlFileName
    *           XML file name
    * @param ctx
    *           {@link ExecutionContext} object that contains a map of named objects as
    *           string-object pairs.
    * @return The {@link Vault} object.
    * @throws IOException
    * @throws XmlException
    * @throws VaultDescriptorException
    * @throws FileNotFoundException
    * @throws VaultIOException
    */
   public Vault loadVaultFromXML(final String xmlFileName, final ExecutionContext ctx)
         throws IOException, XmlException, VaultDescriptorException, VaultACLException, FileNotFoundException,
         VaultIOException
   {
      final XMLVaultDescriptor descriptor = new XMLVaultDescriptor(xmlFileName, ctx);
      return loadVaultFromDescriptor(descriptor);
   }

   /**
    * Loads vault from a given {@link VaultDescriptor}.
    * 
    * @param xmlFileName
    *           XML file name.
    * @param ctx
    *           {@link ExecutionContext} object that contains a map of named objects as
    *           string-object pairs.
    * @return The {@link Vault} object.
    * @throws VaultDescriptorException
    */
   public Vault loadVaultFromDescriptor(final VaultDescriptor descriptor)
         throws VaultDescriptorException, VaultACLException
   {
      assert descriptor instanceof XMLVaultDescriptor : "Input descriptor must be of type "
            + XMLVaultDescriptor.class.getName();

      // Create the vault with parameters from the descriptor
      return descriptor.createVaultObject();
   }

   /**
    * Loads vault from a given input stream.
    * 
    * @param xmlInputStream
    *           XML input stream to use
    * @param ctx
    *           {@link ExecutionContext} object that contains a map of named objects as
    *           string-object pairs.
    * @return The {@link Vault} object.
    * @throws VaultDescriptorException
    * @throws VaultIOException
    *            if the vault cannot be loaded.
    */
   public Vault loadVaultFromStream(final InputStream xmlInputStream, final ExecutionContext ctx)
         throws VaultDescriptorException, VaultACLException, VaultIOException
   {
      try
      {
         return this.loadVaultFromXML(xmlInputStream, ctx);
      }
      catch (XmlException ex)
      {
         throw new VaultIOException("Could not load vault from XML stream", ex);
      }
      catch (IOException ex)
      {
         throw new VaultIOException("Could not load vault from XML stream", ex);
      }
   }

   public VaultDescriptor loadVaultDescriptor(
         InputStream vaultDescriptorStream,
         final ExecutionContext ctx) throws VaultDescriptorException, VaultACLException,
         VaultIOException
   {
      try
      {
         return new XMLVaultDescriptor(vaultDescriptorStream, ctx);
      }
      catch (IOException e)
      {
         throw new VaultIOException("Error loading vault descriptor", e);
      }
      catch (XmlException e)
      {
         throw new VaultIOException("XML error while loading vault descriptor", e);
      }
   }
}
