//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: May 30, 2007
//---------------------

package org.cleversafe.vault.storage;

import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;
import java.util.Map.Entry;

import javax.crypto.Cipher;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.util.ASN1UniqueIdentifier;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultKeyException;
import org.cleversafe.vault.exceptions.VaultKeyLookupException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.cleversafe.vault.storage.asn1.EncryptedKeyInfo;
import org.cleversafe.vault.storage.asn1.PlainKeyInfo;
import org.cleversafe.vault.storage.asn1.VaultACLEntry;
import org.cleversafe.vault.storage.asn1.VaultKeys;
import org.cleversafe.vault.storage.asn1.VaultPermissionEntry;
import org.cleversafe.vault.storage.asn1.VaultPublicKeys;

/**
 * Stores a Vault ACL to ASN.1.
 * 
 * <pre>
 * VaultACL   ::= SEQUENCE  {
 *      vaultIdentifier   UUID,
 *      -- aclPublicKey      KeyInfo,   -- used when acl signing is implemented
 *      vaultOwner        UUID,
 *      -- keyDigests        Digests,   -- used when key digests are implemented
 *      publicKeys        VaultPublicKeys,
 *      acl               SEQUENCE OF VaultACLEntry  }
 * </pre>
 */
public class ASN1VaultACL extends ASN1Encodable
{
   public static class EncryptedPrivateKey implements PrivateKey
   {
      private static final long serialVersionUID = 1L;

      private EncryptedKeyInfo keyInfo;

      public EncryptedPrivateKey(EncryptedKeyInfo keyInfo)
      {
         this.keyInfo = keyInfo;
      }

      public String getAlgorithm()
      {
         return keyInfo.getAlgorithm();
      }

      public byte[] getEncoded()
      {
         throw new RuntimeException(new VaultSecurityException("Attempted to use an encrypted key"));
      }

      public String getFormat()
      {
         return "null";
      }
   }

   private ASN1UniqueIdentifier vaultIdentifier;

   // TODO: Add ACL modification keypair
   // private KeyInfo aclPublicKey;

   private ASN1UniqueIdentifier owner;
   private VaultPublicKeys publicKeys;

   private Map<UUID, VaultACLEntry> aclMap;

   private static Map<UUID, VaultACLEntry> constructACLMap(ASN1Sequence seq)
   {
      Map<UUID, VaultACLEntry> aclMap = new HashMap<UUID, VaultACLEntry>();

      Enumeration<?> e = seq.getObjects();

      while (e.hasMoreElements())
      {
         VaultACLEntry entry = VaultACLEntry.getInstance(e.nextElement());
         aclMap.put(entry.getAccountIdentifier(), entry);
      }

      return aclMap;
   }

   private static ASN1Sequence constructACLSequence(Map<UUID, VaultACLEntry> map)
   {
      ASN1EncodableVector v = new ASN1EncodableVector();

      Iterator<VaultACLEntry> it = map.values().iterator();
      while (it.hasNext())
      {
         v.add((VaultACLEntry) it.next());
      }

      return new DERSequence(v);
   }

   // Constructor for use with VaultACLFactory
   public ASN1VaultACL()
   {
      this.vaultIdentifier = null;
      this.owner = null;
      this.publicKeys = new VaultPublicKeys();
      this.aclMap = new HashMap<UUID, VaultACLEntry>();
   }

   public ASN1VaultACL(ASN1VaultACL acl)
   {
      this.vaultIdentifier = acl.vaultIdentifier;
      this.owner = acl.owner;
      this.publicKeys = new VaultPublicKeys(acl.publicKeys);
      this.aclMap = new HashMap<UUID, VaultACLEntry>(); // VaultACLEntry is mutable
      for (Map.Entry<UUID, VaultACLEntry> entry : acl.aclMap.entrySet())
      {
         // TODO: VaultACLEntry needs to implement clone() for this to really work properly
         this.aclMap.put(entry.getKey(), new VaultACLEntry(entry.getValue()));
      }
   }

   public ASN1VaultACL(UUID vaultIdentifier, UUID vaultOwner)
   {
      super();
      this.vaultIdentifier = new ASN1UniqueIdentifier(vaultIdentifier);
      this.owner = new ASN1UniqueIdentifier(vaultOwner);
      this.aclMap = new HashMap<UUID, VaultACLEntry>();
   }

   public ASN1VaultACL(ASN1Sequence seq)
   {
      if (seq.size() != 4)
      {
         throw new IllegalArgumentException("Bad sequence size: " + seq.size());
      }

      Enumeration<?> e = seq.getObjects();

      this.vaultIdentifier = ASN1UniqueIdentifier.getInstance(e.nextElement());
      this.owner = ASN1UniqueIdentifier.getInstance(e.nextElement());
      this.publicKeys = VaultPublicKeys.getInstance(e.nextElement());
      // TODO: Need to catch a ClassCastException here or check instanceof within construct()
      this.aclMap = constructACLMap((ASN1Sequence) e.nextElement());
   }

   public VaultACLEntry getACLEntry(UUID account)
   {
      // Returns null if the account does not have an entry.
      return this.aclMap.get(account);
   }

   /**
    * Returns the public key stored at the given index.
    */
   public Key getPublicKey(UUID account, int index, PrivateKey privateKey)
         throws VaultACLException, VaultKeyException, VaultKeyLookupException
   {
      PlainKeyInfo info = this.publicKeys.get(index);
      // if no plain key is found, determine why and report back
      if (info == null)
      {
         // first, check if a matching key is in the account's ACL entry ...
         VaultACLEntry entry = this.aclMap.get(account);
         if (entry == null)
         {
            // ... we can't do this if the account has no entry, so report back now
            throw new VaultKeyLookupException("no key stored at index " + index
                  + "; no ACL entry for account " + account + "; check vault descriptor");
         }
         EncryptedKeyInfo einfo = entry.getVaultKeys().get(index);
         if (einfo == null)
         {
            // no matching key stored at that index
            throw new VaultKeyLookupException("no key stored at index " + index
                  + "; check vault descriptor.");
         }
         else if (einfo.getType() == Cipher.SECRET_KEY)
         {
            // the key at that index is not asymmetric
            throw new VaultKeyLookupException("key stored at index " + index
                  + " is a symmetric key; no matching public key exists");
         }
         else
         {
            // strange, there is a valid asymmetric matching key, so this must
            // be an improperly formed ACL
            throw new VaultACLException("no matching public key stored for "
                  + "valid assymetric key entry at index " + index);
         }
      }
      else
      {
         // a proper public key was found
         Key key = info.getKey();
         assert key instanceof PublicKey : "stored key class improper; expected public key";
         return info.getKey();
      }
   }

   /**
    * Returns the secret or private key stored at the given index.
    */
   public Key getKey(UUID account, int index, PrivateKey privateKey) throws VaultACLException,
         VaultKeyException, VaultKeyLookupException, VaultSecurityException
   {

      try
      {
         // first check that the given account has an ACL entry
         VaultACLEntry entry = this.aclMap.get(account);
         if (entry == null)
         {
            throw new VaultSecurityException("key lookup failed; account " + account
                  + " does not have access to this vault");
         }
         // then check that there is an actual key entry at that index
         EncryptedKeyInfo info = entry.getVaultKeys().get(index);
         if (info == null)
         {
            throw new VaultKeyLookupException("no key stored at index " + index
                  + "; check vault descriptor");
         }

         if (privateKey != null)
         {
            // then attempt to unwrap the key
            return EncryptedKeyInfo.unwrap(info, privateKey).getKey();
         }
         else
         {
            return new EncryptedPrivateKey(info);
         }
      }
      catch (KeyException e)
      {
         throw new VaultKeyException("invalid key for account " + account
               + ", could not unwrap key index " + index
               + "; see vault descriptor for matching codec.", e);
      }
      catch (GeneralSecurityException e)
      {
         throw new VaultACLException("could not unwrap key index " + index
               + " within entry for account " + account, e);
      }
      catch (SecurityException e)
      {
         throw new VaultSecurityException("key lookup failed; account " + account
               + " does not have access to key index " + index, e);
      }
   }

   public UUID getOwner()
   {
      return this.owner.getIdentifier();
   }

   public UUID getVaultIdentifier()
   {
      return this.vaultIdentifier.getIdentifier();
   }

   public void storeACLEntry(
         UUID account,
         VaultPermissionEntry permission,
         Map<Integer, EncryptedKeyInfo> vaultKeys)
   {
      VaultKeys keys = new VaultKeys();
      for (Entry<Integer, EncryptedKeyInfo> entry : vaultKeys.entrySet())
      {
         keys.put(entry.getKey(), entry.getValue());
      }
      VaultACLEntry entry = new VaultACLEntry(account, permission, keys);
      this.aclMap.put(account, entry);
   }

   public void storeOwner(UUID vaultOwner)
   {
      this.owner = new ASN1UniqueIdentifier(vaultOwner);
   }

   public void storeVaultIdentifier(UUID vaultIdentifier)
   {
      this.vaultIdentifier = new ASN1UniqueIdentifier(vaultIdentifier);
   }

   public void storePublicKeys(final Map<Integer, PlainKeyInfo> publicKeys)
   {
      this.publicKeys = new VaultPublicKeys();
      for (Entry<Integer, PlainKeyInfo> entry : publicKeys.entrySet())
      {
         this.publicKeys.put(entry.getKey(), entry.getValue());
      }
   }

   public static ASN1VaultACL getInstance(ASN1TaggedObject obj, boolean explicit)
   {
      return getInstance(ASN1Sequence.getInstance(obj, explicit));
   }

   public static ASN1VaultACL getInstance(Object obj)
   {
      if (obj instanceof ASN1VaultACL)
      {
         return (ASN1VaultACL) obj;
      }
      else if (obj instanceof ASN1Sequence)
      {
         return new ASN1VaultACL((ASN1Sequence) obj);
      }

      throw new IllegalArgumentException("unknown object in factory");
   }

   @Override
   public DERObject toASN1Object()
   {
      ASN1EncodableVector v = new ASN1EncodableVector();

      v.add(this.vaultIdentifier);
      v.add(this.owner);
      v.add(this.publicKeys);
      v.add(constructACLSequence(this.aclMap));

      return new DERSequence(v);
   }

}
