//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 18, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.util.Enumeration;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Null;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERUTF8String;
import org.cleversafe.vault.Vault;


/**
 * Contains an encrypted key and enough information to decode it given the proper private key.
 * <p>
 * When a new key is generated with {@link PlainKeyInfo#generateKey(String, int, KeyUsage)} it
 * is plaintext. When the key is destined to be access controlled it must be encrypted
 * before it is stored in the ACL. This is accomplished by "wrapping" 
 * (using {@link #wrap(PlainKeyInfo, PublicKey, String)}) the key with an existing public key.
 * This public key is generally referred to as the "Grid Account Public Encryption Key" and is part
 * of the "Grid Account Encryption Keypair". This keypair exists outside of the vault subsystem
 * and must be provided whenever a vault is created or instantiated.
 * <p>
 * As indicated by the name the keypair is associated with a specific grid account. The account
 * represented by a specific ACL Entry will possess an encryption keypair. It is this
 * keypair that must be used whenever the keys in an ACL entry are unwrapped or the ACL Entry
 * is modified.
 * <p>
 * When a {@link Vault} is instantiated the encrypted keys will be needed by the codec objects.
 * Encrypted keys must be unwrapped using {@link #unwrap(EncryptedKeyInfo, PrivateKey)}. The
 * "Grid Account Private Encryption Key" must be provided to perform this operation.
 * 
 * <pre>
 * EncryptedKeyInfo   ::= CHOICE {
 *      empty            NULL,
 *      stored           SEQUENCE  {
 *                            algorithm        UTF8String,
 *                            type             KeyType,
 *                            usage            KeyUsage,
 *                            transformation   UTF8String,       -- Key wrapping transformation
 *                            intermediateKey  SEQUENCE  {
 *                                                 algorithm        UTF8String,
 *                                                 type             KeyType,
 *                                                 transformation   UTF8String,
 *                                                 key              OCTET STRING  },
 *                            encryptedKey     OCTET STRING  } }
 * </pre>
 */
public class EncryptedKeyInfo extends KeyInfo
{
   private final static String DEFAULT_WRAP_ALGORITHM = "AES";
   private final static int DEFAULT_WRAP_SIZE = 128;
   // If we want to use CBC for key usage, we will need to provide an IV. However,
   // since the wrapped data is random, there doesn't seem to be much risk of a watermarking
   // attack. ECB should be find here.
   private final static String DEFAULT_WRAP_TRANSFORMATION = "AES/ECB/PKCS5Padding";
   
   

   
   private static class IntermediateKey extends ASN1Encodable
   {
      private String algorithm;
      private int type;
      private String transformation;
      private byte[] wrappedKey;
      
      private Key key;
      
      public IntermediateKey( ASN1Sequence seq )
      {
         super();
         if ( seq.size() != 4 )
         {
            throw new IllegalArgumentException("Bad sequence size: " + seq.size());
         }
         
         Enumeration<?> e = seq.getObjects();
         
         this.algorithm = DERUTF8String.getInstance(e.nextElement()).getString();
         this.type = DERInteger.getInstance(e.nextElement()).getValue().intValue();
         this.transformation = DERUTF8String.getInstance(e.nextElement()).getString();
         this.wrappedKey = DEROctetString.getInstance(e.nextElement()).getOctets();
         
         this.key = null;
      }
      
      
      public IntermediateKey( PublicKey publicKey, String asymetricTransformation )
            throws GeneralSecurityException, KeyException
      {
         KeyGenerator keygen = KeyGenerator.getInstance(DEFAULT_WRAP_ALGORITHM);
         keygen.init(DEFAULT_WRAP_SIZE);
         this.key = keygen.generateKey();
         
         this.algorithm = this.key.getAlgorithm();
         this.type = Cipher.SECRET_KEY;
         this.transformation = DEFAULT_WRAP_TRANSFORMATION; // symmetric transformation
         
         Cipher c = Cipher.getInstance( asymetricTransformation );
         c.init( Cipher.WRAP_MODE, publicKey );
         this.wrappedKey = c.wrap(this.key);
      }
      
      public Key unwrap( PrivateKey privateKey, String transformation )
            throws GeneralSecurityException, KeyException
      {
         Cipher c = Cipher.getInstance( transformation );
         c.init( Cipher.UNWRAP_MODE, privateKey );
         this.key = c.unwrap(this.wrappedKey, this.algorithm, this.type);
         return this.key;
      }
      
      public String getAlgorithm()
      {
         return algorithm;
      }
      public void setAlgorithm(String algorithm)
      {
         this.algorithm = algorithm;
      }
      public Key getKey()
      {
         if ( key != null )
         {
            return key;
         }
         else
         {
            throw new SecurityException("Key not yet unwrapped");
         }
      }
      
      public byte[] getWrappedKey()
      {
         return wrappedKey;
      }
      
      public void setWrappedKey( byte[] wrappedKey )
      {
         this.wrappedKey = wrappedKey;
      }
      
      public String getTransformation()
      {
         return transformation;
      }
      public void setTransformation(String transformation)
      {
         this.transformation = transformation;
      }
      public int getType()
      {
         return type;
      }
      public void setType(int type)
      {
         this.type = type;
      }

      
      public static IntermediateKey getInstance( Object obj )
      {
         if ( obj instanceof IntermediateKey )
         {
            return (IntermediateKey)obj;
         }
         else if ( obj instanceof ASN1Sequence )
         {
            return new IntermediateKey( (ASN1Sequence)obj );
         }
         
         throw new IllegalArgumentException("unknown object in factory");
      }

      @Override
      public DERObject toASN1Object()
      {
         ASN1EncodableVector v = new ASN1EncodableVector();
         
         v.add( new DERUTF8String(this.algorithm) );
         v.add( new DERInteger(this.type) );
         v.add( new DERUTF8String(this.transformation) );
         v.add( new DEROctetString(this.wrappedKey) );
         
         return new DERSequence(v);
      }
      
      
      
   }
   
   
   private String transformation;
   private byte[] encryptedKey;
   
   private IntermediateKey intermediate;
   
   private boolean invalid;
   
   
   protected EncryptedKeyInfo()
   {
      this.invalid = true;
   }
   

   // Copy constructor
   protected EncryptedKeyInfo( EncryptedKeyInfo info )
   {
      super(info);
      this.invalid = false;
      this.transformation = info.transformation;
      this.intermediate = info.intermediate;
      this.encryptedKey = info.encryptedKey.clone();
   }
   
   
   // Constructor to use from the key generator and wrap method
   protected EncryptedKeyInfo( 
         Key key,
         KeyUsage usage,
         String transformation,
         IntermediateKey intermediateKey,
         byte[] encryptedKey )
   {
      super( key.getAlgorithm(), queryType(key), usage );
      this.invalid = false;
      this.transformation = transformation;
      this.intermediate = intermediateKey;
      this.encryptedKey = encryptedKey;
   }
   
   protected EncryptedKeyInfo( ASN1Sequence seq )
   {
      super();
      this.invalid = false;
      if ( seq.size() != 6 )
      {
         throw new IllegalArgumentException("Bad sequence size: " + seq.size());
      }
      
      Enumeration<?> e = seq.getObjects();
      
      this.setAlgorithm( DERUTF8String.getInstance(e.nextElement()).getString() );
      this.setType( DERInteger.getInstance(e.nextElement()).getValue().intValue() );
      this.setUsage( KeyUsage.getInstance(e.nextElement()) );
      this.transformation = DERUTF8String.getInstance(e.nextElement()).getString();
      this.intermediate = IntermediateKey.getInstance(e.nextElement());
      this.encryptedKey = DEROctetString.getInstance(e.nextElement()).getOctets();
   }
   
   
   // TODO: Implement key digest checking
   
   /**
    * Checks if the key info object 
    * @throws Security
    */
   private boolean isValid() throws SecurityException
   {
      if ( this.invalid )
      {
         throw new SecurityException("invalid encrypted key");
      }
      else
      {
         return true;
      }
   }
   
   /**
    * Unwraps an encrypted key using a private key.
    * @param info The ciphertext key and associated information.
    * @param privateKey The private key used to perform the unwrap operation.
    * @return An unwraped (plaintext) key and associated information.
    * @throws NoSuchPaddingException 
    * @throws NoSuchAlgorithmException 
    */
   public static PlainKeyInfo unwrap( EncryptedKeyInfo info, PrivateKey privateKey )
         throws KeyException, GeneralSecurityException, SecurityException
   {
      // check that the given key is valid
      info.isValid();
      
      // Unwrap the intermediate keys
      info.intermediate.unwrap(privateKey, info.getTransformation());
      
      Cipher c = Cipher.getInstance( info.intermediate.getTransformation() );
      c.init( Cipher.DECRYPT_MODE, info.intermediate.getKey() );
      
      byte[] decryptedKey = c.doFinal( info.encryptedKey );
      Key key = PlainKeyInfo.decodeKey( decryptedKey, info.getAlgorithm(), info.getType() );
      
      return new PlainKeyInfo( key, info.getUsage() );
   }
   
   /**
    * Wraps a plaintext key using a public key.
    *  
    * @param info The plaintext key and associated information.
    * @param publicKey The public key used to perform the wrap operation.
    * @param transformation The wrapping transformation. This is the transformation done onto the
    *                       plaintext key by the public key. The transformation algorithm must
    *                       match the public key algorithm.
    * @return A wrapped (encrypted) key and associated information.
    */
   public static EncryptedKeyInfo wrap( PlainKeyInfo info, PublicKey publicKey, String transformation )
         throws KeyException, GeneralSecurityException
   {
      // Wrapping is done by generating and encrypting a symmetric key, which will then be used
      // to store the incoming key. We do this instead of directly wrapping the incoming key
      // because that key may be an asymmetric key (which cannot be directly encrypted by
      // the wrapping key). Performing this indirect operation on all keys simplifies the
      // wrapping logic.
      IntermediateKey intKey = new IntermediateKey( publicKey, transformation );
      
      Cipher c = Cipher.getInstance( intKey.getTransformation() );
      c.init( Cipher.ENCRYPT_MODE, intKey.getKey() );
      
      byte[] encryptedKey = c.doFinal( PlainKeyInfo.encodeKey(info.getKey(), info.getType()) );
      
      return new EncryptedKeyInfo(
            info.getKey(),
            info.getUsage(),
            transformation,
            intKey,
            encryptedKey );
   }
   
   
   protected String getTransformation()
   {
      return this.transformation;
   }
   
   @Override
   public Key getKey()
   {
      throw new SecurityException("reading key failed; encrypted key has not been unwrapped");
   }
   
   @Override
   protected void setKey( Key key )
   {
      throw new SecurityException("invalid call to setKey(); key must be wrapped");
   }
   
   
   public static EncryptedKeyInfo getInstance( ASN1TaggedObject obj, boolean explicit )
   {
      return getInstance( ASN1Sequence.getInstance(obj, explicit) );
   }
   
   public static EncryptedKeyInfo getInstance( Object obj )
   {
      if ( obj instanceof EncryptedKeyInfo )
      {
         return (EncryptedKeyInfo)obj;
      }
      else if ( obj instanceof ASN1Sequence )
      {
         return new EncryptedKeyInfo( (ASN1Sequence)obj );
      }
      else if ( obj instanceof ASN1Null )
      {
         return new EncryptedKeyInfo();
      }
      
      throw new IllegalArgumentException("unknown object in factory");
   }
   
   
   @Override
   public DERObject toASN1Object()
   {  
      if ( this.invalid )
      {
         return new DERNull();
      }
      else
      {
         ASN1EncodableVector v = new ASN1EncodableVector();
         
         v.add( new DERUTF8String(this.getAlgorithm()) );
         v.add( new DERInteger(this.getType()) );
         v.add( this.getUsage().getDERObject() );
         v.add( new DERUTF8String(this.transformation) );
         v.add( this.intermediate );
         v.add( new DEROctetString(this.encryptedKey) );
         
         return new DERSequence(v);
      }
   }
   
   

}


