//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 18, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;
import java.security.Permission;

import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.DERObject;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultPermission;


/**
 * Decribes the target usage of a given key. The key usage has implications when the
 * {@link VaultACL} calculates key access permissions.
 * <p>
 * When a key is used for "datasource" operations the codec which uses it is applied, when
 * encoding, before the IDA is applied. Likewise "slice" codecs use the key after the IDA is
 * applied during encoding operations.
 * <pre>
 * KeyUsage   ::= ENUMERATED {
 *      datasourceEncryption(0),
 *      datasourceVerification(1),
 *      sliceVerification(2),
 *      ...  }
 * </pre>
 */
public enum KeyUsage implements DEREncodable
{
   /**
    * The key will be used in a datasource encryption codec.
    */
   DATASOURCE_ENCRYPTION(0),
   
   /**
    * The key will be used in a datasource verification codec.
    */
   DATASOURCE_VERIFICATION(1),
   
   /**
    * The key will be used in a slice verification codec.
    */
   SLICE_VERIFICATION(2);

   
   /**
    * Checks the given permission against the permission required for this usage.
    * <p>
    * All checks assume the target is an access controlled key. For example, all accounts are 
    * have permission to verify slice data so access to the slice verification public key is not 
    * controlled. Therefore this method assumes a check for slice verification usage refers to
    * the private key. With symmetric keys access control is always assumed.
    * 
    * @param permission The permissions being compared.
    * @param vault The vault name.
    * @returns True if the permission implies access to keys with this usage.
    */
   public boolean implies( Permission permission, String vault )
   {
      switch (this)
      {
         case DATASOURCE_ENCRYPTION:
            if ( permission.implies(new VaultPermission(vault, VaultPermission.READ)) ||
                 permission.implies(new VaultPermission(vault, VaultPermission.WRITE)) )
            {
               return true;
            }
         case DATASOURCE_VERIFICATION:
            if ( permission.implies(new VaultPermission(vault, VaultPermission.WRITE)) )
            {
               return true;
            }
         case SLICE_VERIFICATION:
            if ( permission.implies(new VaultPermission(vault, VaultPermission.WRITE)) ||
                 permission.implies(new VaultPermission(vault, VaultPermission.REBUILD)) )
            {
               return true;
            }
      }
      
      return false;
   }
   
   
   
   private final int usage;
   
   KeyUsage( int usage )
   {
      this.usage = usage;
   }
   
   private int getUsage()
   {
      return this.usage;
   }
   
   public static KeyUsage getInstance(Object obj)
   {
      if ( obj instanceof KeyUsage )
      {
         return (KeyUsage)obj;
      }
      else if ( obj instanceof DERInteger )
      {
         int value = ((DERInteger)obj).getValue().intValue();
         for ( KeyUsage usage : KeyUsage.values())
         {
            if ( usage.getUsage() == value )
            {
               return usage;
            }
         }
      }
      
      throw new IllegalArgumentException("unknown object in factory");
   }
   
   
   public DERObject getDERObject()
   {
      return new DERInteger(this.usage);
   }

   
   
   

   
   
   
}


