//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 20, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;
import java.util.Enumeration;
import java.util.UUID;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.util.ASN1UniqueIdentifier;


/**
 * Represents a single vault access control list entry.
 * <p>
 * While the VaultACLEntry object is seemingly immutable, its vault permission entry and vault
 * keys are mutable objects. Thus, instead of creating a new vault ACL entry one should obtain
 * the entry's permission or key structure and modify that.
 * <p>
 * The account identifier associated with an ACL entry is immutable. It is suitable for using
 * as a key in map objects.
 * 
 * <pre>
 * VaultACLEntry   ::= SEQUENCE  {
 *      account         UUID,
 *      permission      VaultPermissionEntry,
 *      keys            VaultKeys  }
 * </pre>
 */
public class VaultACLEntry extends ASN1Encodable
{
   private ASN1UniqueIdentifier accountIdentifier;
   private VaultPermissionEntry permission;
   private VaultKeys vaultKeys;
   
   
   public VaultACLEntry( VaultACLEntry obj )
   {
      super();
      this.accountIdentifier = obj.accountIdentifier; // ASN1UniqueIdentifier is immutable
      this.permission = new VaultPermissionEntry( obj.permission );
      this.vaultKeys = new VaultKeys( obj.vaultKeys );
   }

   public VaultACLEntry(
         UUID accountIdentifier,
         VaultPermissionEntry permission,
         VaultKeys vaultKeys )
   {
      super();
      this.accountIdentifier = new ASN1UniqueIdentifier( accountIdentifier );
      this.permission = new VaultPermissionEntry( permission );
      this.vaultKeys = new VaultKeys( vaultKeys );
   }


   protected VaultACLEntry( ASN1Sequence seq )
   {
      if ( seq.size() != 3 )
      {
         throw new IllegalArgumentException("Bad sequence size: " + seq.size());
      }
      
      Enumeration<?> e = seq.getObjects();
      
      this.accountIdentifier = ASN1UniqueIdentifier.getInstance(e.nextElement());
      this.permission = VaultPermissionEntry.getInstance(e.nextElement());
      this.vaultKeys = VaultKeys.getInstance(e.nextElement());
   }
   
   
   public UUID getAccountIdentifier()
   {
      return accountIdentifier.getIdentifier();
   }

   /**
    * The permission entry object is mutable.
    */
   public VaultPermissionEntry getPermission()
   {
      return permission;
   }

   public VaultKeys getVaultKeys()
   {
      return vaultKeys;
   }

   public static VaultACLEntry getInstance( ASN1TaggedObject obj, boolean explicit )
   {
      return getInstance( ASN1Sequence.getInstance(obj, explicit) );
   }
   
   public static VaultACLEntry getInstance( Object obj )
   {
      if ( obj instanceof VaultACLEntry )
      {
         return (VaultACLEntry)obj;
      }
      else if ( obj instanceof ASN1Sequence )
      {
         return new VaultACLEntry( (ASN1Sequence)obj );
      }
      
      throw new IllegalArgumentException("unknown object in factory");
   }

   @Override
   public DERObject toASN1Object()
   {
      ASN1EncodableVector v = new ASN1EncodableVector();
      
      v.add( this.accountIdentifier );
      v.add( this.permission );
      v.add( this.vaultKeys );
      
      return new DERSequence(v);
   }

}


