//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 19, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;
import java.security.GeneralSecurityException;
import java.security.KeyException;
import java.security.PrivateKey;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERTaggedObject;
import org.cleversafe.vault.VaultACL;


/**
 * Contains encrypted key info objects.
 * <p>
 * This class is meant for use within a vault ACL entry. Each entry will have access to one or
 * more keys. The keys accessible within a given entry are stored in a VaultKeys object. Throughout
 * the ACL keys are accessible by an index. Not all indices are accessible to all ACL entries.
 * A VaultKeys instance will only contain those keys accessible to the entry in which it resides.
 * <p>
 * Key indices are global to all keys stored in the ACL regardless of which structure they reside
 * in. For example, an index will not be reused between keys stored in this structure and keys
 * stored in the {@link VaultPublicKeys} structure. ACL code must check all structures when
 * searching for a key with a given index.
 * 
 * <pre>
 * -- VaultKeys.* must be tagged, not present implies NULL
 * VaultKeys   ::= SEQUENCE OF EncryptedKeyInfo
 * </pre>
 * 
 * @see VaultPublicKeys
 */
public class VaultKeys extends ASN1Encodable
{

   Map<Integer,EncryptedKeyInfo> keys;
   
   public VaultKeys()
   {
      keys = new HashMap<Integer,EncryptedKeyInfo>();
   }
   
   public VaultKeys( VaultKeys keys )
   {
      this.keys = new HashMap<Integer,EncryptedKeyInfo>();
      this.keys.putAll(keys.keys);
   }
   
   protected VaultKeys( ASN1Sequence seq )
   {
      this.keys = new HashMap<Integer,EncryptedKeyInfo>();
      
      Enumeration<?> e = seq.getObjects();
      
      while ( e.hasMoreElements() )
      {
         Object obj = e.nextElement();
         if ( obj instanceof DERTaggedObject )
         {
            Integer index = ((DERTaggedObject)obj).getTagNo();
            // TODO: Should we be indicating explicit tagging?
            EncryptedKeyInfo key = EncryptedKeyInfo.getInstance( (DERTaggedObject)obj, true );
            this.keys.put(index, key);
         }
         else
         {
            throw new IllegalArgumentException("expected sequence of tagged objects");
         }
      }
   }
   
   /**
    * Stores encrypted key info in vault key map. A <code>null</code> key indicates an account
    * does not have access to it.
    */
   public void put( int index, EncryptedKeyInfo key )
   {
      this.keys.put( index, key );
   }
   
   public EncryptedKeyInfo get( int index )
   {
      return this.keys.get(index);
   }
   
   /**
    * Unwraps all encrypted keys and returns them in a map.
    * <p>
    * This method provides a map of plaintext key info objects to the {@link VaultACL}, allowing
    * decrypted keys to be combined with already plaintext keys (see {@link VaultPublicKeys}).
    * This allows keys to be queried using a uniform index scheme.
    * 
    * @param privateKey The grid account private encryption key used to unwrap the encrypted keys.
    * @return A map of keys which the specific ACL entry provides access to.
    * @deprecated
    */
   public Map<Integer,KeyInfo> unwrapKeys( PrivateKey privateKey )
         throws GeneralSecurityException, KeyException, SecurityException
   {
      Map<Integer,KeyInfo> plaintextKeys = new HashMap<Integer,KeyInfo>();
      
      for ( Entry<Integer,EncryptedKeyInfo> entry : this.keys.entrySet() )
      {
         plaintextKeys.put( entry.getKey(), EncryptedKeyInfo.unwrap(entry.getValue(), privateKey) );
      }
      
      return plaintextKeys;
   }
   
   
   
   public static VaultKeys getInstance( ASN1TaggedObject obj, boolean explicit )
   {
      return getInstance( ASN1Sequence.getInstance(obj, explicit) );
   }
   
   
   public static VaultKeys getInstance( Object obj )
   {
      if ( obj instanceof VaultKeys )
      {
         return (VaultKeys)obj;
      }
      else if ( obj instanceof ASN1Sequence )
      {
         return new VaultKeys( (ASN1Sequence)obj );
      }
      
      throw new IllegalArgumentException("unknown object in factory");
   }
   
   
   @Override
   public DERObject toASN1Object()
   {
      ASN1EncodableVector v = new ASN1EncodableVector();
      
      for( Entry<Integer,EncryptedKeyInfo> entry : this.keys.entrySet() )
      {
         v.add( new DERTaggedObject(entry.getKey(), entry.getValue()) );
      }
      
      return new DERSequence(v);
   }

}


