//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 19, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERTaggedObject;

/**
 * Contains key info objects.
 * <p>
 * This class is meant for use within a vault ACL. The ACL contains a set of entries which have
 * access to specific keys. There are a subset of keys to which all entries have access. This
 * structure is meant to contain such keys.
 * <p>
 * Throughout the ACL keys are accessible by an index. A VaultPublicKeys index will only contain
 * those keys accessible to all ACL entries. Key indices are global to all keys stored in the ACL
 * regardless of which structure they reside in. For example, an index will not be reused between
 * keys stored in this structure and keys stored in the {@link VaultKeys} structure. ACL code
 * must check all structures when searching for a key within a given index.
 * 
 * <pre>
 * -- VaultPublicKeys.* must be tagged, not present implies null
 * VaultPublicKeys   ::= SEQUENCE OF KeyInfo
 * </pre>
 *
 * @see VaultKeys
 */
public class VaultPublicKeys extends ASN1Encodable
{
   Map<Integer,PlainKeyInfo> keys;
   
   public VaultPublicKeys()
   {
      keys = new HashMap<Integer,PlainKeyInfo>();
   }
   
   public VaultPublicKeys( VaultPublicKeys keys )
   {
      this.keys = new HashMap<Integer,PlainKeyInfo>();
      this.keys.putAll( keys.keys );
   }

   protected VaultPublicKeys( ASN1Sequence seq )
   {
      this.keys = new HashMap<Integer,PlainKeyInfo>();
      
      Enumeration<?> e = seq.getObjects();
      
      while ( e.hasMoreElements() )
      {
         Object obj = e.nextElement();
         if ( obj instanceof DERTaggedObject )
         {
            Integer index = ((DERTaggedObject)obj).getTagNo();
            // TODO: Should we be indicating explicit tagging?
            PlainKeyInfo key = PlainKeyInfo.getInstance( (DERTaggedObject)obj, true );
            this.keys.put(index, key);
         }
         else
         {
            throw new IllegalArgumentException("expected sequence of tagged objects");
         }
      }
   }
   
   public void put( int index, PlainKeyInfo key )
   {
      this.keys.put( index, key );
   }
   
   public PlainKeyInfo get( int index )
   {
      return this.keys.get(index);
   }
   
   public Map<Integer,KeyInfo> getKeys()
   {
      Map<Integer,KeyInfo> value = new HashMap<Integer,KeyInfo>();
      value.putAll( this.keys );
      return value;
   }
   
   public static VaultPublicKeys getInstance( ASN1TaggedObject obj, boolean explicit )
   {
      return getInstance( ASN1Sequence.getInstance(obj, explicit) );
   }
   
   public static VaultPublicKeys getInstance( Object obj )
   {
      if ( obj instanceof VaultPublicKeys )
      {
         return (VaultPublicKeys)obj;
      }
      else if ( obj instanceof ASN1Sequence )
      {
         return new VaultPublicKeys( (ASN1Sequence)obj );
      }
      
      throw new IllegalArgumentException("unknown object in factory");
   }
   
   
   
   @Override
   public DERObject toASN1Object()
   {
      ASN1EncodableVector v = new ASN1EncodableVector();
      
      for( Entry<Integer,PlainKeyInfo> entry : this.keys.entrySet() )
      {
         v.add( new DERTaggedObject(entry.getKey(), entry.getValue()) );
      }
      
      return new DERSequence(v);
   }

}


