//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: jmullin
//
// Date: June 9, 2007
//---------------------

package org.cleversafe.server.handlers.authentication;

import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.FileInputStream;
import java.util.Properties;
import java.util.UUID;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import org.cleversafe.layer.protocol.PasswordAuthenticationRequest;
import org.cleversafe.layer.protocol.PasswordAuthenticationResponse;
import org.cleversafe.server.ClientSession;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * A test of PAM Authentication using JPAM. This unit test is derived from and is very similar to
 * the AlwaysAuthenticateAuthenticationTest class.
 * 
 * @see AlwaysAuthenticateAuthenticationTest
 * @see PasswordAuthenticationHandler
 * @author Joshua Mullin
 */
public class PAMAuthenticationTest
{
   private static Logger _logger = Logger.getLogger(PAMAuthenticationTest.class);

   private static String _jaasConfigFile;
   private static String _jaasProfile;
   private static Properties _properties;

   private static final String TEST_CONFIG_PROPERTY = "org.cleversafe.test.config";
   private static final String TEST_PROPERTIES_FILE = "PAMAuthenticationTest.properties";

   private static final String TEST_CONFIG_PATH =
         "test-data" + File.separator + "org" + File.separator + "cleversafe" + File.separator
               + "server" + File.separator + "handlers" + File.separator + "authentication";

   private static final String VALID_USERNAME_PROPERTY = "valid-username";
   private static final String VALID_PASSWORD_PROPERTY = "valid-password";

   private static final String INVALID_USERNAME_PROPERTY = "invalid-username";
   private static final String INVALID_PASSWORD_PROPERTY = "invalid-password";

   private static final String DISABLED_USERNAME_PROPERTY = "disabled-username";
   private static final String DISABLED_PASSWORD_PROPERTY = "disabled-password";

   /**
    * Loads test properties, including JAAS configuration information. Also initializes logging. The
    * system property org.cleversafe.test.config should point to the directory where
    * PAMAuthentication.properties is located.
    * 
    * @throws Exception
    *            If PAMAuthentication.properties could not be loaded.
    */
   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
      _properties = new Properties();

      if (System.getProperty(TEST_CONFIG_PROPERTY) == null)
      {
         System.setProperty(TEST_CONFIG_PROPERTY, TEST_CONFIG_PATH);
      }

      // Load test properties
      String configFile = "";
      try
      {
         configFile =
               System.getProperty(TEST_CONFIG_PROPERTY) + File.separator + TEST_PROPERTIES_FILE;

         _properties.load(new FileInputStream(configFile));
      }
      catch (Exception ex)
      {
         throw new Exception("Could not open open config file " + configFile);
      }

      // Setup logging
      String logPropertyFile = _properties.getProperty("log-properties");
      if (logPropertyFile != null)
      {
         PropertyConfigurator.configure(logPropertyFile);
      }
      else
      {
         BasicConfigurator.configure();
      }

      // Setup test parameters
      _jaasConfigFile = _properties.getProperty("jaas-config");
      _jaasProfile = _properties.getProperty("jaas-profile");
   }

   /**
    * Confirms that a valid username and password will authenticate.
    * 
    */
   @Test
   public void testValidUsernamePassword()
   {
      _logger.info("Test Case: valid username and password");

      boolean isAuthenticated = false;

      PasswordAuthenticationRequest request = null;
      PasswordAuthenticationResponse response = null;

      try
      {
         request = new PasswordAuthenticationRequest();
         request.setUsername(_properties.getProperty(VALID_USERNAME_PROPERTY));
         request.setPassword(_properties.getProperty(VALID_PASSWORD_PROPERTY));
         request.setGridAccountID(new UUID(0, 0));

         ClientSession session = new ClientSession();

         PasswordAuthenticationHandler handler;
         handler = null; //new PasswordAuthenticationHandler(jaasProfile, jaasConfigFile);

         response = (PasswordAuthenticationResponse) handler.service(request, session);

         isAuthenticated = response.getAuthenticated();
      }
      catch (Exception ex)
      {
         isAuthenticated = false;
      }

      assertTrue("Authentication failed unexpectedly", isAuthenticated);
   }

   /**
    * Confirms that an invalid username and password will not authenticate.
    * 
    */
   @Test
   public void testInvalidUsernamePassword()
   {
      _logger.info("Test Case: invalid username and password");

      boolean isAuthenticated = true;

      PasswordAuthenticationRequest request = null;
      PasswordAuthenticationResponse response = null;

      try
      {
         request = new PasswordAuthenticationRequest();
         request.setUsername(_properties.getProperty(INVALID_USERNAME_PROPERTY));
         request.setPassword(_properties.getProperty(INVALID_PASSWORD_PROPERTY));
         request.setGridAccountID(new UUID(0, 0));

         ClientSession session = new ClientSession();

         PasswordAuthenticationHandler handler;
         handler = null; //new PasswordAuthenticationHandler(jaasProfile, jaasConfigFile);

         response = (PasswordAuthenticationResponse) handler.service(request, session);

         isAuthenticated = response.getAuthenticated();
      }
      catch (Exception ex)
      {
         isAuthenticated = false;
      }

      assertTrue("Authentication succeeded unexpectedly", !isAuthenticated);
   }

   /**
    * Confirms that a disabled username and password will not authenticate.
    * 
    */
   @Test
   public void testDisabledUsernamePassword()
   {
      _logger.info("Test Case: disabled username and password");

      boolean isAuthenticated = true;

      PasswordAuthenticationRequest request = null;
      PasswordAuthenticationResponse response = null;

      try
      {
         request = new PasswordAuthenticationRequest();
         request.setUsername(_properties.getProperty(DISABLED_USERNAME_PROPERTY));
         request.setPassword(_properties.getProperty(DISABLED_PASSWORD_PROPERTY));
         request.setGridAccountID(new UUID(0, 0));

         ClientSession session = new ClientSession();

         PasswordAuthenticationHandler handler;
         handler = null; //new PasswordAuthenticationHandler(jaasProfile, jaasConfigFile);

         response = (PasswordAuthenticationResponse) handler.service(request, session);

         isAuthenticated = response.getAuthenticated();
      }
      catch (Exception ex)
      {
         isAuthenticated = false;
      }

      assertTrue("Authentication succeeded unexpectedly", !isAuthenticated);
   }

}
