
package org.cleversafe.vault;

import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.Set;

import org.cleversafe.codec.Codec;
import org.cleversafe.codec.integrity.CRCIntegrityCodec;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.ida.optimizedcauchy.CauchyInformationDispersalCodec;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.layer.slicestore.memory.MemorySliceStore;
import org.cleversafe.layer.slicestore.unreliable.UnreliableSliceStore;
import org.cleversafe.util.NChooseR;

/**
 * Helper class to generate Vault permutations
 */
public class VaultGenerator
{
   private static Random random = new Random();
   
   // Permutation configuration
   private int idaWidth;
   private int minOutages;
   private int maxOutages;

   // Permutation state
   private int curIdaThreshold;
   private int curReadThreshold;
   private int curWriteThreshold;
   private int curNumOutages;
   private NChooseR curOutages; // n=2*width (read + write), r=total # failures

   public VaultGenerator(int idaWidth, int minOutages, int maxOutages)
   {
      assert maxOutages >= minOutages;

      this.idaWidth = idaWidth;
      this.minOutages = minOutages;
      this.maxOutages = maxOutages;

      this.curIdaThreshold = 1;
      this.curReadThreshold = this.curIdaThreshold;
      this.curWriteThreshold = this.curIdaThreshold;
      this.curNumOutages = minOutages;
      this.curOutages = new NChooseR(2 * this.idaWidth, this.curNumOutages);
   }

   public long getNumVaults()
   {
      long num = 0;

      // read and write thresholds
      for (int i = 1; i <= this.idaWidth; ++i)
      {
         num += i * i;
      }

      // # permutations per outage
      int outages = 0;
      for (int i = minOutages; i <= maxOutages; ++i)
      {
         outages += NChooseR.calculateSize(2 * this.idaWidth, i);
      }
      if (outages > 0)
      {
         num *= outages;
      }

      return num;
   }

   public MockVault next() throws IDAInvalidParametersException
   {
      Set<Integer> outages;

      // Completion condition
      if (this.curIdaThreshold > this.idaWidth)
      {
         return null;
      }

      // Iteration logic
      if (this.curReadThreshold > this.idaWidth)
      {
         ++this.curIdaThreshold;
         this.curReadThreshold = this.curIdaThreshold;
         this.curWriteThreshold = this.curIdaThreshold;
         return this.next();
      }
      else if (this.curWriteThreshold > this.idaWidth)
      {
         this.curWriteThreshold = this.curIdaThreshold;
         ++this.curReadThreshold;
         return this.next();
      }
      else if (this.curNumOutages > this.maxOutages)
      {
         this.curNumOutages = this.minOutages;
         this.curOutages = new NChooseR(2 * this.idaWidth, this.curNumOutages);
         ++this.curWriteThreshold;
         return this.next();
      }
      else if ((outages = this.curOutages.next()) == null)
      {
         ++this.curNumOutages;
         this.curOutages = new NChooseR(2 * this.idaWidth, this.curNumOutages);
         return this.next();
      }

      // Paramers are set, generate the vault
      int numWriteFailures = 0;
      int numReadFailures = 0;

      InformationDispersalCodec ida = null;
      try
      {
         ida = new CauchyInformationDispersalCodec(this.idaWidth, this.curIdaThreshold, 1);
      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }

      int numCumulativeErrors = 0;
      int numCumulativeTxInvalid = 0;
      int numCumulativeCorrupt = 0;
      List<SliceStore> stores = new ArrayList<SliceStore>(this.idaWidth);
      for (int i = 0; i < this.idaWidth; ++i)
      {
         UnreliableSliceStore store = new UnreliableSliceStore(new MemorySliceStore());

         // Write & read failures
         boolean writeFailure = outages.contains(i);
         boolean readFailure = outages.contains(i + this.idaWidth);
         try
         {
            if (writeFailure)
            {
               double rnd = random.nextDouble();
               if (rnd > 0.7)
               {
                  store.setMethodExceptionRate("write", SliceStoreIOException.class, 1.0);
               }
               else if (rnd > .4)
               {
                  store.setMethodExceptionRate("write", SliceStoreTransactionException.class, 1.0);
               }
               else if (rnd > .2)
               {
                  store.setMethodExceptionRate("commitTransaction", SliceStoreIOException.class,
                        1.0);
                  store.setMethodExceptionRate("rollbackTransaction", SliceStoreIOException.class,
                        1.0);
               }
               else
               {
                  store.setMethodExceptionRate("commitTransaction",
                        SliceStoreTransactionException.class, 1.0);
                  store.setMethodExceptionRate("rollbackTransaction",
                        SliceStoreTransactionException.class, 1.0);
               }

               // All write failures result in an invalid transaction on read, 
               // unless overridden by a read failure
               ++numCumulativeTxInvalid;
               ++numWriteFailures;
            }
            if (readFailure)
            {
               double rnd = random.nextDouble();
               if (rnd > 0.5)
               {
                  store.setMethodExceptionRate("read", SliceStoreIOException.class, 1.0);

                  // If a write failure already occurred from this source causing an 
                  // invalid transaction id, this exception overrides it
                  if (writeFailure)
                  {
                     --numCumulativeTxInvalid;
                  }
                  ++numCumulativeErrors;
               }
               else if (rnd > 0.2)
               {
                  // Transaction error does NOT override a transaction error caused by a write failure
                  if (!writeFailure)
                  {
                     // Transaction must be offset down, or this may become the new high tx and throw off our accounting
                     store.setWrongTransactionFailureRate(1.0);
                     ++numCumulativeTxInvalid;
                     store.setTransactionOffset(-1 * numCumulativeTxInvalid);
                  }
               }
               else
               {
                  // Corruption does NOT override a transaction error caused by a write failure
                  if (!writeFailure)
                  {
                     store.setCorruptionFailureRate(1.0);
                     ++numCumulativeCorrupt;
                  }
               }

               ++numReadFailures;
            }
         }
         catch (Exception ex)
         {
            fail(ex.getMessage());
         }

         stores.add(store);
      }

      List<Codec> codecs = new ArrayList<Codec>();
      List<Codec> sliceCodecs = new ArrayList<Codec>();
      sliceCodecs.add(new CRCIntegrityCodec()); // Use CRC to allow us to test corruption

      try
      {
         MockVault vault =
               new MockVault(ida, codecs, sliceCodecs, stores, this.curWriteThreshold,
                     this.curReadThreshold, numWriteFailures, numReadFailures, numCumulativeErrors,
                     numCumulativeTxInvalid, numCumulativeCorrupt, outages);
         vault.startSessions();
         return vault;
      }
      catch (IDAInvalidParametersException ex)
      {
         // Check for expected failure
         if (this.curWriteThreshold < this.curIdaThreshold
               || this.curReadThreshold < this.curIdaThreshold)
         {
            fail("Internal error - invalid vault parameters");
            return null;
         }
         else
         {
            fail("Unexpected vault creation failure");
            return null;
         }
      }
   }
}
