//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: Jason Resch
//
// Date: Sep 20, 2007
//---------------------

package org.cleversafe.codec;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.Random;

import org.cleversafe.layer.grid.SourceName;
import org.junit.Test;

public abstract class CodecPerformanceTest
{
   private static int DATA_SIZE = 8*1024*1024;  // 8 MB
   private static int CHUNK_SIZE = 4096;        // 4 KB
   
   private Codec codec = null;
   
   public abstract Codec getCodec() throws Exception;
   
   @Test
   public void testCodecPerformance() throws Exception
   {
      this.codec = this.getCodec();
      
      System.out.println("Beginning performance test for codec: " + this.codec.getName());
      
      findEncodeRate();
      findDecodeRate();
      
      verifyDataIntegrity();
   }
   
   private void findEncodeRate() throws Exception
   {

      byte data[] = new byte[CHUNK_SIZE];
      Random rng = new Random();
      
      rng.nextBytes(data);
      
      int iterations = DATA_SIZE / CHUNK_SIZE;
      
      ByteArrayOutputStream encodedStream = new ByteArrayOutputStream();
      
      Encoder encoder = this.codec.getEncoder();
      encoder.reset(new SourceName("0"), 1);
      
      System.out.println("Performing encode operation...");
      
      long startTime = System.currentTimeMillis();   
      
      for (int i=0; i<iterations; i++)
      {
         byte encoded[] = encoder.process(data);
         encodedStream.write(encoded);
      }
      encodedStream.write(encoder.finish());
      
      long stopTime = System.currentTimeMillis();
      long encodeTime = stopTime - startTime;
      float throughput = Math.round((DATA_SIZE * 8) / (encodeTime)) * 1000;
      throughput /= (1024 * 1024);
      
      System.out.println("Encode complete, performed at " + throughput + " Mbps");
      System.out.println("");
   }


   private void findDecodeRate() throws Exception
   {
      byte data[] = new byte[CHUNK_SIZE];
      Random rng = new Random();
      
      rng.nextBytes(data);

      int iterations = DATA_SIZE / CHUNK_SIZE;
      
      ByteArrayOutputStream encodedStream = new ByteArrayOutputStream();
      
      Encoder encoder = this.codec.getEncoder();
      encoder.reset(new SourceName("0"), 1);
      
      for (int i=0; i<iterations; i++)
      {
         encodedStream.write(encoder.process(data));
      }
      encodedStream.write(encoder.finish());
      

      // Perform Decode test using the encoded data
      
      byte[] encodedData = encodedStream.toByteArray();
      ByteArrayInputStream inputStream = new ByteArrayInputStream(encodedData);
      
      Decoder decoder = this.codec.getDecoder();
      decoder.reset(new SourceName("0"), 1);
      
      System.out.println("Performing decode operation...");
      
      long startTime = System.currentTimeMillis();
      
      while (inputStream.available() > 0)
      {
         int chunkLen = (inputStream.available() > CHUNK_SIZE ? CHUNK_SIZE : inputStream.available());
         
         byte[] chunk = new byte[chunkLen];
         inputStream.read(chunk, 0, chunk.length);
         
         decoder.process(chunk);
      }
      decoder.finish();
      
      long stopTime = System.currentTimeMillis();
      long encodeTime = stopTime - startTime;
      float throughput = Math.round((DATA_SIZE * 8) / (encodeTime)) * 1000;
      throughput /= (1024 * 1024);
      
      System.out.println("Decode complete, performed at " + throughput + " Mbps");
      System.out.println("");
      
   }
   
   private void verifyDataIntegrity() throws Exception
   {
      
      byte data[] = new byte[CHUNK_SIZE];
      Random rng = new Random();
      
      rng.nextBytes(data);

      int iterations = DATA_SIZE / CHUNK_SIZE;
      
      ByteArrayOutputStream encodedStream = new ByteArrayOutputStream();
      
      Encoder encoder = this.codec.getEncoder();
      encoder.reset(new SourceName("0"), 1);
      
      for (int i=0; i<iterations; i++)
      {
         encodedStream.write(encoder.process(data));
      }
      encodedStream.write(encoder.finish());
      

      // Perform Decode test using the encoded data
      
      byte[] encodedData = encodedStream.toByteArray();
      ByteArrayInputStream inputStream = new ByteArrayInputStream(encodedData);
      
      Decoder decoder = this.codec.getDecoder();
      decoder.reset(new SourceName("0"), 1);
      
      ByteArrayOutputStream decodedStream = new ByteArrayOutputStream();
      
      while (inputStream.available() > 0)
      {
         int chunkLen = (inputStream.available() > CHUNK_SIZE ? CHUNK_SIZE : inputStream.available());
         
         byte[] chunk = new byte[chunkLen];
         inputStream.read(chunk, 0, chunk.length);
         
         decodedStream.write(decoder.process(chunk));
      }
      decodedStream.write(decoder.finish());

      // Compare Decoded data to original

      System.out.println("Verifying data integrity...");
      
      byte[] decodedData = decodedStream.toByteArray();
      
      org.junit.Assert.assertEquals(iterations*CHUNK_SIZE, decodedData.length);
      for (int i=0; i<decodedData.length; i++)
      {
         org.junit.Assert.assertEquals(data[i%CHUNK_SIZE], decodedData[i]);
      }
      
      System.out.println("Data matches, test complete.");
   }
   
}


