
package org.cleversafe.gridapi;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.cauchy.CauchyInformationDispersalCodec;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.GridController;
import org.cleversafe.layer.grid.ReadController;
import org.cleversafe.layer.grid.ReadControllerFactory;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.WriteController;
import org.cleversafe.layer.grid.WriteControllerFactory;
import org.cleversafe.layer.grid.exceptions.ControllerException;
import org.cleversafe.layer.grid.simplecontroller.SimpleReadControllerFactory;
import org.cleversafe.layer.grid.simplecontroller.SimpleWriteControllerFactory;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.unreliable.UnreliableSliceStore;
import org.cleversafe.test.IteratedWorkItem;
import org.cleversafe.test.PerformanceTestResult;
import org.cleversafe.test.PerformanceTester;
import org.cleversafe.test.SimplePerformanceTester;
import org.cleversafe.test.ThreadedPerformanceTester;
import org.cleversafe.test.WorkItem;
import org.cleversafe.test.WorkItemProvider;
import org.cleversafe.util.math.CombinationGenerator;
import org.cleversafe.vault.BaseVault;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultACLFactory;
import org.cleversafe.vault.storage.VaultKeyInfo;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

public class SimpleSliceStoreControllerPerformanceTest
{
   private static final int GRID_PILLARS = 16;
   private static final int IDA_THRESHOLD = 12;

   private static final String TEST_PATH =
         "test-data/" + GridController.class.getName().replace('.', '/');

   private static final String TEST_OUTPUT_PATH = TEST_PATH + "/output/";

   private static final int DATA_SIZE = 4096;

   // The minimum write rate is 10 megabits / second
   private static final double MIN_WRITE_RATE = (100 * 1024 * 1024) / 8;

   // The minimum read and rate is 10 megabits / second
   private static final double MIN_READ_RATE = (10 * 1024 * 1024) / 8;

   private static final long PERFORMANCE_TEST_DATA_SIZE = 12 * 1024 * 1024;
   private static final int PERFORMANCE_TEST_THREAD_POOL_SIZE = 20;

   private Vault vault;
   private GridController gridController = null;
   private byte data[];
   private byte data2[];

   @Before
   public void setUp()
   {
      try
      {
         vault = getVault();
         WriteControllerFactory cwf = new SimpleWriteControllerFactory();
         ReadControllerFactory crf = new SimpleReadControllerFactory();
         this.gridController = new GridController(vault, cwf, crf);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }

      data = new byte[DATA_SIZE];
      randomizeBuffer(data);

      data2 = new byte[DATA_SIZE];
      randomizeBuffer(data2);
   }

   @After
   public void cleanUp() throws SliceStoreLayerException
   {
      for (int idx = 0; idx < vault.getSliceStores().size(); idx++)
      {
         if (vault.getSliceStores().get(idx) != null)
         {
            vault.getSliceStores().get(idx).endSession();
         }
      }
      assertTrue(deleteDirectory(new File(TEST_OUTPUT_PATH)));
   }

   @Test
   public void testSingleReadWrite()
   {
      try
      {
         testReadWrite();
      }
      catch (Exception e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
         fail(e.getMessage());
      }
   }

   public void testReadWrite() throws Exception
   {
      for (int blockId = 0; blockId < 100; blockId++)
      {
         testReadWrite(blockId);
      }
   }

   public void testReadWrite(long blockId) throws Exception
   {
      ReadController readController = gridController.getReadController();
      WriteController writeController = gridController.getWriteController();
      SourceName name = new SourceName(String.valueOf(blockId));

      try
      {
         DataSource dataSource = new DataSource(name, writeController.getTransactionId(), data);
         writeController.write(dataSource);
         writeController.commit();
      }
      catch (Exception ex)
      {
         writeController.rollback();
         throw ex;
      }

      DataSource readData = readController.read(name);
      assertEquals(name, readData.getName());
      assertEquals(writeController.getTransactionId(), readData.getTransactionId());
      assertTrue(Arrays.equals(readData.getData(), data));
   }

   @Test
   public void testReadUsingUnreliableSliceStores()
   {
      // Test read errors
      testReadWriteUsingUnreliableSliceStores(1, 0);
   }

   @Test
   public void testWriteUsingUnreliableSliceStores()
   {
      // Test write errors
      testReadWriteUsingUnreliableSliceStores(0, 1);
   }

   @Test
   public void testReadWriteUsingUnreliableSliceStores()
   {
      // Test read and write errors
      testReadWriteUsingUnreliableSliceStores(1, 1);
   }

   public void testReadWriteUsingUnreliableSliceStores(
         double readFailureRate,
         double writeFailureRate)
   {
      // TODO: Figure out why the performance is no longer sufficient to run an
      // exhaustive test

      // CombinationGenerator combinationGenerator =
      // new CombinationGenerator(vault.getWidth(), vault.getLoseCount() );

      CombinationGenerator combinationGenerator = new CombinationGenerator(vault.getWidth(), 2);

      SliceStore reliableStores[] = new SliceStore[vault.getSliceStores().size()];
      for (int idx = 0; idx < reliableStores.length; idx++)
      {
         reliableStores[idx] = vault.getSliceStores().get(idx);
      }

      while (combinationGenerator.hasMore())
      {
         List<SliceStore> unreliableStores = vault.getSliceStores();

         for (int idx = 0; idx < unreliableStores.size(); idx++)
         {
            unreliableStores.set(idx, reliableStores[idx]);
         }

         int combination[] = combinationGenerator.getNext();

         for (int idx = 0; idx < combination.length; idx++)
         {
            int storeIdx = combination[idx];

            try
            {
               unreliableStores.set(storeIdx, new UnreliableSliceStore(
                     unreliableStores.get(storeIdx), readFailureRate, writeFailureRate, 0, 0));
            }
            catch (NoSuchMethodException e)
            {
               e.printStackTrace();
               fail(e.getMessage());
            }
         }

         if (writeFailureRate > 0)
         {
            try
            {
               testReadWrite();
               fail("Write should have failed due to missing slice stores");
            }
            catch (Exception ex)
            {
            }
         }
         else
         {
            try
            {
               testReadWrite();
            }
            catch (Exception ex)
            {
               fail(ex.getMessage());
            }
         }
      }

   }

   @Test
   public void testTransactions()
   {

      try
      {
         ReadController readController = gridController.getReadController();
         WriteController writeController = gridController.getWriteController();

         SourceName name = new SourceName("0");
         DataSource dataSource = new DataSource(name, writeController.getTransactionId(), data);
         writeController.write(dataSource);

         DataSource readData = null;

         // Attempt to read uncommitted transaction
         try
         {
            readData = readController.read(name);
            // Warning was able to read uncommitted transaction
         }
         catch (Exception ex)
         {
            // This is expected behavior
         }

         writeController.commit();
         readData = readController.read(name);

         assertEquals(name, readData.getName());
         assertEquals(writeController.getTransactionId(), readData.getTransactionId());
         assertTrue(Arrays.equals(readData.getData(), data));

         // Test rollback of a transaction
         writeController = gridController.getWriteController();
         writeController.write(new DataSource(name, writeController.getTransactionId(), data2));
         writeController.rollback();

         readData = readController.read(name);
         assertTrue(!data.equals(readData.getData()));
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         fail(ex.getMessage());
      }
   }

   @Test
   public void testReadPerformance() throws Exception
   {
      testReadPerformance(new SimplePerformanceTester());
   }

   @Test
   public void testThreadedReadPerformance() throws Exception
   {
      testReadPerformance(new ThreadedPerformanceTester(PERFORMANCE_TEST_THREAD_POOL_SIZE));
   }

   public void testReadPerformance(PerformanceTester performanceTester) throws Exception
   {
      // TODO: Modify this test to test reading from a large number of different
      // sources

      WriteController writeController = gridController.getWriteController();
      DataSource dataSource =
            new DataSource(new SourceName("0"), writeController.getTransactionId(), data);
      try
      {
         writeController.write(dataSource);
         writeController.commit();
      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }

      final int workUnits = DATA_SIZE;
      final int iterations = 15;

      WorkItemProvider workItemProvider = new WorkItemProvider()
      {
         public WorkItem createWorkItem(long id) throws CloneNotSupportedException,
               ControllerException
         {
            return new IteratedWorkItem(id, workUnits, iterations)
            {
               ReadController readController = gridController.getReadController();

               public void performIteration() throws Exception
               {
                  readController.read(new SourceName("0"));
               }
            };
         }
      };

      PerformanceTestResult result =
            performanceTester.run(workItemProvider, PERFORMANCE_TEST_DATA_SIZE);

      System.out.println("Read rate is: " + result.getWorkUnitsPerSecond() + " KB/s ");

      if (result.getWorkUnitsPerSecond() < MIN_READ_RATE)
      {
         fail("The read rate " + result.getWorkUnitsPerSecond() + " was less than the minimum");
      }
   }

   private static void randomizeBuffer(byte data[])
   {
      for (int idx = 0; idx < data.length; idx++)
      {
         data[idx] = (byte) (Math.random() * 256);
      }
   }

   private Vault getVault() throws Exception
   {
      // Initialize slice storage
      File gridBase = new File(TEST_OUTPUT_PATH);
      gridBase.mkdirs();

      VaultACL acl = createACL();

      List<SliceStore> sliceStores = new ArrayList<SliceStore>();
      for (int i = 0; i < GRID_PILLARS; ++i)
      {
         File ssPath = getSliceStoreDirectory(gridBase, i);

         try
         {

            // TODO: Replace with generic SliceStore mechanism
            SliceStore ss = null; //new BDBSliceStore(ssPath);
            ss.createStore("Dummy", -1, SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED, acl, null);

            // int port = 5000 + i + 1;
            // SliceStore ss = new RemoteSliceStore( i + 1, "127.0.0.1:" + port,
            // "", "", null );

            // SliceStore ss =
            // new BlockFileSliceStore(ssPath.getAbsolutePath() + "/vault.dat", 345, false);

            sliceStores.add(ss);
         }
         catch (Exception e)
         {
            sliceStores.add(null);
         }
      }

      // assert sliceStores.size() == GRID_PILLARS;

      // Initialize IDA
      InformationDispersalCodec ida =
            new CauchyInformationDispersalCodec(GRID_PILLARS, IDA_THRESHOLD);

      // Initialize vault
      class SimpleVault extends BaseVault
      {
         public SimpleVault(
               String type,
               InformationDispersalCodec idaCodec,
               List<SliceStore> sliceStores)
         {
            super(type, idaCodec, null, null, sliceStores);
         }

         public long getSize()
         {
            return -1;
         }

         public long getMaxSliceSize()
         {
            return -1;
         }

         public void optimizeVault()
         {
            // Does nothing
         }
      }
      return new SimpleVault("Dummy", ida, sliceStores);
   }

   private static VaultACL createACL() throws Exception
   {
      UUID vaultIdentifier = UUID.randomUUID();

      String vaultFileName = TEST_OUTPUT_PATH + vaultIdentifier.toString() + ".der";
      FileOutputStream out = new FileOutputStream(vaultFileName);

      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("vaultacltest");
      credentials.setPassword("password");

      KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
      keygen.initialize(512);
      KeyPair keypair = keygen.generateKeyPair();

      List<VaultKeyInfo> keyInfoList = new ArrayList<VaultKeyInfo>();

      VaultACLFactory fact = new VaultACLFactory();
      fact.create(out, vaultIdentifier, credentials, keypair.getPublic(), keyInfoList);
      out.close();

      FileInputStream in = new FileInputStream(vaultFileName);

      VaultACL vaultACL = fact.getInstance(in);
      in.close();

      return vaultACL;
   }

   public static boolean deleteDirectory(File dir)
   {
      if (dir.isDirectory())
      {
         String[] children = dir.list();
         for (int i = 0; i < children.length; i++)
         {
            boolean success = deleteDirectory(new File(dir, children[i]));

            if (!success)
            {
               return false;
            }
         }
      }

      // The directory is now empty so delete it
      return dir.delete();
   }

   private static File getSliceStoreDirectory(File gridBase, int sliceID)
   {
      return new File(gridBase, String.format("ss%d", sliceID));
   }
}
