//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Vance Thornton
//
//---------------------

package org.cleversafe.ida;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.ida.exceptions.IDAEncodeException;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.ida.exceptions.IDANotInitializedException;
import org.cleversafe.test.PerformanceTestResult;
import org.cleversafe.test.PerformanceTester;
import org.cleversafe.test.SimplePerformanceTester;
import org.cleversafe.test.WorkItem;
import org.cleversafe.test.WorkItemProvider;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.cleversafe.util.math.CombinationGenerator;
import org.junit.BeforeClass;
import org.junit.Test;

public class InformationDispersalPerformanceTest
{
   private static String[] referralsList = new String[]{
         "cauchy", "optimizedcauchy", "replicate"
   };
   private static String currentReferral = null;

   private static int DATA_SIZE = 4096;

   // @ConfigurableField(path = "org.cleversafe.ida.thread_pool_size")
   // private static int THREAD_POOL_SIZE = 15;

   private static int MBPS = (1024 * 1024) / 8;

   // The minimum encode rate is 100 megabits / second
   //   @ConfigurableField(path = "org.cleversafe.ida.min_encode_rate")
   //   private static float MIN_ENCODE_RATE_MBPS = 10; // MBPS;

   // The minimum decode and rate is 100 megabits / second
   //   @ConfigurableField(path = "org.cleversafe.ida.min_decode_rate")
   //   private static float MIN_DECODE_RATE_MBPS = 10; // MBPS;

   // Data size for performance tests is 10 megabytes
   private static long PERFORMANCE_TEST_DATA_SIZE = 10 * 1024 * 1024;

   static int threadPoolSizes[] = {
         2, 5, 10, 20
   };
   final int optimalShares[] = {
         0, 1, 2, 3, 4, 5, 6, 7, 9, 10, 11, 12
   };

   final int missingShares[] = {
         0, 2, 3, 5, 6, 7, 9, 11, 12, 13, 14, 15
   };

   static abstract private class TestWorker
   {
      protected PerformanceTester tester;

      TestWorker(PerformanceTester tester)
      {
         this.tester = tester;
      }

      abstract PerformanceTestResult run() throws Exception;
   }

   @BeforeClass
   public static void initializeTests()
   {
      int sizeInMB = Integer.getInteger("test.size", 10);
      PERFORMANCE_TEST_DATA_SIZE = sizeInMB * 1024 * 1024;
      System.out.println("Test size " + sizeInMB + "MB");

      String concurrencyStr = System.getProperty("test.concurrency");
      if (concurrencyStr != null)
      {
         String[] values = concurrencyStr.split(" ");
         threadPoolSizes = new int[values.length];
         for (int i = 0; i < values.length; i++)
         {
            threadPoolSizes[i] = Integer.parseInt(values[i]);
         }
      }
   }

   // @Test
   public void testEncodeDecode() throws Exception
   {
      // Generate a random 4KB data buffer
      byte data1[] = new byte[DATA_SIZE];
      randomizeBuffer(data1);

      byte data2[] = new byte[DATA_SIZE * 2];
      randomizeBuffer(data2);

      byte data3[] = new byte[DATA_SIZE + 71];
      randomizeBuffer(data3);

      byte data4[] = new byte[DATA_SIZE - 71];
      randomizeBuffer(data4);

      byte data5[] = new byte[0];
      randomizeBuffer(data5);

      byte data6[] = new byte[4128];
      randomizeBuffer(data6);

      for (String referral : referralsList)
      {
         System.out.println("Testing encode/decode for referral " + referral + "... ");

         InformationDispersalCodec ida =
               (InformationDispersalCodec) ConfigurationFactory.getBindingsProvider(
                     ConfigurationFactory.XML_CONFIG_TYPE).getImplementation("IDA", referral);

         // Execute encode / decode operation
         System.out.print(".");
         System.out.flush();
         executeEncodeDecode(data1, ida.getEncoder(), ida.getDecoder());
         System.out.print(".");
         System.out.flush();
         executeEncodeDecode(data2, ida.getEncoder(), ida.getDecoder());
         System.out.print(".");
         System.out.flush();
         executeEncodeDecode(data3, ida.getEncoder(), ida.getDecoder());
         System.out.print(".");
         System.out.flush();
         executeEncodeDecode(data4, ida.getEncoder(), ida.getDecoder());
         System.out.print(".");
         System.out.flush();
         executeEncodeDecode(data5, ida.getEncoder(), ida.getDecoder());
         System.out.print(".");
         System.out.flush();
         executeEncodeDecode(data6, ida.getEncoder(), ida.getDecoder());
         System.out.println(" Done.");
      }
      System.out.println("");
   }

   @Test
   public void testEncodePerformance() throws Exception
   {
      for (String referral : referralsList)
      {
         currentReferral = referral;

         System.out.println("Testing encode performance for referral " + referral + "... ");
         try
         {
            PerformanceTestResult result = testEncodePerformance(new SimplePerformanceTester());
            System.out.println((result.getWorkUnitsPerSecond() / MBPS) + " mbps");
         }
         catch (Exception e)
         {
            fail(e.getMessage());
         }

      }
      System.out.println("");
   }

   @Test
   public void testThreadedEncodePerformance() throws Exception
   {
      class Worker extends TestWorker
      {
         public Worker(PerformanceTester tester)
         {
            super(tester);
         }

         @Override
         public PerformanceTestResult run() throws Exception
         {
            return testEncodePerformance(this.tester);
         }
      }
      System.out.println("Multi-threaded encode All missing");
      testThreadedPerformance(new Worker(new SimplePerformanceTester()));
   }

   @Test
   public void testDecodePerformanceNoneMissing() throws Exception
   {
      for (String referral : referralsList)
      {
         currentReferral = referral;

         System.out.println("Testing decode (none missing) performance for referral " + referral
               + "... ");

         PerformanceTestResult result =
               testDecodePerformance(new SimplePerformanceTester(), this.optimalShares);
         System.out.println((result.getWorkUnitsPerSecond() / MBPS) + " mbps");
      }
      System.out.println("");
   }

   @Test
   public void testThreadedDecodePerformanceNoneMissing() throws Exception
   {
      class Worker extends TestWorker
      {
         public Worker(PerformanceTester tester)
         {
            super(tester);
         }

         @Override
         public PerformanceTestResult run() throws Exception
         {
            return testDecodePerformance(this.tester,
                  InformationDispersalPerformanceTest.this.optimalShares);
         }
      }
      System.out.println("Multi-threaded decode (none missing)");
      testThreadedPerformance(new Worker(new SimplePerformanceTester()));
   }

   @Test
   public void testDecodePerformanceAllMissing() throws Exception
   {
      for (String referral : referralsList)
      {
         currentReferral = referral;

         System.out.println("Testing decode (all missing) performance for referral " + referral
               + "... ");

         PerformanceTestResult result =
               testDecodePerformance(new SimplePerformanceTester(), this.missingShares);
         System.out.println((result.getWorkUnitsPerSecond() / MBPS) + " mbps");
      }
      System.out.println("");
   }

   @Test
   public void testThreadedDecodePerformanceAllMissing() throws Exception
   {
      class Worker extends TestWorker
      {
         public Worker(PerformanceTester tester)
         {
            super(tester);
         }

         @Override
         public PerformanceTestResult run() throws Exception
         {
            return testDecodePerformance(this.tester,
                  InformationDispersalPerformanceTest.this.missingShares);
         }
      }
      System.out.println("Multi-threaded decode (all missing");
      testThreadedPerformance(new Worker(new SimplePerformanceTester()));
   }

   private PerformanceTestResult testEncodePerformance(PerformanceTester tester) throws Exception
   {
      // Generate a random 4KB data buffer
      final byte data[] = new byte[DATA_SIZE];
      randomizeBuffer(data);

      InformationDispersalCodec idc = null;
      try
      {
         idc =
               (InformationDispersalCodec) ConfigurationFactory.getBindingsProvider(
                     ConfigurationFactory.XML_CONFIG_TYPE).getImplementation("IDA", currentReferral);
      }
      catch (ConfigurationItemNotDefinedException e)
      {
         fail(e.getMessage());
      }
      catch (ObjectInstantiationException e)
      {
         fail(e.getMessage());
      }
      catch (ObjectInitializationException e)
      {
         fail(e.getMessage());
      }
      assertNotNull(idc);

      final InformationDispersalCodec finalIDC = idc;

      WorkItemProvider workItemProvider = new WorkItemProvider()
      {
         public WorkItem createWorkItem(long id) throws Exception
         {
            return new WorkItem(id, DATA_SIZE)
            {
               @Override
               public void run()
               {
                  try
                  {
                     InformationDispersalEncoder encoder = finalIDC.getEncoder();
                     encoder.finish(data);
                  }
                  catch (Exception e)
                  {
                     fail(e.getMessage());
                  }
               }
            };
         }
      };
      return tester.run(workItemProvider, PERFORMANCE_TEST_DATA_SIZE);
   }

   private PerformanceTestResult testDecodePerformance(
         PerformanceTester tester,
         int availableShares[]) throws Exception
   {
      // Generate a random 4KB data buffer
      final byte data[] = new byte[DATA_SIZE];
      randomizeBuffer(data);

      // Build a list of available shares
      final List<byte[]> input = new ArrayList<byte[]>();

      final int finalAvailableShares[] = availableShares;

      InformationDispersalCodec idc = null;
      try
      {
         idc =
               (InformationDispersalCodec) ConfigurationFactory.getBindingsProvider(
                     ConfigurationFactory.XML_CONFIG_TYPE).getImplementation("IDA", currentReferral);
      }
      catch (ConfigurationItemNotDefinedException e)
      {
         fail(e.getMessage());
      }
      catch (ObjectInstantiationException e)
      {
         fail(e.getMessage());
      }
      catch (ObjectInitializationException e)
      {
         fail(e.getMessage());
      }
      assertNotNull(idc);

      List<byte[]> encodedData = null;

      InformationDispersalEncoder encoder = idc.getEncoder();

      try
      {
         // Encode the message
         encodedData = encoder.finish(data);
      }
      catch (IDAEncodeException e)
      {
         fail(e.getMessage());
      }
      catch (IDANotInitializedException e)
      {
         fail(e.getMessage());
      }

      final List<byte[]> finalEncodedData = encodedData;

      final InformationDispersalCodec finalIDC = idc;

      WorkItemProvider workItemProvider = new WorkItemProvider()
      {
         public WorkItem createWorkItem(long id) throws Exception
         {
            return new WorkItem(id, DATA_SIZE)
            {
               @Override
               public void run()
               {
                  input.clear();
                  for (int idx = 0; idx < finalAvailableShares.length; idx++)
                  {
                     input.add(finalEncodedData.get(finalAvailableShares[idx]));
                  }

                  try
                  {
                     InformationDispersalDecoder decoder = finalIDC.getDecoder();
                     decoder.finish(input);
                  }
                  catch (Exception e)
                  {
                     fail(e.getMessage());
                  }
               }
            };
         }
      };

      return tester.run(workItemProvider, PERFORMANCE_TEST_DATA_SIZE);
   }

   private void testThreadedPerformance(final TestWorker worker) throws Exception
   {

      for (String referral : referralsList)
      {
         currentReferral = referral;
         for (int i = 0; i < threadPoolSizes.length; i++)
         {
            System.out.println("Testing threaded (" + threadPoolSizes[i]
                  + ")  performance for referral " + referral + "... ");

            ExecutorService executor =
                  new BoundedThreadPoolExecutor("performance", threadPoolSizes[i]);
            final PerformanceTestResult result = new PerformanceTestResult(0, 0);
            for (int j = 0; j < threadPoolSizes[i]; j++)
            {
               executor.execute(new Thread()
               {
                  @Override
                  public void run()
                  {
                     try
                     {
                        PerformanceTestResult r = worker.run();
                        result.add(r);
                     }
                     catch (Exception e)
                     {
                        // TODO Auto-generated catch block
                        e.printStackTrace();
                     }
                  }
               });
            }
            executor.shutdown();
            executor.awaitTermination(60, TimeUnit.SECONDS);
            double throughput = result.getWorkUnitsPerSecond() / MBPS;
            System.out.println("Percieved:" + throughput + " mbps" + " Total:" + throughput
                  * threadPoolSizes[i] + " mbps");
         }
      }
      System.out.println("");

   }

   private void executeEncodeDecode(
         byte data[],
         InformationDispersalEncoder encoder,
         InformationDispersalDecoder decoder) throws Exception
   {
      // Encode the message
      List<byte[]> encodedData = encoder.finish(data);

      // For each supported outage case
      CombinationGenerator combinationGenerator =
            new CombinationGenerator(encoder.getNumSlices(), encoder.getThreshold());

      while (combinationGenerator.hasMore())
      {
         // Build a list of availble shares
         List<byte[]> input = new ArrayList<byte[]>();

         int availableShares[] = combinationGenerator.getNext();

         for (int idx = 0; idx < availableShares.length; idx++)
         {
            input.add(encodedData.get(availableShares[idx]));
         }

         // Decode the data
         byte decodedData[] = decoder.finish(input);

         assertEquals(data.length, decodedData.length);

         // Compare the decoded data to the original
         for (int idx = 0; idx < data.length; idx++)
         {
            assertEquals(data[idx], decodedData[idx]);
         }
      }
   }

   private void randomizeBuffer(byte buffer[])
   {
      for (int idx = 0; idx < buffer.length; idx++)
      {
         byte value = (byte) (Math.random() * 256);
         buffer[idx] = value;
      }
   }

}
