//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: May 9, 2008
//---------------------

package org.cleversafe.layer.grid;

import static org.junit.Assert.fail;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;
import org.cleversafe.config.BindingsProvider;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.layer.grid.exceptions.ControllerException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.unreliable.UnreliableSliceStore;
import org.cleversafe.server.exceptions.ServerConfigurationLoadException;
import org.cleversafe.storage.ss.SliceServerConfiguration;
import org.cleversafe.storage.ss.SliceServerDaemon;
import org.cleversafe.storage.ss.configuration.ConfigurationLoader;
import org.cleversafe.storage.ss.configuration.XMLConfigurationLoader;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.cleversafe.vault.BaseVault;
import org.cleversafe.vault.Vault;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

public class ControllerPerformanceTest extends GridBaseTest
{
   private static Logger _logger = Logger.getLogger(ControllerPerformanceTest.class);

   public static final boolean CSV = true; // CSV output
   public static final boolean VERBOSE = false; // Verbose per-task logging

   public static final int NUM_SOURCES = 128; // Sources per request
   public static final int SOURCE_SIZE = 4096; // Size of each source in bytes
   public static final int NUM_REQUESTS = 1024; // Number of requests per test
   public static final int TOTAL_TRANSFER = NUM_SOURCES * SOURCE_SIZE * NUM_REQUESTS;

   /**
    * BEGIN SliceServer INITIALIZATION
    * FIXME: This will not be necessary post-1.0
    */

   private static SliceServerDaemon[] ssSet = null;
   private static Thread[] daemonThreads = null;

   @BeforeClass
   public static void runBeforeTests() throws Exception
   {
      // Start remote stores if necessary
      //String storeString = System.getProperty("org.cleversafe.storage.ss.daemons");
      //int numStores = storeString != null ? Integer.parseInt(storeString) : 0;
      int numStores = 1;

      if (numStores > 0)
      {
         // Check required system properties
         String serverConfiguration =
               System.getProperty("org.cleversafe.storage.ss.xml.configuration");
         String bindingsConfiguration = System.getProperty("org.cleversafe.xml.configuration");

         if (serverConfiguration == null)
         {
            fail("Server configuration file not specified");
         }

         if (bindingsConfiguration == null)
         {
            fail("Bindings configuration file not specified");
         }

         // Launch daemon
         try
         {
            // Create daemons with generated configurations
            ssSet = new SliceServerDaemon[numStores];
            for (int daemonID = 1; daemonID <= ssSet.length; daemonID++)
            {
               Properties properties = new Properties();
               properties.setProperty("DAEMON.ID", Integer.toString(daemonID));

               ConfigurationLoader configLoader = new XMLConfigurationLoader(serverConfiguration);
               SliceServerConfiguration configuration = configLoader.getConfiguration(properties);

               ssSet[daemonID - 1] = new SliceServerDaemon(configuration);
            }

            // Create threads for non-primary daemons
            daemonThreads = new Thread[ssSet.length];
            for (int a = 0; a < ssSet.length; a++)
            {
               daemonThreads[a] = new Thread(ssSet[a], "daemon-" + (a + 1));
               daemonThreads[a].start();
            }

            // Wait until servers have started
            for (SliceServerDaemon daemon : ssSet)
            {
               try
               {
                  daemon.awaitStart();
               }
               catch (InterruptedException ignored)
               {
               }
            }
         }
         catch (ServerConfigurationLoadException ex)
         {
            _logger.fatal("Unable to load configuration", ex);
            fail(ex.getMessage());
         }
      }

      // Wait until the grid comes up....
      Thread.sleep(5000);
   }

   @AfterClass
   public static void runAfterTests() throws Exception
   {
      if (ssSet != null)
      {
         // Shutdown SliceServers
         for (SliceServerDaemon ss : ssSet)
         {
            ss.shutdown();
         }

         // Join Threads
         for (int a = 0; a < daemonThreads.length; a++)
         {
            daemonThreads[a].join();
         }
      }
   }

   /**
    * END SliceServer INITIALIZATION
    * FIXME: This will not be necessary post-1.0
    */

   @Test
   public void testCommonScenarios()
   {
      VaultType[] vaultTypes = new VaultType[]{
         VaultType.REMOTE_LOCAL_1X,
         //VaultType.REMOTE_LOCAL,
         //VaultType.BLOCKMULTIFILE
         //VaultType.MEMORY
         };
      int[] bandwidths = new int[]{
         0 /* No delay */,
      //1024 * 1024 * 1024
            /* 1Gbps */
            //, 100 * 1024 * 1024
            /* 100Mbps */};
      int[] rtts = new int[]{
         0, //20, 100
         };
      int minThreads = 32;
      int maxThreads = 32;

      for (VaultType vaultType : vaultTypes)
      {
         System.out.println("Starting vault type: " + vaultType);
         for (int bw : bandwidths)
         {
            for (int rtt : rtts)
            {
               if (!CSV)
               {
                  System.out.println(String.format("RUNNING CONFIG: %s, %dMbps, %dms RTT",
                        vaultType, bw / (1024 * 1024), rtt));
               }

               for (int threads = minThreads; threads <= maxThreads; threads *= 2)
               {
                  MockNetworkVault vault = new MockNetworkVault(vaultType, rtt, bw);

                  double writePerf = testWrite(vault, threads);
                  double readPerf = 0; //testRead(vault, threads);
                  if (CSV)
                  {
                     System.out.println(String.format("%d,%d,%d,%.3f,%.3f", bw / (1024 * 1024),
                           rtt, threads, writePerf, readPerf));
                  }

                  this.cleanupVault(vault);
                  vault = null;
                  System.gc();
               }
            }
         }
      }
   }

   /**
    * Perform a multi-threaded write
    * @param vault
    * @param numThreads
    * @return MB/s for write
    */
   private double testWrite(MockNetworkVault vault, int numThreads)
   {
      ExecutorService executor = new BoundedThreadPoolExecutor("testWrite", numThreads);
      final byte[] sourceData = new byte[SOURCE_SIZE];

      try
      {
         final GridController gridController = this.getGridController(vault);

         class WriteTask implements Callable<Void>
         {
            private int index;
            private List<DataSource> sources;
            private WriteController writeController;

            public WriteTask(int taskIndex, List<SourceName> sourceNames)
            {
               this.index = taskIndex;

               try
               {
                  this.writeController = gridController.getWriteController();
               }
               catch (ControllerException e)
               {
                  fail("Error creating controller");
               }

               this.sources = new ArrayList<DataSource>(sourceNames.size());
               for (SourceName name : sourceNames)
               {
                  sources.add(new DataSource(name, this.writeController.getTransactionId(),
                        sourceData));
               }
            }

            public Void call() throws Exception
            {
               long beginTime = System.currentTimeMillis();
               this.writeController.write(this.sources);
               this.writeController.commit();

               long totalTime = System.currentTimeMillis() - beginTime;
               double mBytesPerSec = NUM_SOURCES * SOURCE_SIZE / (totalTime * 1.e6 / 1.e3);
               if (VERBOSE)
               {
                  _logger.info(String.format("Write<%d>: %.3fMB, %.3fs, %.3fMB/s", this.index,
                        NUM_SOURCES * SOURCE_SIZE / 1.e6, totalTime / 1000., mBytesPerSec));
               }

               return null;
            }
         }

         // Initialize tasks
         List<Callable<Void>> tasks = new ArrayList<Callable<Void>>(NUM_REQUESTS);
         for (int req = 0; req < NUM_REQUESTS; ++req)
         {
            List<SourceName> sourceNames = new ArrayList<SourceName>(NUM_SOURCES);
            for (int i = 0; i < NUM_SOURCES; ++i)
            {
               sourceNames.add(new SourceName(Integer.toString(req * NUM_SOURCES + i)));
            }

            tasks.add(new WriteTask(req + 1, sourceNames));
         }

         // Do writes
         long beginTime = System.currentTimeMillis();
         List<Future<Void>> futures = executor.invokeAll(tasks);
         long totalTime = System.currentTimeMillis() - beginTime;

         // Check time
         double mBytesPerSec = TOTAL_TRANSFER / (totalTime * 1.e6 / 1.e3);
         if (!CSV)
         {
            System.out.println(String.format("TOTAL Write: %.3fMB, %d threads, %.3fMB/s",
                  TOTAL_TRANSFER / 1.e6, numThreads, mBytesPerSec));
         }

         // Check for errors
         for (Future<Void> future : futures)
         {
            future.get();
         }

         return mBytesPerSec;
      }
      catch (Exception ex)
      {
         _logger.error(ex);
         fail("Unexpected failure");
         return 0;
      }
   }

   /**
    * Perform a multi-threaded read
    * @param vault
    * @param numThreads
    * @return MB/s for read
    */
   private double testRead(MockNetworkVault vault, int numThreads)
   {
      ExecutorService executor = new BoundedThreadPoolExecutor("testRead", numThreads);

      try
      {
         final GridController gridController = this.getGridController(vault);

         class ReadTask implements Callable<List<DataSource>>
         {
            private int index;
            private List<SourceName> sourceNames;

            public ReadTask(int taskIndex, List<SourceName> sourceNames)
            {
               this.index = taskIndex;
               this.sourceNames = sourceNames;
            }

            public List<DataSource> call() throws Exception
            {
               ReadController rc = gridController.getReadController();

               long beginTime = System.currentTimeMillis();
               List<DataSource> sources = rc.read(this.sourceNames);

               long totalTime = System.currentTimeMillis() - beginTime;
               double mBytesPerSec = NUM_SOURCES * SOURCE_SIZE / (totalTime * 1.e6 / 1.e3);
               if (VERBOSE)
               {
                  _logger.info(String.format("Read<%d>: %.3fMB, %.3fs, %.3fMB/s", this.index,
                        NUM_SOURCES * SOURCE_SIZE / 1.e6, totalTime / 1000., mBytesPerSec));
               }

               return sources;
            }
         }

         // Initialize tasks
         List<Callable<List<DataSource>>> tasks =
               new ArrayList<Callable<List<DataSource>>>(NUM_REQUESTS);
         for (int req = 0; req < NUM_REQUESTS; ++req)
         {
            List<SourceName> sourceNames = new ArrayList<SourceName>(NUM_SOURCES);
            for (int i = 0; i < NUM_SOURCES; ++i)
            {
               sourceNames.add(new SourceName(Integer.toString(req * NUM_SOURCES + i)));
            }

            tasks.add(new ReadTask(req + 1, sourceNames));
         }

         // Do reads
         long beginTime = System.currentTimeMillis();
         List<Future<List<DataSource>>> futures = executor.invokeAll(tasks);
         long totalTime = System.currentTimeMillis() - beginTime;

         // Check time
         double mBytesPerSec = TOTAL_TRANSFER / (totalTime * 1.e6 / 1.e3);
         if (!CSV)
         {
            System.out.println(String.format("TOTAL Read: %.3fMB, %d threads, %.3fMB/s",
                  TOTAL_TRANSFER / 1.e6, numThreads, mBytesPerSec));
         }

         // Check for errors
         for (Future<List<DataSource>> future : futures)
         {
            future.get();
         }

         return mBytesPerSec;
      }
      catch (Exception ex)
      {
         _logger.error(ex);
         fail("Unexpected failure");
         return 0;
      }
   }

   /**
    * Constructs a GridController from the current bindings configuration for the given vault
    * @param vault
    * @throws ConfigurationLoadException 
    * @throws ObjectInitializationException 
    * @throws ObjectInstantiationException 
    * @throws ConfigurationItemNotDefinedException 
    */
   private GridController getGridController(Vault vault) throws ConfigurationLoadException,
         ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException
   {
      // Construct controller
      BindingsProvider bindingsProvider =
            ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE);
      ReadControllerFactory rfc =
            bindingsProvider.getDefaultImplementation(ReadControllerFactory.class);
      WriteControllerFactory wfc =
            bindingsProvider.getDefaultImplementation(WriteControllerFactory.class);
      return new GridController(vault, wfc, rfc);
   }

   /**
    * Simple mock vault
    */
   public enum VaultType
   {
      REMOTE_LOCAL("remote-local-vault-descriptor.xml"), // Remote-Local 
      REMOTE_LOCAL_1X("remote-local-vault-descriptor-1store.xml"), // Remote-Local w/1 store
      BLOCKMULTIFILE("local-multi-file-vault-descriptor.xml"), // Local BlockMultiFile
      MEMORY("local-memory-vault-descriptor.xml"); // Local Memory

      private String path;

      private VaultType(String path)
      {
         this.path = path;
      }

      public String getVaultDescriptorName()
      {
         return this.path;
      }
   };
   private class MockNetworkVault extends BaseVault
   {
      private int latency;
      private long bandwidth;

      public MockNetworkVault(VaultType vaultType, int latency, long bandwidth)
      {
         super("mock");

         this.latency = latency;
         this.bandwidth = bandwidth;

         try
         {
            vaultDescriptor = File.separator + vaultType.getVaultDescriptorName();
            Vault vault = createVault();

            List<SliceStore> stores = new ArrayList<SliceStore>(vault.getWidth());
            for (SliceStore innerStore : vault.getSliceStores())
            {
               UnreliableSliceStore store = new UnreliableSliceStore(innerStore);

               store.setLatency(latency);
               store.setBandwidth(bandwidth);

               stores.add(store);
            }

            this.setInformationDispersalCodec(vault.getInformationDispersalCodec());
            this.setDatasourceCodecs(vault.getDatasourceCodecs());
            this.setSliceCodecs(vault.getSliceCodecs());

            // Store sessions have already been started
            this.setSliceStores(stores);

            this.initialize();
         }
         catch (Exception ex)
         {
            _logger.error(ex);
            fail("Error creating vault");
         }
      }

      public long getBandwidth()
      {
         return this.bandwidth;
      }

      public int getLatency()
      {
         return this.latency;
      }

      public long getMaxSliceSize()
      {
         return 0;
      }

      public long getSize()
      {
         return 0;
      }

      public void optimizeVault()
      {
      }

      public String toString()
      {
         return String.format("n=%d, t=%d, w=%d, r=%d, latency=%d, bandwidth=%d",
               this.idaCodec.getNumSlices(), this.idaCodec.getThreshold(),
               this.getWriteThreshold(), this.getReadThreshold(), this.latency, this.bandwidth);
      }
   }
}
