//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: Jason
//
// Date: Sep 27, 2007
//---------------------

package org.cleversafe.layer.network;

import org.cleversafe.layer.communication.network.mina.MinaConnector;
import org.cleversafe.layer.communication.policy.SimpleConnectorManager;
import org.cleversafe.layer.protocol.NoopRequest;
import org.cleversafe.serialization.asn1.ASN1GridProtocolMessageFactory;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * This test calculates the amount of data that could be theoretically written to a remote server
 * using the current networking framework. This test simulates the serialization, threading, server
 * handlers, and optionally latency in sending a begin transaction, write, and commit transaction
 * protocol message.
 */
public class NetworkFrameworkThroughputUnitTest
{
   // Option controls whether or not the connector should use SSL
   private static boolean enableSSL = false;

   // Option controls whether this test should start the acceptor or another
   // process, potentially running on a different machine, should listen
   private static boolean startAcceptor = true;

   // Option specifies the host address to connect to, should not be changed unless
   // the acceptor is running on a different machine
   private static String remoteHost = "localhost";

   // The port on which the acceptor should listen, if startAcceptor is true and
   // the port that the connector will attempt to connect to on the remoteHost
   private static int connectPort = 62817;

   // When true, the test will await a response before sending the next message
   // FIXME: When asynchronous message transfers are possible then use this
   private static boolean sendSynchronous = true;

   // The size of the message that will be sent with each call to send
   // private static int messageLength = 128*4096;
   // private static int messageLength = 128*353;
   // private static int messageLength = 60000;
   private static int messageLength = 15000;

   // Programmable Latency in milliseconds, which can be used to simulate the
   // roundtrip time of a request to a response
   private static long addedLatency = 0;

   // The listening NoopServer, if startAcceptor is true
   private NoopServer server = null;

   // The mina connector
   private MinaConnector connector = null;

   // Noop message with an empty payload
   private NoopRequest emptyRequest = null;

   // Noop message with a payload
   private NoopRequest payloadRequest = null;

   // Test length in milliseconds
   private long testTime = 10 * 1000;

   @BeforeClass
   public static void runBeforeTests() throws Exception
   {
      if (System.getProperties().containsKey("test.enablessl"))
      {
         enableSSL = Boolean.getBoolean("test.enablessl");
      }
      if (System.getProperties().containsKey("test.startacceptor"))
      {
         startAcceptor = Boolean.getBoolean("test.startacceptor");
      }
      if (System.getProperties().containsKey("test.remotehost"))
      {
         remoteHost = System.getProperty("test.remotehost");
      }
      if (System.getProperties().containsKey("test.connectport"))
      {
         connectPort = Integer.getInteger("test.connectport");
      }
      if (System.getProperties().containsKey("test.sendsynchronous"))
      {
         sendSynchronous = Boolean.getBoolean("test.sendsynchronous");
      }
      if (System.getProperties().containsKey("test.messagelength"))
      {
         messageLength = Integer.getInteger("test.messagelength");
      }
      if (System.getProperties().containsKey("test.addedlatency"))
      {
         addedLatency = Integer.getInteger("test.addedlatency");
      }
   }

   @Before
   public void setUp() throws Exception
   {
      if (startAcceptor == true)
      {
         startListener();
      }
   }

   @After
   public void tearDown() throws Exception
   {
      if (startAcceptor == true)
      {
         if (this.server != null)
         {
            this.server.stop();
         }
      }
   }

   @Test
   public void testNetworkThroughput() throws Exception
   {
      long dataSent = 0;

      System.out.println("Testing amount of data that can be transferred in " + (testTime / 1000)
            + " seconds...");

      connect();

      emptyRequest = new NoopRequest();
      payloadRequest = new NoopRequest();
      payloadRequest.setPayload(new byte[messageLength]);

      // Conduct Network Throughput Test
      long startTime = System.currentTimeMillis();
      while ((System.currentTimeMillis() - startTime) < testTime)
      {
         sendMessages();

         dataSent += messageLength;
      }

      float throughputInBpms = (dataSent / testTime);
      float throughputInBps = throughputInBpms * 1000;
      float throughputInMBps = throughputInBps / (1024 * 1024);
      float throughputInMbps = throughputInMBps * 8;

      System.out.println("Throughput in Mbps: " + throughputInMbps);
   }

   private void startListener() throws Exception
   {
      this.server = new NoopServer(connectPort, enableSSL, new ASN1GridProtocolMessageFactory());

      this.server.start();

   }

   private void connect() throws Exception
   {
      connector = new MinaConnector(remoteHost, connectPort, new SimpleConnectorManager(), new ASN1GridProtocolMessageFactory());
      connector.connect();
   }

   private void sendMessages() throws Exception
   {
      connector.exchange(emptyRequest); // Simulated Begin Transaction Request
      if (addedLatency > 0)
         Thread.sleep(addedLatency);

      connector.exchange(payloadRequest); // Simulated Write Request
      if (addedLatency > 0)
         Thread.sleep(addedLatency);

      connector.exchange(emptyRequest); // Simulated Commit Transaction Request
      if (addedLatency > 0)
         Thread.sleep(addedLatency);
   }

}
