//
//Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
//Cleversafe Dispersed Storage(TM) is software for secure, private and
//reliable storage of the world's data using information dispersal.
//
//Copyright (C) 2005-2007 Cleversafe, Inc.
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
//USA.
//
//Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
//Chicago IL 60616
//email licensing@cleversafe.org
//
//END-OF-HEADER
//-----------------------
//@author: Jason
//
//Date: Sep 27, 2007
//---------------------

package org.cleversafe.layer.network;

import static org.junit.Assert.fail;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ServerSocket;
import java.net.Socket;

import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.exceptions.CommunicationTransmissionException;
import org.cleversafe.layer.communication.network.mina.MinaConnector;
import org.cleversafe.layer.communication.policy.SimpleConnectorManager;
import org.cleversafe.layer.protocol.NoopRequest;
import org.cleversafe.serialization.asn1.ASN1GridProtocolMessageFactory;
import org.cleversafe.server.exceptions.ServerIOException;
import org.cleversafe.server.exceptions.ServerSecurityException;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class NetworkLatencyTest
{

   // Option controls whether this test should start the acceptor or another
   // process, potentially running on a different machine, should listen
   private static boolean startAcceptor = true;

   // Option specifies the host address to connect to, should not be changed unless
   // the acceptor is running on a different machine
   private static String remoteHost = "localhost";

   // The port on which the acceptor should listen, if startAcceptor is true and
   // the port that the connector will attempt to connect to on the remoteHost
   private static int connectPort = 62817;

   // When true, the test will await a response before sending the next message
   // FIXME: When asynchronous message transfers are possible then use this
   private static boolean sendSynchronous = true;

   // The size of the message that will be sent with each call to send
   private static int messageLength = 128 * 4096;
   // private static int messageLength = 60000;
   // private static int messageLength = 15000;

   // Programmable Latency in milliseconds, which can be used to simulate the
   // roundtrip time of a request to a response
   private static long addedLatency = 0;

   // Specifies whether Mina or a raw socket should be used to test
   // the throughput of the network
   private static boolean useMina = true;

   // The listening NoopServer, if startAcceptor is true
   private NoopServer server = null;

   // The mina connector
   private MinaConnector connector = null;

   // The java server's socket
   private Socket serverSocket = null;

   // The java client's socket
   private Socket clientSocket = null;

   // The java socket listener
   private ServerSocket listenSocket = null;

   // Noop message with an empty payload
   private NoopRequest emptyRequest = null;

   // Empty payload message bytes
   private byte[] emptyRequestBytes = null;

   // Noop message with a payload
   private NoopRequest payloadRequest = null;

   // Payload message bytes
   private byte[] payloadRequestBytes = null;

   // Test length in milliseconds
   private long testTime = 30 * 1000;

   private static int testIterations = 100;

   @BeforeClass
   public static void runBeforeTests() throws Exception
   {
      if (System.getProperties().containsKey("test.startacceptor"))
      {
         startAcceptor = Boolean.getBoolean("test.startacceptor");
      }
      if (System.getProperties().containsKey("test.remotehost"))
      {
         remoteHost = System.getProperty("test.remotehost");
      }
      if (System.getProperties().containsKey("test.connectport"))
      {
         connectPort = Integer.getInteger("test.connectport");
      }
      if (System.getProperties().containsKey("test.sendsynchronous"))
      {
         sendSynchronous = Boolean.getBoolean("test.sendsynchronous");
      }
      if (System.getProperties().containsKey("test.messagelength"))
      {
         messageLength = Integer.getInteger("test.messagelength");
      }
      if (System.getProperties().containsKey("test.addedlatency"))
      {
         addedLatency = Integer.getInteger("test.addedlatency");
      }
      if (System.getProperties().containsKey("test.usemina"))
      {
         useMina = Boolean.getBoolean("test.usemina");
      }

      testIterations = Integer.getInteger("test.operations", 1000);
   }

   @Before
   public void setUp()
   {
      if (startAcceptor == true)
      {
         startListener();
      }
   }

   @After
   public void tearDown() throws Exception
   {
      if (startAcceptor == true)
      {
         if (this.server != null)
         {
            this.server.stop();
         }
      }
   }

   @Test
   public void testNetworkLatency()
   {
      System.out.println("Testing roundtrip latency for various size data sent.");
      connect();

      System.out.println("NumOps   bytes/msg    latency(ms)");
      for (int dataSize = 256; dataSize <= 4096; dataSize *= 2)
      {
         emptyRequest = new NoopRequest();
         payloadRequest = new NoopRequest();
         payloadRequest.setPayload(new byte[dataSize]);

         // Conduct Network latency Test
         long startTime = System.currentTimeMillis();
         for (int i = 0; i < testIterations; i++)
         {
            try
            {
               connector.exchange(payloadRequest); // Simulated Write Request
            }
            catch (CommunicationInterruptedException e)
            {
               e.printStackTrace();
               fail(e.getMessage());
            }
            catch (CommunicationResponseException e)
            {
               e.printStackTrace();
               fail(e.getMessage());
            }
            catch (CommunicationTransmissionException e)
            {
               e.printStackTrace();
               fail(e.getMessage());
            }
         }

         long endTime = System.currentTimeMillis();

         double latency = (endTime - startTime) * 1.0 / testIterations;

         System.out.println(testIterations + "      " + dataSize + "         " + latency);
      }
   }

   private void startListener()
   {
      if (useMina)
      {
         this.server = new NoopServer(connectPort, false, new ASN1GridProtocolMessageFactory());
         try
         {
            this.server.start();
         }
         catch (ServerIOException e)
         {
            e.printStackTrace();
            fail(e.getMessage());
         }
      }
      else
      {
         try
         {
            this.listenSocket = new ServerSocket(connectPort);
         }
         catch (IOException e)
         {
            e.printStackTrace();
            fail(e.getMessage());
         }
      }
   }

   private void connect()
   {
      if (useMina)
      {
         connector =
               new MinaConnector(remoteHost, connectPort, new SimpleConnectorManager(),
                     new ASN1GridProtocolMessageFactory());

         try
         {
            connector.connect();
         }
         catch (CommunicationIOException e)
         {
            e.printStackTrace();
            fail(e.getMessage());
         }
         catch (CommunicationConnectionException e)
         {
            e.printStackTrace();
            fail(e.getMessage());
         }
      }
      else
      {
         try
         {
            this.clientSocket = new Socket(remoteHost, connectPort);
            this.serverSocket = this.listenSocket.accept();
            Thread thread = new Thread(new ServerReader(this.serverSocket));
            thread.start();
         }
         catch (IOException e)
         {
            e.printStackTrace();
            fail(e.getMessage());
         }
      }
   }

   private void sendMessages() throws CommunicationInterruptedException,
         CommunicationResponseException, CommunicationTransmissionException, InterruptedException,
         IOException
   {
      if (useMina)
      {
         connector.exchange(emptyRequest); // Simulated Begin Transaction Request
         Thread.sleep(addedLatency);

         connector.exchange(payloadRequest); // Simulated Write Request
         Thread.sleep(addedLatency);

         connector.exchange(emptyRequest); // Simulated Commit Transaction Request
         Thread.sleep(addedLatency);
      }
      else
      {
         OutputStream out = this.clientSocket.getOutputStream();

         out.write(emptyRequestBytes);
         Thread.sleep(addedLatency);

         out.write(payloadRequestBytes);
         Thread.sleep(addedLatency);

         out.write(emptyRequestBytes);
         Thread.sleep(addedLatency);
      }
   }

   private class ServerReader implements Runnable
   {
      Socket serverSocket = null;

      public ServerReader(Socket serverSocket)
      {
         this.serverSocket = serverSocket;
      }

      public void run()
      {
         try
         {
            InputStream in = this.serverSocket.getInputStream();
            byte[] readBytes = new byte[4096];

            while (true)
            {
               in.read(readBytes);
            }

         }
         catch (IOException ex)
         {
         }
      }
   }
}
