
package org.cleversafe.layer.slicestore;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.GridBaseTest;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.util.Tuple3;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.junit.Test;

public class SliceStorePerformanceTest extends GridBaseTest
{
   private static Logger logger = Logger.getLogger(SliceStorePerformanceTest.class);

   private static final int SLICESTORE_IDX = 0;
   private static final int DEFAULT_DURATION = 15 * 1000;

   private Properties properties;

   private Random random = new Random(0x00000L);

   private abstract class IOTestTask implements Callable<Float>
   {
      protected int minIterations;
      protected int minDuration;

      // Variable parameters
      protected int requestsPerTransaction;
      protected int slicesPerRequest;
      protected int sliceSize;

      protected long maxSourceName;
      protected SliceStore sliceStore;

      protected byte[] data;

      private List<Tuple3<Long, Long, Long>> ioCompletionTimes;

      public void init(
            Vault vault,
            int minIterations,
            int minDuration,
            int requestsPerTransaction,
            int slicesPerRequest,
            int sliceSize) throws SliceStoreTransactionException, SliceStoreIOException,
            SliceStoreNotFoundException, IllegalSourceNameException, SliceStoreQuotaException
      {
         this.sliceStore = vault.getSliceStores().get(SLICESTORE_IDX);
         this.minIterations = minIterations;
         this.minDuration = minDuration;

         this.requestsPerTransaction = requestsPerTransaction;
         this.slicesPerRequest = slicesPerRequest;
         this.sliceSize = sliceSize;

         maxSourceName = vault.getSliceStoreSize(SLICESTORE_IDX) / vault.getMaxSliceSize();

         data = new byte[sliceSize];
      }

      public abstract String getType();

      public Float call() throws Exception
      {
         int iteration = 0;
         long startTime = System.currentTimeMillis();
         long bytesProcessed = 0;

         ioCompletionTimes = new ArrayList<Tuple3<Long, Long, Long>>();

         while (iteration < minIterations || (System.currentTimeMillis() - startTime < minDuration))
         {
            long transactionId = iteration++;

            long iterationStartTime = System.currentTimeMillis();

            SliceStoreTransaction transaction = sliceStore.createTransaction(transactionId);
            sliceStore.beginTransaction(transaction);

            for (int request = 0; request < requestsPerTransaction; request++)
            {
               List<Long> sliceNames = new ArrayList<Long>();
               for (int requestSource = 0; requestSource < slicesPerRequest; requestSource++)
               {
                  sliceNames.add(getSourceName(request, requestSource));
               }

               List<DataSlice> dataSlices = performIO(sliceNames, transactionId);

               for (DataSlice dataSlice : dataSlices)
               {
                  if (dataSlice.getData() != null)
                  {
                     bytesProcessed += dataSlice.getData().length;
                  }
               }
            }

            sliceStore.commitTransaction(transaction);

            long iterationDuration = (System.currentTimeMillis() - iterationStartTime);

            ioCompletionTimes.add(new Tuple3<Long, Long, Long>(iterationStartTime,
                  iterationDuration, bytesProcessed));
         }

         float duration = (System.currentTimeMillis() - startTime) / 1000.0f;

         return (duration > 0) ? bytesProcessed / duration : 0;
      }

      protected abstract List<DataSlice> performIO(List<Long> sourceNames, long transactionId)
            throws SliceStoreIOException, IllegalSourceNameException,
            SliceStoreTransactionException, SliceStoreQuotaException, SliceStoreNotFoundException;

      protected abstract long getSourceName(int request, int requestSource);

      public int getRequestsPerTransaction()
      {
         return requestsPerTransaction;
      }

      public void setRequestsPerTransaction(int requestsPerTransaction)
      {
         this.requestsPerTransaction = requestsPerTransaction;
      }

      public int getSlicesPerRequest()
      {
         return slicesPerRequest;
      }

      public void setSlicesPerRequest(int slicesPerRequest)
      {
         this.slicesPerRequest = slicesPerRequest;
      }

      public int getSliceSize()
      {
         return sliceSize;
      }

      public void setSliceSize(int sliceSize)
      {
         this.sliceSize = sliceSize;
         data = new byte[sliceSize];
      }

      public List<Tuple3<Long, Long, Long>> getIOCompletionTimes()
      {
         return ioCompletionTimes;
      }

      protected DataSlice createDataSlice(long sourceName, long transactionId)
      {
         return new DataSlice(new SliceName(String.valueOf(sourceName), SLICESTORE_IDX),
               transactionId, data);
      }
   }

   private abstract class WriteIOTestTask extends IOTestTask
   {
      protected List<DataSlice> performIO(List<Long> sourceNames, long transactionId)
            throws SliceStoreIOException, IllegalSourceNameException,
            SliceStoreTransactionException, SliceStoreQuotaException, SliceStoreNotFoundException
      {
         List<DataSlice> dataSlices = new ArrayList<DataSlice>();

         for (Long sourceName : sourceNames)
         {
            dataSlices.add(createDataSlice(sourceName, transactionId));
         }

         sliceStore.write(dataSlices);
         return dataSlices;
      }
   }

   private class SequentialWriteIOTestTask extends WriteIOTestTask
   {
      public String getType()
      {
         return "sequential-write";
      }

      protected long getSourceName(int request, int requestSource)
      {
         return (request * slicesPerRequest + requestSource) % maxSourceName;
      }
   }

   private class SequentialWriteIOTestTaskFactory implements IOTestTaskFactory
   {
      public IOTestTask createTestTask()
      {
         return new SequentialWriteIOTestTask();
      }
   }

   private class RandomWriteIOTestTask extends WriteIOTestTask
   {
      public String getType()
      {
         return "random-write";
      }

      protected long getSourceName(int request, int requestSource)
      {
         return Math.abs(random.nextLong() % maxSourceName);
      }
   }

   private class RandomWriteIOTestTaskFactory implements IOTestTaskFactory
   {
      public IOTestTask createTestTask()
      {
         return new RandomWriteIOTestTask();
      }
   }

   private abstract class ReadIOTestTask extends IOTestTask
   {
      protected List<DataSlice> performIO(List<Long> sourceNames, long transactionId)
            throws SliceStoreIOException, IllegalSourceNameException,
            SliceStoreTransactionException, SliceStoreQuotaException, SliceStoreNotFoundException
      {
         List<SliceName> sliceNames = new ArrayList<SliceName>();

         for (Long sourceName : sourceNames)
         {
            sliceNames.add(new SliceName(new SourceName(String.valueOf(sourceName)), 0));
         }

         return sliceStore.read(sliceNames);
      }
   }

   private class SequentialReadIOTestTask extends ReadIOTestTask
   {
      public String getType()
      {
         return "sequential-read";
      }

      protected long getSourceName(int request, int requestSource)
      {
         return (request * slicesPerRequest + requestSource) % maxSourceName;
      }
   }

   private class SequentialReadIOTestTaskFactory implements IOTestTaskFactory
   {
      public IOTestTask createTestTask()
      {
         return new SequentialReadIOTestTask();
      }
   }

   private class RandomReadIOTestTask extends WriteIOTestTask
   {
      public String getType()
      {
         return "random-read";
      }

      protected long getSourceName(int request, int requestSource)
      {
         return Math.abs(random.nextLong() % maxSourceName);
      }
   }

   private class RandomReadIOTestTaskFactory implements IOTestTaskFactory
   {
      public IOTestTask createTestTask()
      {
         return new RandomReadIOTestTask();
      }
   }

   private interface IOTestTaskFactory
   {
      IOTestTask createTestTask();
   }

   private abstract class IOTestTaskParameterVariator
   {
      int valueIdx = -1;

      public abstract String getName();

      public boolean hasMoreVariations()
      {
         return valueIdx < getValues().length - 1;
      }

      public void nextVariation()
      {
         valueIdx++;
      }

      public abstract void applyVariation(IOTestTask ioTestTask);

      public int getParameterValue()
      {
         return getValues()[valueIdx];
      }

      protected abstract int[] getValues();
   }

   private class FixedIOTestTaskParameterVariator extends IOTestTaskParameterVariator
   {
      protected int[] VALUES = {
         0
      };

      public String getName()
      {
         return "fixed";
      }

      public void applyVariation(IOTestTask ioTestTask)
      {
      }

      protected int[] getValues()
      {
         return VALUES;
      }
   }

   private class RequestsPerTransactionIOTestTaskParameterVariator
         extends
            IOTestTaskParameterVariator
   {
      protected int[] VALUES = {
            1, 2, 4, 8, 16, 32, 128, 512, 1024
      };

      public String getName()
      {
         return "requestsPerTransaction";
      }

      public void applyVariation(IOTestTask ioTestTask)
      {
         ioTestTask.setRequestsPerTransaction(getParameterValue());
      }

      protected int[] getValues()
      {
         return VALUES;
      }
   }

   private class SlicesPerRequestIOTestTaskParamterVariator extends IOTestTaskParameterVariator
   {
      private final int[] VALUES = {
            1, 8, 16, 32, 128, 256, 512, 1024, 2048, 3072, 4096
      };

      public String getName()
      {
         return "slicesPerRequest";
      }

      public void applyVariation(IOTestTask ioTestTask)
      {
         ioTestTask.setSlicesPerRequest(getParameterValue());
      }

      protected int[] getValues()
      {
         return VALUES;
      }
   }

   private class SliceSizeIOTestTaskParamterVariator extends IOTestTaskParameterVariator
   {
      int[] VALUES = {
            1, 16, 32, 64, 128, 256, 512
      // , 1024, 2048, 3072, 4096
            };

      public String getName()
      {
         return "sliceSize";
      }

      public void applyVariation(IOTestTask ioTestTask)
      {
         ioTestTask.setSliceSize(getParameterValue());
      }

      protected int[] getValues()
      {
         return VALUES;
      }
   }

   private class ListTestTask implements Callable<Float>
   {
      protected int minIterations;
      protected int minDuration;

      protected SliceStore sliceStore;

      public void init(Vault vault, int minIterations, int minDuration)
      {
         this.sliceStore = vault.getSliceStores().get(SLICESTORE_IDX);
         this.minIterations = minIterations;
         this.minDuration = minDuration;
      }

      public Float call() throws Exception
      {
         int iteration = 0;
         long startTime = System.currentTimeMillis();
         long itemsProcessed = 0;

         sliceStore.listBegin();

         while (iteration < minIterations || (System.currentTimeMillis() - startTime < minDuration))
         {
            if (!sliceStore.listInProgress())
            {
               sliceStore.listBegin();
            }

            List<SliceInfo> slices = sliceStore.listContinue();

            itemsProcessed += slices.size();
            iteration++;
         }

         float duration = (System.currentTimeMillis() - startTime) / 1000.0F;
         return itemsProcessed / duration;
      }

      protected void processSliceList(List<SliceInfo> slices) throws VaultDescriptorException,
            SliceStoreIOException, IllegalSourceNameException, VaultIOException,
            VaultSecurityException, SliceStoreNotFoundException
      {
      }
   }

   private class IntegrityVerificationListTestTask extends ListTestTask
   {
      protected void processSliceList(List<SliceInfo> slices) throws VaultDescriptorException,
            SliceStoreIOException, IllegalSourceNameException, VaultIOException,
            VaultSecurityException, SliceStoreNotFoundException
      {
         sliceStore.verifyIntegrity(slices);
      }
   }

   public SliceStorePerformanceTest()
   {
      properties = new Properties();

      properties.setProperty("min-iterations", "0");
      properties.setProperty("min-duration", String.valueOf(DEFAULT_DURATION));
      properties.setProperty("thread-count", "1");

      properties.setProperty("slice-size", "683");
      properties.setProperty("slices-per-request", "32");
      properties.setProperty("requests-per-transaction", "1");
   }

   @Test
   public void testSequentialWritePerformance() throws Exception
   {
      runTests(new IOTestTaskFactory()
      {
         public IOTestTask createTestTask()
         {
            return new SequentialWriteIOTestTask();
         }
      });
   }

   @Test
   public void testRandomWritePerformance() throws Exception
   {
      runTests(new IOTestTaskFactory()
      {
         public IOTestTask createTestTask()
         {
            return new RandomWriteIOTestTask();
         }
      });
   }

   @Test
   public void testSequentialReadPerformance() throws Exception
   {
      runTests(new IOTestTaskFactory()
      {
         public IOTestTask createTestTask()
         {
            return new SequentialReadIOTestTask();
         }
      });
   }

   @Test
   public void testRandomReadPerformance() throws Exception
   {
      runTests(new IOTestTaskFactory()
      {
         public IOTestTask createTestTask()
         {
            return new RandomReadIOTestTask();
         }
      });
   }

   @Test
   public void testListingPerformance() throws Exception
   {
      Vault vault = createVault();
      fillSliceStore(vault, SLICESTORE_IDX);

      ListTestTask listTestTask = new ListTestTask();
      listTestTask.init(vault, getMinIterations(), getMinDuration());
      float itemsPerSecond = listTestTask.call();

      logger.info("Sources listed per second " + itemsPerSecond);
   }

   @Test
   public void testIntegrityVerificationPerformance() throws Exception
   {
      Vault vault = createVault();
      fillSliceStore(vault, SLICESTORE_IDX);

      ListTestTask listTestTask = new IntegrityVerificationListTestTask();
      listTestTask.init(vault, getMinIterations(), getMinDuration());
      float itemsPerSecond = listTestTask.call();

      logger.info("Sources verified per second " + itemsPerSecond);
   }

   private void runTests(IOTestTaskFactory testTaskFactory) throws Exception
   {
      try
      {
         IOTestTaskParameterVariator varaitor;

         for (int numThreads = 1; numThreads <= getThreadCount(); numThreads++)
         {
            varaitor = new RequestsPerTransactionIOTestTaskParameterVariator();
            testPerformance(testTaskFactory, varaitor, numThreads);

            varaitor = new SlicesPerRequestIOTestTaskParamterVariator();
            testPerformance(testTaskFactory, varaitor, numThreads);

            varaitor = new SliceSizeIOTestTaskParamterVariator();
            testPerformance(testTaskFactory, varaitor, numThreads);
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         throw ex;
      }
   }

   private void testPerformance(
         IOTestTaskFactory testTaskFactory,
         IOTestTaskParameterVariator variator,
         int numThreads) throws Exception
   {
      ExecutorService executor = Executors.newCachedThreadPool();

      // Create a vault for each thread
      Map<Integer, Vault> threadVaultMap = new HashMap<Integer, Vault>();
      for (int thread = 0; thread < numThreads; thread++)
      {
         logger.info("Creating vault");
         Vault vault = createVault();
         logger.info("Filling slice store with data");
         fillSliceStore(vault, SLICESTORE_IDX);

         threadVaultMap.put(thread, vault);
      }

      IOTestTask ioTask = testTaskFactory.createTestTask();

      logger.info("Testing " + ioTask.getType() + " performance using variator "
            + variator.getName());
      logger.info("numThreads = " + numThreads);

      // Open file to write results to
      PrintStream out =
            new PrintStream(getOutputDirectory().getPath() + System.getProperty("file.separator")
                  + ioTask.getType() + "-" + variator.getName() + "-" + numThreads + ".csv");

      PrintStream ioCompletionTimesOut =
            new PrintStream(getOutputDirectory().getPath() + System.getProperty("file.separator")
                  + ioTask.getType() + "-" + variator.getName() + "-" + numThreads + "-"
                  + "iocompletion" + ".csv");

      while (variator.hasMoreVariations())
      {
         variator.nextVariation();

         List<Callable<Float>> ioTasks = new ArrayList<Callable<Float>>();
         for (int thread = 0; thread < numThreads; thread++)
         {
            ioTask = testTaskFactory.createTestTask();

            Vault vault = threadVaultMap.get(thread);

            ioTask.init(vault, getMinIterations(), getMinDuration(), getRequestsPerTransaction(),
                  getSlicesPerRequest(), getSliceSize());
            variator.applyVariation(ioTask);

            ioTasks.add(ioTask);
         }

         List<Future<Float>> ioTaskResults = executor.invokeAll(ioTasks);

         float totalBytesPerSecond = 0;
         for (int idx = 0; idx < ioTasks.size(); idx++)
         {
            ioTask = (IOTestTask) ioTasks.get(idx);
            Future<Float> ioTaskResult = ioTaskResults.get(idx);
            totalBytesPerSecond += ioTaskResult.get();

            for (Tuple3<Long, Long, Long> completionTime : ioTask.getIOCompletionTimes())
            {
               Long startTime = completionTime.getFirst();
               Long duration = completionTime.getSecond();
               Long bytesTransferred = completionTime.getThird();

               String line = idx + "," + startTime + "," + duration + "," + bytesTransferred;

               ioCompletionTimesOut.println(line);
            }
         }

         String line =
               variator.getParameterValue() + "," + (totalBytesPerSecond / (1024.0F * 1024.0F));

         logger.info(line);
         out.println(line);
      }

      out.close();
      ioCompletionTimesOut.close();

      // Cleanup vaults 
      for (Vault vault : threadVaultMap.values())
      {
         cleanupVault(vault);
      }

      executor.shutdown();
      while (!executor.awaitTermination(10, TimeUnit.SECONDS));
   }

   public int getMinIterations()
   {
      return Integer.parseInt(properties.getProperty("min-iterations"));
   }

   public int getMinDuration()
   {
      return Integer.parseInt(properties.getProperty("min-duration"));
   }

   public int getThreadCount()
   {
      return Integer.parseInt(properties.getProperty("thread-count"));
   }

   public int getRequestsPerTransaction()
   {
      return Integer.parseInt(properties.getProperty("requests-per-transaction"));
   }

   public int getSlicesPerRequest()
   {
      return Integer.parseInt(properties.getProperty("slices-per-request"));
   }

   public int getSliceSize()
   {
      return Integer.parseInt(properties.getProperty("slice-size"));
   }

   public void setProperty(String name, String value)
   {
      properties.setProperty(name, value);
   }

   public static void main(String args[]) throws Exception
   {
      final SliceStorePerformanceTest test = new SliceStorePerformanceTest();

      for (int argIdx = 1; argIdx < args.length; argIdx++)
      {
         String[] elements = args[argIdx].split("=");
         test.setProperty(elements[0], elements[1]);
      }

      if (args.length == 0)
      {
         throw new Exception("No test name specified");
      }
      if (args[0].equalsIgnoreCase("sequential-write"))
      {
         test.testPerformance(test.new SequentialWriteIOTestTaskFactory(),
               test.new FixedIOTestTaskParameterVariator(), test.getThreadCount());
      }
      else if (args[0].equalsIgnoreCase("random-write"))
      {
         test.testPerformance(test.new RandomWriteIOTestTaskFactory(),
               test.new FixedIOTestTaskParameterVariator(), test.getThreadCount());
      }
      else if (args[0].equalsIgnoreCase("sequential-read"))
      {
         test.testPerformance(test.new SequentialReadIOTestTaskFactory(),
               test.new FixedIOTestTaskParameterVariator(), test.getThreadCount());
      }
      else if (args[0].equalsIgnoreCase("random-read"))
      {
         test.testPerformance(test.new RandomReadIOTestTaskFactory(),
               test.new FixedIOTestTaskParameterVariator(), test.getThreadCount());
      }
      else if (args[0].equalsIgnoreCase("list"))
      {
         test.testListingPerformance();
      }
      else if (args[0].equalsIgnoreCase("verification"))
      {
         test.testListingPerformance();
      }
   }

}
