//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Aug 16, 2007
//---------------------

package org.cleversafe.serialization;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.List;

import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.protocol.GridProtocolOperation;
import org.cleversafe.layer.protocol.MultipleReadRequest;
import org.cleversafe.layer.protocol.MultipleReadResponse;
import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.exceptions.OperationNotRegisteredException;
import org.cleversafe.layer.protocol.exceptions.ProtocolDeserializationException;
import org.cleversafe.layer.protocol.exceptions.ProtocolSerializationException;
import org.cleversafe.serialization.asn1.ASN1;
import org.cleversafe.serialization.asn1.ASN1GridProtocolMessageFactory;
import org.cleversafe.serialization.asn1.custom.Registrator;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

// TODO: Describe class or interface
public class ReadComparisonTest
{
   private static ProtocolMessageFactory pmFactory = null;

   private static SliceName[] sliceNames = null;
   private static DataSlice[] dataSlices = null;
   private static int numIterations = 10000;
   private static int numDataSlices = 128;
   private static int numSliceNames = 128;
   private static int SLICE_BLOCK_SIZE = 353;

   private static int GRID_WIDTH = 16;

   private static int BLOCK_SIZE = 4096;

   @BeforeClass
   public static void setup()
   {
      numIterations = Integer.getInteger("test.iteration", numIterations);
      numSliceNames = numDataSlices = Integer.getInteger("test.numSlices", numSliceNames);

      pmFactory = new ASN1GridProtocolMessageFactory();
      sliceNames = ProtocolMessageFactoryTestBase.getRandomSliceNames(numSliceNames);
      dataSlices =
            ProtocolMessageFactoryTestBase.getRandomDataSliceArray(numDataSlices, SLICE_BLOCK_SIZE);
   }

   @AfterClass
   public static void tearDown()
   {
      pmFactory = null;
   }

   private void testMessageSerializationPerformance(
         ProtocolMessage message,
         String description,
         Serializer serializer) throws Exception
   {
      // serialize once to get the serialized output
      byte[] output = serializer.serialize(message);

      long startReadTime = System.currentTimeMillis();

      for (int i = 0; i < numIterations; ++i)
      {
         serializer.serialize(message);
      }
      long endReadTime = System.currentTimeMillis() - startReadTime;

      long startWriteTime = System.currentTimeMillis();
      for (int i = 0; i < numIterations; ++i)
      {
         serializer.deserialize(message.getOperation().getOperationCode(), output);
      }
      long endWriteTime = System.currentTimeMillis() - startWriteTime;

      double averageReadTime = ((double) endReadTime) / ((double) numIterations);
      double averageWriteTime = ((double) endWriteTime) / ((double) numIterations);
      double bytesInMbit = 1024 * 1024 / 8;
      double sendsPerMbit = bytesInMbit / (BLOCK_SIZE * numDataSlices);
      double conversionsForMbit = sendsPerMbit * GRID_WIDTH;
      double timePerMbit = conversionsForMbit * ((averageReadTime + averageWriteTime) / 1000);

      System.out.println(String.format(
            "%s: \t%.3f ms./serial\t%.3f ms./deserial\t%.2f kB/message\tThroughput: %.0f Mbps",
            description, averageReadTime, averageWriteTime, output.length / 1024.0 / 8,
            1 / timePerMbit));

   }

   private void configureProtocolMessage(ProtocolMessage pm)
   {
      if (pm instanceof MultipleReadRequest)
      {
         ((MultipleReadRequest) pm).setSliceNames(sliceNames);
      }
      else if (pm instanceof MultipleReadResponse)
      {
         ((MultipleReadResponse) pm).setDataSlices(dataSlices);
      }
   }

   private void createMessageAndTest(
         String description,
         Class<? extends ProtocolMessage> cls,
         Serializer serializer) throws Exception
   {
      Runtime.getRuntime().gc();
      ProtocolMessage message = cls.newInstance();
      configureProtocolMessage(message);
      testMessageSerializationPerformance(message, description, serializer);
   }

   @Test
   public void testProtocolMessageSerializationPerformance() throws Exception
   {

      createMessageAndTest("ASN custom request", MultipleReadRequest.class, new ASNSerializer());

      ASN1.DERConverter.unregisterDefaultEncodeDecode(MultipleReadRequest.class);
      createMessageAndTest("ASN generic request", MultipleReadRequest.class, new ASNSerializer());
      Registrator.registorCustomObjects();

      /*
      createMessageAndTest("Java serial request", MultipleReadRequest.class, new JavaSerializer());
      createMessageAndTest("Java extenal response", ExternalizableMultipleReadRequest.class,
            new JavaSerializer());
       */
      createMessageAndTest("Raw serial request", ExternalizableMultipleReadRequest.class,
            new RawSerializer());

      createMessageAndTest("ASN custom response", MultipleReadResponse.class, new ASNSerializer());

      ASN1.DERConverter.unregisterDefaultEncodeDecode(MultipleReadResponse.class);
      createMessageAndTest("ASN generic response", MultipleReadResponse.class, new ASNSerializer());
      Registrator.registorCustomObjects();

      /*
      createMessageAndTest("Java serial response", MultipleReadResponse.class, new JavaSerializer());
      createMessageAndTest("Java extenal response", ExternalizableMultipleReadResponse.class,
            new JavaSerializer());
      */
      createMessageAndTest("Raw serial response", ExternalizableMultipleReadResponse.class,
            new RawSerializer());

   }

   public interface Serializer
   {
      byte[] serialize(ProtocolMessage message) throws ProtocolSerializationException, IOException;

      public ProtocolMessage deserialize(int operationCode, byte[] message)
            throws OperationNotRegisteredException, ProtocolDeserializationException, IOException,
            ClassNotFoundException;
   }

   public class ASNSerializer implements Serializer
   {

      public byte[] serialize(ProtocolMessage message) throws ProtocolSerializationException
      {
         ByteArrayOutputStream bytes = new ByteArrayOutputStream();
         DataOutputStream out = new DataOutputStream(bytes);
         pmFactory.serialize(out, message);
         return bytes.toByteArray();
      }

      public ProtocolMessage deserialize(int operationCode, byte[] message)
            throws OperationNotRegisteredException, ProtocolDeserializationException
      {
         return pmFactory.deserialize(operationCode, new DataInputStream(new ByteArrayInputStream(
               message)));

      }
   }

   public class JavaSerializer implements Serializer
   {
      ByteArrayOutputStream bytesOut;
      ObjectOutputStream out;

      public JavaSerializer() throws IOException
      {
         bytesOut = new ByteArrayOutputStream();
         out = new ObjectOutputStream(bytesOut);
      }

      public byte[] serialize(ProtocolMessage message) throws IOException
      {
         out.writeObject(message);
         byte[] rVal = bytesOut.toByteArray();
         out.reset();
         bytesOut.reset();

         return rVal;
      }

      public ProtocolMessage deserialize(int operationCode, byte[] message)
            throws OperationNotRegisteredException, ProtocolDeserializationException, IOException,
            ClassNotFoundException
      {
         ByteArrayInputStream bytesIn = new ByteArrayInputStream(message);
         ObjectInputStream in = new ObjectInputStream(bytesIn);
         return (ProtocolMessage) in.readObject();
      }
   }

   public class RawSerializer implements Serializer
   {

      public ProtocolMessage deserialize(int operationCode, byte[] message)
            throws OperationNotRegisteredException, ProtocolDeserializationException, IOException,
            ClassNotFoundException
      {
         ByteArrayInputStream bytesIn = new ByteArrayInputStream(message);
         DataInputStream in = new DataInputStream(bytesIn);

         if (operationCode == GridProtocolOperation.MULTIPLE_READ_RESPONSE.getOperationCode())
         {
            ExternalizableMultipleReadResponse response = new ExternalizableMultipleReadResponse();
            response._readExternal(in);
            return response;
         }
         else if (operationCode == GridProtocolOperation.MULTIPLE_READ_REQUEST.getOperationCode())
         {
            ExternalizableMultipleReadRequest response = new ExternalizableMultipleReadRequest();
            response._readExternal(in);
            return response;
         }
         else
         {
            throw new ProtocolDeserializationException("Unknown message type " + operationCode);
         }
      }
      ByteArrayOutputStream bytes = new ByteArrayOutputStream();
      DataOutputStream out = new DataOutputStream(bytes);

      public byte[] serialize(ProtocolMessage message) throws ProtocolSerializationException,
            IOException
      {
         bytes.reset();
         // out.reset();
         if (message instanceof ExternalizableMultipleReadResponse)
            ((ExternalizableMultipleReadResponse) message)._writeExternal(out);
         else if (message instanceof ExternalizableMultipleReadRequest)
            ((ExternalizableMultipleReadRequest) message)._writeExternal(out);
         return bytes.toByteArray();
      }
   }

   public static class ExternalizableMultipleReadResponse extends MultipleReadResponse
         implements
            Externalizable
   {

      public ExternalizableMultipleReadResponse()
      {
         super();
      }

      public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
      {
         _readExternal(in);
      }

      public void writeExternal(ObjectOutput out) throws IOException
      {
         _writeExternal(out);
      }

      public void _readExternal(DataInput in) throws IOException, ClassNotFoundException
      {
         int size = in.readInt();
         List<DataSlice> dataSlices = new ArrayList<DataSlice>(size);

         for (int i = 0; i < size; i++)
         {
            dataSlices.add(readSlice(in));
         }
         this.setDataSlices(dataSlices);
      }

      public void _writeExternal(DataOutput out) throws IOException
      {
         DataSlice[] dataSlices = getDataSlices();
         out.writeInt(dataSlices.length);
         for (DataSlice slice : dataSlices)
         {
            writeSlice(out, slice);
         }
      }

      private DataSlice readSlice(DataInput in) throws IOException
      {
         DataSlice slice = new DataSlice();
         SliceName sliceName = new SliceName();
         slice.setSliceName(sliceName);

         SourceName sourceName = new SourceName();
         sliceName.setSourceName(sourceName);

         sourceName.setName(in.readUTF());
         sliceName.setSliceIndex(in.readInt());
         slice.setTransactionId(in.readLong());

         int dataLength = in.readInt();
         byte[] data = new byte[dataLength];
         in.readFully(data, 0, dataLength);
         slice.setData(data);

         return slice;
      }

      // format: sourceName, sliceIndex, sliceTransactionId, sliceDataLen, sliceData
      private void writeSlice(DataOutput out, DataSlice slice) throws IOException
      {
         SliceName sliceName = slice.getSliceName();
         SourceName sourceName = sliceName.getSourceName();

         out.writeUTF(sourceName.getName());
         out.writeInt(sliceName.getSliceIndex());

         out.writeLong(slice.getTransactionId());
         out.writeInt(slice.getData().length);
         out.write(slice.getData(), 0, slice.getData().length);
      }
   }

   public static class ExternalizableMultipleReadRequest extends MultipleReadRequest
         implements
            Externalizable
   {

      public ExternalizableMultipleReadRequest()
      {
      }

      public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
      {
         _readExternal(in);
      }

      public void writeExternal(ObjectOutput out) throws IOException
      {
         _writeExternal(out);
      }

      public void _readExternal(DataInput in) throws IOException, ClassNotFoundException
      {
         int size = in.readInt();
         List<SliceName> sliceNames = new ArrayList<SliceName>(size);

         for (int i = 0; i < size; i++)
         {
            sliceNames.add(readSliceName(in));
         }
         this.setSliceNames(sliceNames);

      }

      private SliceName readSliceName(DataInput in) throws IOException
      {
         SliceName sliceName = new SliceName();

         SourceName sourceName = new SourceName();
         sliceName.setSourceName(sourceName);

         sliceName.setSliceIndex(in.readInt());
         sourceName.setName(in.readUTF());

         return sliceName;
      }

      public void _writeExternal(DataOutput out) throws IOException
      {
         SliceName[] sliceNames = getSliceNames();
         out.writeInt(sliceNames.length);
         for (SliceName sliceName : sliceNames)
         {
            writeSliceName(out, sliceName);
         }
      }

      private void writeSliceName(DataOutput out, SliceName sliceName) throws IOException
      {
         SourceName sourceName = sliceName.getSourceName();
         out.writeInt(sliceName.getSliceIndex());
         out.writeUTF(sourceName.getName());
      }
   }

}
