//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Aug 8, 2007
//---------------------

package org.cleversafe.test;

import java.io.File;
import java.io.RandomAccessFile;
import java.util.Arrays;
import java.util.Random;

public class DiskIOPerformance
{
   private static int minRepeatFactor = 1000;

   /**
    * @param args
    */
   public static void main(String[] args) throws Exception
   {
      int singleBlockSize = Integer.parseInt(args[0]);
      minRepeatFactor = Integer.parseInt(args[1]);
      
      for (int i = 2; i < args.length; i++)
      {
         int fileSizeMB = Integer.parseInt(args[i]);
         int fileSize = fileSizeMB * 1024 * 1024;

         RandomAccessFile blockFile = null;
         try
         {
            File f = new File("test" + fileSizeMB + ".dat");
            f.deleteOnExit();
            blockFile = new RandomAccessFile(f, "rw");
            long start = System.currentTimeMillis();
            
            // create full size
            // The fastes way to allocate diskspace            
            blockFile.setLength(fileSize);

            long end = System.currentTimeMillis();
            System.out.println("=========================================================");
            System.out.println("File size " + fileSizeMB + "MB created in " + (end - start)
                  + " msec");

            for (int nblocks : new int[]{
                  1, 4, 8, 16, 32, 128
            })
            {
               int blockSize = nblocks * singleBlockSize;
               double mbytesSecWrite = testWrite(blockFile, fileSize, blockSize);
               System.out.println("Write(" + nblocks + " x " + singleBlockSize + "B) "
                     + mbytesSecWrite + " MB/sec");

               double mbytesSecread = testRead(blockFile, fileSize, blockSize);
               System.out.println("Read (" + nblocks + " x " + singleBlockSize + "B) "
                     + mbytesSecread + " MB/sec");
            }
         }
         finally
         {
            if (blockFile != null)
               blockFile.close();

            System.out.println();
         }
      }
   }

   // MB per sec
   private static double testWrite(
         RandomAccessFile blockFile,
         int fileSize,
         int blockSize) throws Exception
   {
      // 1000 random writes, 1000 random reads

      byte[] data = new byte[blockSize];
      Arrays.fill(data, (byte) 'a');

      final int maxBlock = fileSize / blockSize;

      Random rng = new Random();

      long start = System.currentTimeMillis();
      long end = System.currentTimeMillis();;
      while (end - start < 1000) // At least 1 sec
      {
         for (int i = 0; i < minRepeatFactor; i++)
         {
            int blkNumber = Math.abs(rng.nextInt()) % maxBlock;
            blockFile.seek(blkNumber * blockSize);
            blockFile.write(data);
         }
         end = System.currentTimeMillis();
      }
      // System.out.println (end + " " + start + " " + (end-start));
      double bytesPerMsec = ((double) minRepeatFactor) * blockSize / (end - start);
      return bytesPerMsec / (1024 * 1024) * 1000;
   }

   // MB per sec
   private static double testRead(
         RandomAccessFile blockFile,
         int fileSize,
         int blockSize) throws Exception
   {
      // 1000 random writes, 1000 random reads

      byte[] data = new byte[blockSize];

      final int maxBlock = fileSize / blockSize;

      Random rng = new Random();

      long start = System.currentTimeMillis();
      long end = System.currentTimeMillis();;
      while (end - start < 1000) // At least 1 sec to make stat trustworthy
      {
         for (int i = 0; i < minRepeatFactor; i++)
         {
            int blkNumber = Math.abs(rng.nextInt()) % maxBlock;
            blockFile.seek(blkNumber * blockSize);
            blockFile.read(data);
         }
         end = System.currentTimeMillis();
      }
      // System.out.println (end + " " + start + " " + (end-start));
      double bytesPerMsec = ((double) minRepeatFactor) * blockSize / (end - start);
      return bytesPerMsec / (1024 * 1024) * 1000;
   }

}
