//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: Jason Resch
//
// Date: Sep 20, 2007
//---------------------

package org.cleversafe.codec;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Arrays;
import java.util.List;
import java.util.Random;

import org.cleversafe.codec.exceptions.CodecInvalidParametersException;
import org.cleversafe.layer.grid.SourceName;
import org.junit.Test;


public abstract class CodecTestBase
{
   private static int _NUM_ITERATIONS = 50;
   
   private static int _SMALL_MESSAGE_MAX_SIZE = 260;
   
   private static int _MAX_DATA_SIZE = 64*1024;      // 64 KB
   private static int _MIN_DATA_SIZE = 0;            // 0 B
   
   private static int _MAX_CHUNK_SIZE = 8096;        // 8 KB
   private static int _MIN_CHUNK_SIZE = 0;           // 0 B
   
   private static final int _RAND_SEED = 12345;
   
   public abstract List<Codec> getCodecPermutations() throws Exception;
   
   public final List<Codec> finalCodecs; 
   
   public CodecTestBase() throws Exception
   {
      finalCodecs = this.getCodecPermutations();
   }
   
   @Test
   public void getCodecName()
   {
      if (this.finalCodecs != null)
      {
         for (Codec codec : finalCodecs)
         {
            System.out.println("Testing codec: " + codec.getName());
            System.out.flush();
         }
      }
      else
      {
         fail("Codec was not loaded properly");
      }
   }
   
   @Test
   public void testOneCallToFinish() throws Exception
   {
      List<Codec> codecs = this.getCodecPermutations();
      for (Codec codec : codecs)
      {
      
         // Use the random number generator to produce a reproduceable data stream
         Random dataGenerator1 = new Random();
         Random dataGenerator2 = new Random();
         Random variableGenerator = new Random();
         
         dataGenerator1.setSeed(_RAND_SEED);
         dataGenerator2.setSeed(_RAND_SEED);
         variableGenerator.setSeed(_RAND_SEED);
         
         // Create an Encoder and Decoder
         Encoder encoder = codec.getEncoder();
         Decoder decoder = codec.getDecoder();
         
         for (int itr = 0; itr < _NUM_ITERATIONS; itr++)
         {
            // Reset the codec
            encoder.reset(new SourceName("0"), 0);
            decoder.reset(new SourceName("0"), 0);
            
            // Generate random data length to encode
            int dataSize = variableGenerator.nextInt(_MAX_DATA_SIZE - _MIN_DATA_SIZE) 
               + _MIN_DATA_SIZE;
            long actualDecodedSize = 0;
            
            // Save expected size to compare later
            long expectedEncodedSize = codec.getEncodedSize(dataSize);
            long actualEncodedSize = 0;
            
   
            // Encode the entire piece in one chunk
            int chunkLength = dataSize;
   
            byte[] data = new byte[chunkLength];
            this.nextRandomBytes(dataGenerator1, data);
   
            // Encode Data
            byte[] encodedData = encoder.finish(data);
            actualEncodedSize += encodedData.length;
               
            // Decode Data
            byte[] decodedData = decoder.finish(encodedData);
            actualDecodedSize += decodedData.length;
               
            // Compare data
            byte[] compareData = new byte[decodedData.length];
            this.nextRandomBytes(dataGenerator2, compareData);
            assertTrue(Arrays.equals(decodedData, compareData));
            
            // Compare expected lengths
            assertTrue(expectedEncodedSize == actualEncodedSize);
       
            // Compare decoded length
            assertTrue(dataSize == actualDecodedSize);
         }
      }
   }
   
   @Test
   public void testManyProcessCallsOneFinishWithData() throws Exception
   {
      List<Codec> codecs = this.getCodecPermutations();
      for (Codec codec : codecs)
      {
         
         // Use the random number generator to produce a reproduceable data stream
         Random dataGenerator1 = new Random();
         Random dataGenerator2 = new Random();
         Random variableGenerator = new Random();
         
         dataGenerator1.setSeed(_RAND_SEED);
         dataGenerator2.setSeed(_RAND_SEED);
         variableGenerator.setSeed(_RAND_SEED);
         
         // Create an Encoder and Decoder
         Encoder encoder = codec.getEncoder();
         Decoder decoder = codec.getDecoder();
         
         for (int itr = 0; itr < _NUM_ITERATIONS; itr++)
         {
            // Reset the codec
            encoder.reset(new SourceName("0"), 0);
            decoder.reset(new SourceName("0"), 0);
            
            // Generate random data length to encode
            int dataSize = variableGenerator.nextInt(_MAX_DATA_SIZE - _MIN_DATA_SIZE) 
               + _MIN_DATA_SIZE;
            long actualDecodedSize = 0;
            
            // Save expected size to compare later
            long expectedEncodedSize = codec.getEncodedSize(dataSize);
            long actualEncodedSize = 0;
         
            int amountEncoded = 0;
            while (amountEncoded < dataSize)
            {
               // Generate random chunk length to process
               int chunkLength = variableGenerator.nextInt(_MAX_CHUNK_SIZE - _MIN_CHUNK_SIZE) 
                  + _MIN_CHUNK_SIZE;
               if (chunkLength + amountEncoded > dataSize)
               {
                  chunkLength = dataSize - amountEncoded;
               }
               byte[] data = new byte[chunkLength];
               this.nextRandomBytes(dataGenerator1, data);
               amountEncoded += data.length;
               
               // Encode Data
               byte[] encodedData = (amountEncoded == dataSize) ? 
                     encoder.finish(data) : encoder.process(data);
               
               actualEncodedSize += encodedData.length;
               
               // Decode Data
               byte[] decodedData = decoder.process(encodedData);
               actualDecodedSize += decodedData.length;
               
               // Compare data
               byte[] compareData = new byte[decodedData.length];
               this.nextRandomBytes(dataGenerator2, compareData);
               assertTrue(Arrays.equals(decodedData, compareData));
            }
            
            // Finish the codec
            byte[] decodedData = decoder.finish();
            actualDecodedSize += decodedData.length;
            
            // Compare data
            byte[] compareData = new byte[decodedData.length];
            this.nextRandomBytes(dataGenerator2, compareData);
            assertTrue(Arrays.equals(decodedData, compareData));
            
            // Compare expected lengths
            assertTrue(expectedEncodedSize == actualEncodedSize);
       
            // Compare decoded length
            assertTrue(dataSize == actualDecodedSize);
            
         }
      }
   }
   

   
   @Test
   public void testManyProcessCallsOneFinishWithoutData() throws Exception
   {
      List<Codec> codecs = this.getCodecPermutations();
      for (Codec codec : codecs)
      {

         // Use the random number generator to produce a reproduceable data stream
         Random dataGenerator1 = new Random();
         Random dataGenerator2 = new Random();
         Random variableGenerator = new Random();
         
         dataGenerator1.setSeed(_RAND_SEED);
         dataGenerator2.setSeed(_RAND_SEED);
         variableGenerator.setSeed(_RAND_SEED);
         
         // Create an Encoder and Decoder
         Encoder encoder = codec.getEncoder();
         Decoder decoder = codec.getDecoder();
         
         for (int itr = 0; itr < _NUM_ITERATIONS; itr++)
         {
            // Reset the codec
            encoder.reset(new SourceName("0"), 0);
            decoder.reset(new SourceName("0"), 0);
            
            // Generate random data length to encode
            int dataSize = variableGenerator.nextInt(_MAX_DATA_SIZE - _MIN_DATA_SIZE) 
               + _MIN_DATA_SIZE;
            long actualDecodedSize = 0;
            
            // Save expected size to compare later
            long expectedEncodedSize = codec.getEncodedSize(dataSize);
            long actualEncodedSize = 0;
         
            int amountEncoded = 0;
            while (amountEncoded < dataSize)
            {
               // Generate random chunk length to process
               int chunkLength = variableGenerator.nextInt(_MAX_CHUNK_SIZE - _MIN_CHUNK_SIZE) 
                  + _MIN_CHUNK_SIZE;
               if (chunkLength + amountEncoded > dataSize)
               {
                  chunkLength = dataSize - amountEncoded;
               }
               byte[] data = new byte[chunkLength];
               this.nextRandomBytes(dataGenerator1, data);
               amountEncoded += data.length;
               
               // Encode Data
               byte[] encodedData = encoder.process(data);
               actualEncodedSize += encodedData.length;
               
               // Decode Data
               byte[] decodedData = decoder.process(encodedData);
               actualDecodedSize += decodedData.length;
               
               // Compare data
               byte[] compareData = new byte[decodedData.length];
               this.nextRandomBytes(dataGenerator2, compareData);
               assertTrue(Arrays.equals(decodedData, compareData));
            }
            
            // Encode Data
            byte[] encodedData = encoder.finish();
            actualEncodedSize += encodedData.length;
            
            // Finish the codec
            byte[] decodedData = decoder.finish(encodedData);
            actualDecodedSize += decodedData.length;
            
            // Compare data
            byte[] compareData = new byte[decodedData.length];
            this.nextRandomBytes(dataGenerator2, compareData);
            assertTrue(Arrays.equals(decodedData, compareData));
            
            // Compare expected lengths
            assertTrue(expectedEncodedSize == actualEncodedSize);
       
            // Compare decoded length
            assertTrue(dataSize == actualDecodedSize);
         }
      }
   }
   
   @Test
   public void testSomeZeroSizedProcessCalls() throws Exception
   {
      List<Codec> codecs = this.getCodecPermutations();
      for (Codec codec : codecs)
      {
         
         // Use the random number generator to produce a reproduceable data stream
         Random dataGenerator1 = new Random();
         Random dataGenerator2 = new Random();
         Random variableGenerator = new Random();
         
         dataGenerator1.setSeed(_RAND_SEED);
         dataGenerator2.setSeed(_RAND_SEED);
         variableGenerator.setSeed(_RAND_SEED);
         
         // Create an Encoder and Decoder
         Encoder encoder = codec.getEncoder();
         Decoder decoder = codec.getDecoder();
         
         for (int itr = 0; itr < _NUM_ITERATIONS; itr++)
         {
            // Reset the codec
            encoder.reset(new SourceName("0"), 0);
            decoder.reset(new SourceName("0"), 0);
            
            // Generate random data length to encode
            int dataSize = variableGenerator.nextInt(_MAX_DATA_SIZE - _MIN_DATA_SIZE) 
               + _MIN_DATA_SIZE;
            long actualDecodedSize = 0;
            
            // Save expected size to compare later
            long expectedEncodedSize = codec.getEncodedSize(dataSize);
            long actualEncodedSize = 0;
         
            byte[] emptyBuffer = new byte[0];
            
            int amountEncoded = 0;
            while (amountEncoded < dataSize)
            {
               // Generate random chunk length to process
               int chunkLength = variableGenerator.nextInt(_MAX_CHUNK_SIZE - _MIN_CHUNK_SIZE) 
                  + _MIN_CHUNK_SIZE;
               if (chunkLength + amountEncoded > dataSize)
               {
                  chunkLength = dataSize - amountEncoded;
               }
               byte[] data = new byte[chunkLength];
               this.nextRandomBytes(dataGenerator1, data);
               amountEncoded += data.length;
               
               // Zero Length Process Call
               encoder.process(emptyBuffer);
               
               // Encode Data
               byte[] encodedData = (amountEncoded == dataSize) ? 
                     encoder.finish(data) : encoder.process(data);
               
               actualEncodedSize += encodedData.length;
               
               // Zero Length Process Call
               if (amountEncoded != dataSize)
               {
                  encoder.process(emptyBuffer);
               }
                  
               // Zero Length Process Call
               decoder.process(emptyBuffer);
               
               // Decode Data
               byte[] decodedData = decoder.process(encodedData);
               actualDecodedSize += decodedData.length;
               
               // Zero Length Process Call
               decoder.process(emptyBuffer);
               
               // Compare data
               byte[] compareData = new byte[decodedData.length];
               this.nextRandomBytes(dataGenerator2, compareData);
               assertTrue(Arrays.equals(decodedData, compareData));
            }
            
            // Zero Length Process Call
            decoder.process(emptyBuffer);
            
            // Finish the codec
            byte[] decodedData = decoder.finish();
            actualDecodedSize += decodedData.length;
            
            // Compare data
            byte[] compareData = new byte[decodedData.length];
            this.nextRandomBytes(dataGenerator2, compareData);
            assertTrue(Arrays.equals(decodedData, compareData));
            
            // Compare expected lengths
            assertTrue(expectedEncodedSize == actualEncodedSize);
       
            // Compare decoded length
            assertTrue(dataSize == actualDecodedSize);
            
         }
      }
   }
   
   @Test
   public void testEncodeDecodeEmptyBuffer() throws Exception
   {
      List<Codec> codecs = this.getCodecPermutations();
      for (Codec codec : codecs)
      {
         Encoder encoder = codec.getEncoder();
         Decoder decoder = codec.getDecoder();
         
         // Reset the codec
         encoder.reset(new SourceName("0"), 0);
         decoder.reset(new SourceName("0"), 0);
         
         // Create an empty buffer to test against
         byte[] emptyBuffer = new byte[0];
         
         // Perform encode/decode
         byte[] encodedData = encoder.finish(emptyBuffer);
         byte[] decodedData = decoder.finish(encodedData);
         
         // Compare
         assertTrue(Arrays.equals(emptyBuffer, decodedData));
      }
   }
   
   @Test
   public void testEncodeDecodeSmallMessages() throws Exception
   {
      List<Codec> codecs = this.getCodecPermutations();
      for (Codec codec : codecs)
      {
         Encoder encoder = codec.getEncoder();
         Decoder decoder = codec.getDecoder();
         
         for (int size = 0; size < _SMALL_MESSAGE_MAX_SIZE; size++)
         {
         
            // Reset the codec
            encoder.reset(new SourceName("0"), 0);
            decoder.reset(new SourceName("0"), 0);
            
            // Create an empty buffer to test against
            byte[] emptyBuffer = new byte[size];
            
            // Perform encode/decode
            byte[] encodedData = encoder.finish(emptyBuffer);
            byte[] decodedData = decoder.finish(encodedData);
            
            // Compare
            assertTrue(Arrays.equals(emptyBuffer, decodedData));
            
         }
      }
   }
   
   /**
    * Test that codec implementation does not share encoders, which may not be thread
    * safe depending on how threads obtain and use their encoders.
    * @throws Exception 
    */
   @Test
   public void testNewEncoderInstance() throws Exception
   {
      List<Codec> codecs = this.getCodecPermutations();
      for (Codec codec : codecs)
      {
         try
         {
            assertTrue(codec.getEncoder() != codec.getEncoder());
         }
         catch (CodecInvalidParametersException e)
         {
            e.printStackTrace();
            fail("An exception occurred attempting to get the encoder");
         }
      }
   }
   
   /**
    * Test that codec implementation does not share decoders, which may not be thread
    * safe depending on how threads obtain and use their encoders.
    * @throws Exception 
    */
   @Test
   public void testNewDecoderInstance() throws Exception
   {
      List<Codec> codecs = this.getCodecPermutations();
      for (Codec codec : codecs)
      {
         try
         {
            assertTrue(codec.getDecoder() != codec.getDecoder());
         }
         catch (CodecInvalidParametersException e)
         {
            e.printStackTrace();
            fail("An exception occurred attempting to get the decoder");
         }
      }
   }
   
   /**
    * Ensures that when reset is called, any temporary data is reset and the codec is 
    * not affected
    */
   @Test
   public void testResetWithoutFinish() throws Exception
   {
      List<Codec> codecs = this.getCodecPermutations();
      for (Codec codec : codecs)
      {
         
         // Use the random number generator to produce a reproduceable data stream
         Random dataGenerator1 = new Random();
         Random dataGenerator2 = new Random();
         Random variableGenerator = new Random();
         
         dataGenerator1.setSeed(_RAND_SEED);
         dataGenerator2.setSeed(_RAND_SEED);
         variableGenerator.setSeed(_RAND_SEED);
         
         // Create an Encoder and Decoder
         Encoder encoder = codec.getEncoder();
         Decoder decoder = codec.getDecoder();
         
         // Reset the codec
         encoder.reset(new SourceName("0"), 0);
         decoder.reset(new SourceName("0"), 0);
         
         // Generate random data length to encode
         int dataSize = variableGenerator.nextInt(_MAX_DATA_SIZE - _MIN_DATA_SIZE) 
            + _MIN_DATA_SIZE;
         long actualDecodedSize = 0;
         
         // Save expected size to compare later
         long expectedEncodedSize = codec.getEncodedSize(dataSize);
         long actualEncodedSize = 0;
      
         int amountEncoded = 0;
         while (amountEncoded < dataSize)
         {
            // Generate random chunk length to process
            int chunkLength = variableGenerator.nextInt(_MAX_CHUNK_SIZE - _MIN_CHUNK_SIZE) 
               + _MIN_CHUNK_SIZE;
            if (chunkLength + amountEncoded > dataSize)
            {
               chunkLength = dataSize - amountEncoded;
            }
            byte[] data = new byte[chunkLength];
            this.nextRandomBytes(dataGenerator1, data);
            amountEncoded += data.length;
            
            // Encode Data
            byte[] encodedData = encoder.process(data);
            
            actualEncodedSize += encodedData.length;
            
            // Decode Data
            byte[] decodedData = decoder.process(encodedData);
            actualDecodedSize += decodedData.length;
            
            // Compare data
            byte[] compareData = new byte[decodedData.length];
            this.nextRandomBytes(dataGenerator2, compareData);
            assertTrue(Arrays.equals(decodedData, compareData));
         }
         
         // Reset the encoder and decoder without finishing, and try to use them again
         encoder.reset(new SourceName("0"), 0);
         decoder.reset(new SourceName("0"), 0);
         
         // Also reset the random generators
         dataGenerator1.setSeed(_RAND_SEED);
         dataGenerator2.setSeed(_RAND_SEED);
         
         
         
         // Generate random data length to encode
         actualDecodedSize = 0;
         
         // Save expected size to compare later
         expectedEncodedSize = codec.getEncodedSize(dataSize);
         actualEncodedSize = 0;
      
         // Retry encode/decode to ensure reset fixed the codec
         amountEncoded = 0;
         while (amountEncoded < dataSize)
         {
            // Generate random chunk length to process
            int chunkLength = variableGenerator.nextInt(_MAX_CHUNK_SIZE - _MIN_CHUNK_SIZE) 
               + _MIN_CHUNK_SIZE;
            if (chunkLength + amountEncoded > dataSize)
            {
               chunkLength = dataSize - amountEncoded;
            }
            byte[] data = new byte[chunkLength];
            this.nextRandomBytes(dataGenerator1, data);
            amountEncoded += data.length;
            
            // Encode Data
            byte[] encodedData = (amountEncoded == dataSize) ? 
                  encoder.finish(data) : encoder.process(data);
            
            actualEncodedSize += encodedData.length;
            
            // Decode Data
            byte[] decodedData = decoder.process(encodedData);
            actualDecodedSize += decodedData.length;
            
            // Compare data
            byte[] compareData = new byte[decodedData.length];
            this.nextRandomBytes(dataGenerator2, compareData);
            assertTrue(Arrays.equals(decodedData, compareData));
         }
         
         // Finish the codec
         byte[] decodedData = decoder.finish();
         actualDecodedSize += decodedData.length;
         
         // Compare data
         byte[] compareData = new byte[decodedData.length];
         this.nextRandomBytes(dataGenerator2, compareData);
         assertTrue(Arrays.equals(decodedData, compareData));
         
         // Compare expected lengths
         assertTrue(expectedEncodedSize == actualEncodedSize);
    
         // Compare decoded length
         assertTrue(dataSize == actualDecodedSize);
      }
   }

   /**
    * This is needed to compensate for odd behavior in Java where two calls to
    * nextBytes from a Java random object result in different byte values than
    * a single call requesting two bytes.
    * 
    * @param rnd
    * @param array
    */
   private void nextRandomBytes(Random rnd, byte[] array)
   {
      for (int idx = 0; idx < array.length; idx++)
      {
         array[idx] = (byte) rnd.nextInt();
      }
   }
   
}


