//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: May 20, 2007
//---------------------

package org.cleversafe.config;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Set;

import org.cleversafe.UnitTests;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.test.BaseTest;
import org.cleversafe.util.Version;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

// TODO: Describe class or interface
public class XMLBindingsProviderTest
{
   private static BindingsProvider bindings = null;

   private static String XMLFileName;

   @BeforeClass
   public static void setUpBeforeClass()
   {
      UnitTests.resetConfiguration();
      XMLFileName =
            System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".")
                  + "/org/cleversafe/config/bindingsTest.xml";
      ConfigurationFactory.unloadBindings(ConfigurationFactory.XML_CONFIG_TYPE);
   }

   @AfterClass
   public static void tearDownAfterClass()
   {
      UnitTests.resetConfiguration();
   }

   // XML Constants
   private static final String[] codecKindsFromXML = new String[]{
         "Encryption", "Compression"
   };

   private static final String stringValue = "MyBaseImplementation";
   private static final int intValue = -100;

   // XML parameter constants
   private static final String stringValueFromXML = "This is the string value param test";
   private static final int intValueFromXML = 45;

   public static interface Codec
   {
      public void encode();

      public void decode();
   }

   public static class MyCodec implements Codec
   {
      public void encode()
      {

      }

      public void decode()
      {

      }
   }

   public static class MyCodec2 implements Codec
   {
      public void encode()
      {

      }

      public void decode()
      {

      }
   }

   public static interface Encryption extends Codec
   {
      public void encrypt();

      public void decrypt();
   }

   public static class MyEncryption implements Encryption
   {
      private byte[] data = null;
      boolean encrypted = false;

      public void encode()
      {
         encrypt();
      }

      public void decode()
      {
         decrypt();
      }

      public void encrypt()
      {
         if (!this.encrypted && this.data != null)
         {
            // swap
            byte firstByte = this.data[0];
            this.data[0] = this.data[this.data.length - 1];
            this.data[this.data.length - 1] = firstByte;
            this.encrypted = true;
         }
      }

      public void decrypt()
      {
         if (this.encrypted && this.data != null)
         {
            // swap back
            byte firstByte = this.data[0];
            this.data[0] = this.data[this.data.length - 1];
            this.data[this.data.length - 1] = firstByte;
            this.encrypted = false;
         }
      }

      public void setData(byte[] data)
      {
         this.data = data;
      }

      public byte[] getData()
      {
         return this.data;
      }

      public boolean isEncrypted()
      {
         return this.encrypted;
      }
   }

   public static interface MyBaseInterface
   {
      public int getIntValue();

      public String getStringValue();
   }

   public static interface MyInterface extends MyBaseInterface
   {
      public void setIntValue(int value);

      public void setStringValue(String strValue);
   }

   public static class MyBaseImplementation implements MyBaseInterface
   {
      public MyBaseImplementation()
      {
      }

      public int getIntValue()
      {
         return intValue;
      }

      public String getStringValue()
      {
         return stringValue;
      }
   }

   public static class MyImplementation implements MyInterface
   {
      private int value;
      private String stringValue;

      public MyImplementation()
      {
         this.value = 0;
         this.stringValue = null;
      }

      public void setIntValue(int value)
      {
         this.value = value;
      }

      public int getIntValue()
      {
         return this.value;
      }

      public void setStringValue(String strValue)
      {
         this.stringValue = strValue;
      }

      public String getStringValue()
      {
         return this.stringValue;
      }
   }

   // Constructor
   public XMLBindingsProviderTest()
   {
      if (bindings == null)
      {
         try
         {
            System.setProperty(ConfigurationFactory.XML_BINDINGS_CONFIG_PROPERTY, XMLFileName);
            bindings =
                  ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE);
            assertTrue(bindings instanceof XMLBindingsProvider);
         }
         catch (Exception e)
         {
            e.printStackTrace();
         }
      }
   }

   @Test
   public void testGetImplementation()
   {
      try
      {
         Object myObj = bindings.getImplementation(new String[]{
               "Codec", "Encryption"
         }, "MyEncryption");

         assertTrue(myObj instanceof MyEncryption);

         // Use convenience method in BindingsProviderBase to do the same
         Object myObj2 = bindings.getImplementation("Codec", "Encryption", "MyEncryption");

         assertTrue(myObj2 instanceof MyEncryption);

      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testListImplementations()
   {
      try
      {
         Set<Class<?>> classes = bindings.getImplementationClassSet(new String[]{
            "Codec"
         });

         assertTrue(classes.size() == 2);
         assertTrue(classes.contains(MyCodec.class));
         assertTrue(classes.contains(MyCodec2.class));
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testListSubKinds()
   {
      try
      {
         Set<String> subKinds = bindings.getSubKindsList(new String[]{
            "Codec"
         });

         for (String codecKind : XMLBindingsProviderTest.codecKindsFromXML)
         {
            assertTrue(subKinds.contains(codecKind));
         }
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testGetRequiredInterface()
   {
      try
      {
         Class<?> codecInterfaceClass = bindings.getRequiredInterface(new String[]{
            "Codec"
         });

         assertEquals(codecInterfaceClass, Codec.class);

         Class<?> encryptionInterfaceClass = bindings.getRequiredInterface(new String[]{
               "Codec", "Encryption"
         });

         assertEquals(encryptionInterfaceClass, Encryption.class);

      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   // Tests kinds that have no required interface defined and have no sub kinds.
   @Test
   public void testEmptyKinds()
   {
      try
      {
         Class<?> codecInterfaceClass1 = bindings.getRequiredInterface(new String[]{
            "AnEmptyKind"
         });

         assertTrue(codecInterfaceClass1 == null);

         Set<String> subKinds1 = bindings.getSubKindsList(new String[]{
            "AnEmptyKind"
         });

         assertTrue(subKinds1.size() == 0);

         Class<?> codecInterfaceClass2 = bindings.getRequiredInterface(new String[]{
               "Codec", "Compression", "ImageCompression"
         });

         assertTrue(codecInterfaceClass2 == Codec.class);

         Set<String> subKinds2 = bindings.getSubKindsList(new String[]{
               "Codec", "Compression", "ImageCompression"
         });

         assertTrue(subKinds2.size() == 0);

      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testGetInterfaceWithoutParamsImpl()
   {
      try
      {
         MyBaseInterface myImpl = bindings.getDefaultImplementation(MyBaseInterface.class);

         assertTrue(myImpl instanceof MyBaseImplementation);

         assertEquals(myImpl.getIntValue(), intValue);
         assertEquals(myImpl.getStringValue(), stringValue);
      }
      catch (ConfigurationException ce)
      {
         ce.printStackTrace();
         fail(ce.getMessage());
      }

   }

   @Test
   public void testGetInterfaceWithParamsImpl()
   {
      try
      {
         MyInterface myImpl = bindings.getDefaultImplementation(MyInterface.class);

         assertTrue(myImpl instanceof MyImplementation);

         assertEquals(myImpl.getIntValue(), intValueFromXML);
         assertEquals(myImpl.getStringValue(), stringValueFromXML);
      }
      catch (ConfigurationException ce)
      {
         ce.printStackTrace();
         fail(ce.getMessage());
      }

   }

   @Test
   public void testDefaultKindImpl()
   {
      try
      {
         Object myImpl = bindings.getDefaultImplementation(new String[]{
            "Codec"
         });

         assertTrue(myImpl instanceof MyCodec);

         Object myImpl2 = bindings.getDefaultImplementation(new String[]{
               "Codec", "Encryption"
         });

         assertTrue(myImpl2 instanceof MyEncryption);

      }
      catch (ConfigurationException ce)
      {
         ce.printStackTrace();
         fail(ce.getMessage());
      }

   }

   @Test
   public void testInterfaceReferrals()
   {
      try
      {
         MyInterface myImpl = bindings.getImplementation(MyInterface.class, "a");
         assertEquals(myImpl.getIntValue(), 45);

         MyInterface myImpl2 = bindings.getImplementation(MyInterface.class, "b");
         assertEquals(myImpl2.getIntValue(), 1045);
      }
      catch (ConfigurationException ce)
      {
         ce.printStackTrace();
         fail(ce.getMessage());
      }
   }

   @Test
   public void testKindInterfaceReferrals()
   {
      try
      {
         Object myImpl = bindings.getDefaultImplementation(new String[]{
            "Codec"
         });

         Codec myImpl2 = bindings.getDefaultImplementation(Codec.class);
         assertEquals(myImpl.getClass(), myImpl2.getClass());

         Object myImpl3 = bindings.getImplementation(new String[]{
            "Codec"
         }, "MyCodec2");
         Codec myImpl4 = bindings.getImplementation(Codec.class, "MyCodec2");
         assertEquals(myImpl3.getClass(), myImpl4.getClass());

         // Use convenience method in BindingsProviderBase to do the same
         Object myImpl5 = bindings.getImplementation("Codec", "MyCodec2");
         assertEquals(myImpl3.getClass(), myImpl5.getClass());

      }
      catch (ConfigurationException ce)
      {
         ce.printStackTrace();
         fail(ce.getMessage());
      }
   }

   @Test
   public void testKindInterfaceReferralsVersions()
   {
      try
      {
         Object myImpl = bindings.getImplementation(new String[]{
            "Codec"
         }, "MyCodec2", Version.fromString("1.0"));
         Codec myImpl2 =
               bindings.getImplementation(Codec.class, "MyCodec2", Version.fromString("2.4.9"));
         assertEquals(myImpl.getClass(), myImpl2.getClass());

      }
      catch (ConfigurationException ce)
      {
         ce.printStackTrace();
         fail(ce.getMessage());
      }
   }

   @Test(expected = ConfigurationItemNotDefinedException.class)
   public void testKindInterfaceReferralsVersions2() throws ConfigurationException
   {
      bindings.getImplementation(new String[]{
         "Codec"
      }, "MyCodec2", Version.fromString("2.5.1"));
   }

   @Test(expected = ConfigurationItemNotDefinedException.class)
   public void testKindInterfaceReferralsVersions3() throws ConfigurationException
   {
      bindings.getImplementation(Codec.class, "MyCodec2", Version.fromString("2.5.1"));
   }

   @Test(expected = ConfigurationItemNotDefinedException.class)
   public void testKindInterfaceReferralsVersions4() throws ConfigurationException
   {
      bindings.getImplementation(Codec.class, "MyCodec2", Version.fromString("0.9.9"));
   }

   @Test
   public void testGetImplementationClassFromKindReferral()
   {
      try
      {
         Class<?> myClass = bindings.getImplementationClass(new String[]{
               "Codec", "Encryption"
         }, "MyEncryption");

         assertEquals(myClass, MyEncryption.class);

         // Use convenience method in BindingsProviderBase to do the same
         Class<?> myClass2 = bindings.getImplementationClass("Codec", "Encryption", "MyEncryption");
         assertEquals(myClass2, MyEncryption.class);

         Class<?> myCodecClass = bindings.getImplementationClass(new String[]{
            "Codec"
         }, "MyCodec");

         assertEquals(myCodecClass, MyCodec.class);

         Class<?> myCodec2Class = bindings.getImplementationClass(new String[]{
            "Codec"
         }, "MyCodec2");

         assertEquals(myCodec2Class, MyCodec2.class);

         // Use convenience method in BindingsProviderBase to do the same
         Class<?> myCodec2Class2 = bindings.getImplementationClass("Codec", "MyCodec2");
         assertEquals(myCodec2Class2, MyCodec2.class);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testGetImplementationClassFromKindReferralVersion()
   {
      try
      {
         Class<?> myCodec2Class = bindings.getImplementationClass(new String[]{
            "Codec"
         }, "MyCodec2", Version.fromString("1.5"));

         assertEquals(myCodec2Class, MyCodec2.class);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test(expected = ConfigurationItemNotDefinedException.class)
   public void testGetImplementationClassFromKindReferralVersion2() throws ConfigurationException
   {
      bindings.getImplementationClass(new String[]{
         "Codec"
      }, "MyCodec", Version.fromString("1.5"));
   }

   @Test
   public void testGetImplementationClassFromInterfaceReferral()
   {
      try
      {
         Class<?> myClass = bindings.getImplementationClass(Encryption.class, "MyEncryption");
         assertEquals(myClass, MyEncryption.class);

         Class<?> myCodecClass = bindings.getImplementationClass(Codec.class, "MyCodec");
         assertEquals(myCodecClass, MyCodec.class);

         Class<?> myCodec2Class = bindings.getImplementationClass(Codec.class, "MyCodec2");
         assertEquals(myCodec2Class, MyCodec2.class);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testGetImplementationClassFromInterfaceReferralVersion()
   {
      try
      {
         Class<?> myCodec2Class =
               bindings.getImplementationClass(Codec.class, "MyCodec2", Version.fromString("1.5"));
         assertEquals(myCodec2Class, MyCodec2.class);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testGetDefaultImplementationClassFromInterface()
   {
      try
      {
         Class<?> myCodecClass = bindings.getDefaultImplementationClass(Codec.class);
         assertEquals(myCodecClass, MyCodec.class);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testGetDefaultImplementationClassFromKind()
   {
      try
      {
         Class<?> myCodecClass = bindings.getDefaultImplementationClass(new String[]{
            "Codec"
         });
         assertEquals(myCodecClass, MyCodec.class);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testTemp() throws Exception
   {
      System.setProperty(ConfigurationFactory.XML_BINDINGS_CONFIG_PROPERTY,
            "conf/dev/core/core-org-bindings.xml");

      ConfigurationFactory.unloadBindings(ConfigurationFactory.XML_CONFIG_TYPE);

      BindingsProvider bindingsProvider =
            ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE);

      // There should be no Vault implementations defined in the core-org bindings
      assertEquals(bindingsProvider.getImplementationClassSet(new String[]{
         "Vault"
      }).size(), 0);

      System.setProperty(ConfigurationFactory.XML_BINDINGS_CONFIG_PROPERTY,
            "conf/dev/core/core-org-bindings.xml;conf/dev/block-device/blockdevice-org-bindings.xml");

      ConfigurationFactory.unloadBindings(ConfigurationFactory.XML_CONFIG_TYPE);

      BindingsProvider bindingsProvider2 =
            ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE);

      // There should be a new Vault implementation in the new bindings file
      assertEquals(bindingsProvider2.getImplementationClassSet(new String[]{
         "Vault"
      }).size(), 1);

      // Number of codecs must not change by adding the new bindings file
      assertEquals(bindingsProvider.getImplementationClassSet(new String[]{
         "Codec"
      }).size(), bindingsProvider2.getImplementationClassSet(new String[]{
         "Codec"
      }).size());
   }
}
